/**
 * @copyright 2021 Tridium, Inc. All Rights Reserved.
 */

/*jshint browser: true */ /* eslint-env browser */

define(['jquery'], function ($) {
  'use strict';

  /**
   * API Status: **Private**
   *
   * Utility for automatically scrolling while dragging around the edges of a div.
   *
   * @exports nmodule/wiresheet/rc/wb/util/edgeScrollUtil
   */
  var exports = {};
  exports.$intervalTicks = 100;

  /**
   * @param {Element} dom
   * @returns {Function} a function to release any registered event listeners on
   * the provided element
   */
  exports.initialize = function (dom) {
    var jq = $(dom),
      documentJq = $(document);
    var mousedown = false;
    jq.on('mousedown.edgeScroll', function (event) {
      if (event.which === 1) {
        mousedown = true;
      }
    });
    var mouseupHandler = function mouseupHandler(event) {
      if (event.which === 1) {
        mousedown = false;
      }
    };
    documentJq.on('mouseup', mouseupHandler);
    jq.on('dragover.edgeScroll', function (event) {
      exports.$updateScroll(event, dom);
    });
    jq.on('mousemove.edgeScroll', function (event) {
      if (mousedown) {
        exports.$updateScroll(event, dom);
      }
    });
    jq.on('touchmove.edgeScroll', function (event) {
      var touches = event.touches;
      if (touches.length === 1) {
        exports.$updateScroll(event, dom);
      }
    });
    var release = function release() {
      jq.off('dragover.edgeScroll mousemove.edgeScroll touchmove.edgeScroll mousedown.edgeScroll');
      documentJq.off('mouseup', mouseupHandler);
      var edgeScrollState = dom.$edgeScrollState;
      if (edgeScrollState && edgeScrollState.$pendingCleanup) {
        edgeScrollState.$pendingCleanup();
      }
      delete dom.$edgeScrollState;
    };
    return release;
  };

  /**
   * @private
   * @param {Event} event
   * @param {Element} dom
   * @return {{deltaX: number, deltaY: number}}
   */
  exports.$getDeltaScroll = function (event, dom) {
    var _getScrollCoords = getScrollCoords(event, dom),
      x = _getScrollCoords.x,
      y = _getScrollCoords.y;
    var scrollBarWidth = dom.offsetWidth - dom.clientWidth;
    var edge = 16; //same as BScrollPane.pulseViewPort

    var width = dom.offsetWidth;
    var height = dom.offsetHeight;
    var deltaX = 0,
      deltaY = 0;
    var _getWindowTouchCoords = getWindowTouchCoords(event),
      pageX = _getWindowTouchCoords.pageX,
      pageY = _getWindowTouchCoords.pageY;
    var _exports$$getWindowSi = exports.$getWindowSize(),
      windowWidth = _exports$$getWindowSi.windowWidth,
      windowHeight = _exports$$getWindowSi.windowHeight;

    //mobile sidebars can cause wiresheet to only be partially visible on a phone so use the window edge
    //as well as long as the window edge overlaps the wiresheet.
    if (isBetween(x, 0, width) && isBetween(y, 0, height)) {
      if (isBetween(pageX, windowWidth - edge, windowWidth)) {
        deltaX = edge;
      } else if (isBetween(pageX, 0, edge)) {
        deltaX = -edge;
      }
      if (isBetween(pageY, windowHeight - edge, windowHeight)) {
        deltaY = edge;
      } else if (isBetween(pageY, 0, edge)) {
        deltaY = -edge;
      }
    }
    if (isBetween(x, width - scrollBarWidth - edge, width)) {
      deltaX = edge;
    } else if (isBetween(x, 0, edge)) {
      deltaX = -edge;
    }
    if (isBetween(y, height - scrollBarWidth - edge, height)) {
      deltaY = edge;
    } else if (isBetween(y, 0, edge)) {
      deltaY = -edge;
    }
    return {
      deltaX: deltaX,
      deltaY: deltaY
    };
  };

  /**
   * @private
   * @param {Event} event
   * @param {Element} dom
   */
  exports.$updateScroll = function (event, dom) {
    var _exports$$getDeltaScr = exports.$getDeltaScroll(event, dom),
      deltaX = _exports$$getDeltaScr.deltaX,
      deltaY = _exports$$getDeltaScr.deltaY;
    var edgeScrollState = dom.$edgeScrollState = dom.$edgeScrollState || {};

    //all the same deltas, no change
    if (edgeScrollState.$pendingCleanup && edgeScrollState.$deltaX === deltaX && edgeScrollState.$deltaY === deltaY) {
      return;
    }
    edgeScrollState.$deltaX = deltaX;
    edgeScrollState.$deltaY = deltaY;
    edgeScrollState.$firstInterval = true;
    if (!deltaX && !deltaY) {
      if (edgeScrollState.$pendingCleanup) {
        edgeScrollState.$pendingCleanup();
      }
      return;
    }
    var scroll = function scroll() {
      //skip the first interval to give the user a better chance to drag something off the wiresheet without scrolling too much
      if (edgeScrollState.$firstInterval) {
        edgeScrollState.$firstInterval = false;
        return;
      }
      dom.scrollLeft += deltaX;
      dom.scrollTop += deltaY;
    };
    if (!edgeScrollState.$pendingCleanup) {
      var cleanupCaller = function cleanupCaller() {
        if (edgeScrollState.$pendingCleanup) {
          edgeScrollState.$pendingCleanup();
        }
      };
      edgeScrollState.$pendingCleanup = function () {
        clearInterval(edgeScrollState.$interval);
        delete edgeScrollState.$interval;
        dom.removeEventListener('drop', cleanupCaller, true);
        dom.removeEventListener('mouseup', cleanupCaller, true);
        dom.removeEventListener('mouseleave', cleanupCaller, true);
        dom.removeEventListener('touchend', cleanupCaller, true);
        dom.removeEventListener('touchcancel', cleanupCaller, true);
        edgeScrollState.$pendingCleanup = null;
      };

      //use capture events in case there are an event listener which stops the bubbling (like there is with drop)
      dom.addEventListener('drop', cleanupCaller, true);
      dom.addEventListener('mouseup', cleanupCaller, true);
      dom.addEventListener('mouseleave', cleanupCaller, true);
      dom.addEventListener('touchend', cleanupCaller, true);
      dom.addEventListener('touchcancel', cleanupCaller, true);
    }
    if (!edgeScrollState.$interval) {
      edgeScrollState.$interval = setInterval(function () {
        scroll();
      }, exports.$intervalTicks);
    }
  };

  /**
   * @private
   * @param {Event} event
   * @return {{windowHeight: number, windowWidth: number}}
   */
  exports.$getWindowSize = function () {
    return {
      windowWidth: window.screen.width,
      windowHeight: window.screen.height
    };
  };

  /**
   * @param {Event} event
   * @return {{pageY: number, pageX: number}}
   */
  function getWindowTouchCoords(event) {
    var touches = event.touches;
    var actualEvent = touches && touches.length === 1 ? touches[0] : event;
    var pageX = actualEvent.pageX,
      pageY = actualEvent.pageY;
    return {
      pageX: pageX,
      pageY: pageY
    };
  }

  /**
   * @param {Event} event
   * @param {Element} dom
   * @return {{x: number, y: number}}
   */
  function getScrollCoords(event, dom) {
    if (!dom) {
      throw new Error('dom required');
    }
    var touches = event.touches;
    var actualEvent = touches && touches.length === 1 ? touches[0] : event;
    var pageX = actualEvent.pageX,
      pageY = actualEvent.pageY;
    var _ref = $(dom).offset() || {},
      _ref$left = _ref.left,
      left = _ref$left === void 0 ? 0 : _ref$left,
      _ref$top = _ref.top,
      top = _ref$top === void 0 ? 0 : _ref$top;
    return {
      x: pageX - left,
      y: pageY - top
    };
  }

  /**
   * @param {Number} value
   * @param {Number} min
   * @param {Number} max
   * @return {boolean}
   */
  function isBetween(value, min, max) {
    return value >= min && value <= max;
  }
  return exports;
});
