/**
 * @copyright 2020 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

define([], function () {
  'use strict';

  /**
   * API Status: **Private**
   * @exports module:nmodule/wiresheet/rc/wb/render/canvas/canvasUtils
   */
  var exports = {};

  /**
   * @param {object} params
   * @param {CanvasRenderingContext2D} params.ctx
   * @param {string} params.text the text to trim
   * @param {number} params.width width in pixels to fit
   * @returns {string} the text trimmed to fit
   */
  exports.trimText = function (_ref) {
    var ctx = _ref.ctx,
      text = _ref.text,
      width = _ref.width;
    while (!canFit(ctx, text, width)) {
      text = chopLastCharacter(text);
    }
    return text;
  };

  /**
   * @param {object} params
   * @param {CanvasRenderingContext2D} params.ctx
   * @param {string} params.text the text to wrap
   * @param {number} params.width width in pixels to fit
   * @returns {string[]} the text wrapped to fit
   */
  exports.wrapText = function (_ref2) {
    var ctx = _ref2.ctx,
      text = _ref2.text,
      width = _ref2.width;
    if (width <= 0) {
      return [''];
    }
    var wrappedLines = [];
    text.split('\n').forEach(function (line) {
      var currentLine = '';
      var words = line.trim().split(/\s+/);
      words.forEach(function (word) {
        var testLine = currentLine + (currentLine ? ' ' : '') + word;
        if (canFit(ctx, testLine, width)) {
          currentLine = testLine;
        } else {
          wrappedLines.push(currentLine);
          currentLine = exports.trimText({
            ctx: ctx,
            text: word,
            width: width
          });
        }
      });
      if (currentLine) {
        wrappedLines.push(currentLine);
      }
    });
    return wrappedLines;
  };
  var LINEAR_GRADIENT_PARAMS_REGEX = /linear-gradient\((-?\d+deg|to right|to bottom|to top|to left|)(?:, )?(.*)\)/;
  var COLOR_STRING_REGEX = /(#[0-9A-Fa-f]{6}|rgb\(\d+, \d+, \d+\)|rgba\(\d+, \d+, \d+, \d*\.?\d*\))(?: (\d+%))?/g;
  var DEFAULT_CSS_GRADIENT = {
    angle: 0,
    stops: []
  };

  /**
   * Takes the css style of a linear-gradient and converts it into the associated
   * angle and color stops with values adjusted for display on a canvas.
   *
   * @param {string} cssText the text value of the linear gradient, this function
   * is intended to target the computed style for the linear gradient.
   * @returns {{angle: number, stops: Array.<{ color: string, offset: number }>}}
   */
  exports.extractLinearGradientStyleFromCss = function (cssText) {
    var match = LINEAR_GRADIENT_PARAMS_REGEX.exec(cssText);
    if (!match || match.length === 1) {
      return DEFAULT_CSS_GRADIENT;
    }
    var direction = getAngleFromText(match[1]);
    var stops = [];
    var result;
    while ((result = COLOR_STRING_REGEX.exec(match[2])) !== null) {
      var color = result[1],
        offsetText = result[2];
      var offset = offsetText && parseInt(offsetText) / 100;
      stops.push({
        color: color,
        offset: offset
      });
    }
    stops.forEach(function (stop, index) {
      if (stop.offset === undefined) {
        var lastStopIndex = stops.length - 1;
        if (index === lastStopIndex) {
          stop.offset = 1;
        } else {
          stop.offset = index / lastStopIndex;
        }
      }
    });
    return {
      angle: direction,
      stops: stops
    };
  };
  function getAngleFromText(text) {
    var direction = parseFloat(text);
    if (!isNaN(direction)) {
      return direction;
    }
    if (text === "to top") {
      return 0;
    }
    if (text === "to right") {
      return 90;
    }
    if (text === "to bottom") {
      return 180;
    }
    if (text === "to left") {
      return 270;
    }
    return 180;
  }

  /**
   * Calculates the position for the linear gradient line start and end.
   *
   * @param {number} canvasAngle
   * @param {number} width
   * @param {number} height
   * @returns {{startX: number, startY: number, endX: number, endY: number}}
   */
  exports.calcLinearGradientRect = function (canvasAngle, width, height) {
    var radians = canvasAngle * Math.PI / 180,
      halfWidth = width / 2,
      halfHeight = height / 2,
      gradientLength = getGradientLength(canvasAngle, radians, halfWidth, halfHeight);
    var startX = halfWidth + cosine(-Math.PI / 2 + radians) * -gradientLength,
      startY = halfHeight + sine(-Math.PI / 2 + radians) * -gradientLength,
      endX = halfWidth + cosine(-Math.PI / 2 + radians) * gradientLength,
      endY = halfHeight + sine(-Math.PI / 2 + radians) * gradientLength;
    return {
      startX: startX,
      startY: startY,
      endX: endX,
      endY: endY
    };
  };
  function getGradientLength(canvasAngle, radians, halfWidth, halfHeight) {
    if (canvasAngle % 180 === 0) {
      return halfHeight;
    }
    var xIntersectHyp = calcLongestSideFromOp(halfHeight, radians),
      fromTopRightX = halfWidth - Math.sqrt(Math.abs(xIntersectHyp * xIntersectHyp - halfHeight * halfHeight)),
      isGradientIntersectingTop = fromTopRightX < halfWidth;
    if (isGradientIntersectingTop) {
      var _outerLength = fromTopRightX * cosine(radians);
      return xIntersectHyp + _outerLength;
    }
    var yIntersectHyp = calcLongestSideFromAdj(halfWidth, radians),
      distanceFromTop = halfHeight - Math.sqrt(Math.abs(yIntersectHyp * yIntersectHyp - halfWidth * halfWidth)),
      fromTopRightY = halfHeight - distanceFromTop,
      outerLength = fromTopRightY * cosine(radians);
    return yIntersectHyp + outerLength;
  }
  function calcLongestSideFromAdj(adj, radians) {
    var trigResult = cosine(radians);
    if (trigResult === 0) {
      return adj;
    }
    return Math.abs(adj / trigResult);
  }
  function calcLongestSideFromOp(op, radians) {
    var trigResult = sine(radians);
    if (trigResult === 0) {
      return op;
    }
    return Math.abs(op / trigResult);
  }
  function cosine(val) {
    var result = Math.cos(val);
    if (Math.abs(result) < Number.EPSILON) {
      return 0;
    }
    return result;
  }
  function sine(val) {
    var result = Math.sin(val);
    if (Math.abs(result) < Number.EPSILON) {
      return 0;
    }
    return result;
  }
  function canFit(ctx, text, width) {
    return ctx.measureText(text).width <= width;
  }
  function chopLastCharacter(text) {
    return text.substring(0, text.length - 1);
  }
  return exports;
});
