function _slicedToArray(r, e) { return _arrayWithHoles(r) || _iterableToArrayLimit(r, e) || _unsupportedIterableToArray(r, e) || _nonIterableRest(); }
function _nonIterableRest() { throw new TypeError("Invalid attempt to destructure non-iterable instance.\nIn order to be iterable, non-array objects must have a [Symbol.iterator]() method."); }
function _unsupportedIterableToArray(r, a) { if (r) { if ("string" == typeof r) return _arrayLikeToArray(r, a); var t = {}.toString.call(r).slice(8, -1); return "Object" === t && r.constructor && (t = r.constructor.name), "Map" === t || "Set" === t ? Array.from(r) : "Arguments" === t || /^(?:Ui|I)nt(?:8|16|32)(?:Clamped)?Array$/.test(t) ? _arrayLikeToArray(r, a) : void 0; } }
function _arrayLikeToArray(r, a) { (null == a || a > r.length) && (a = r.length); for (var e = 0, n = Array(a); e < a; e++) n[e] = r[e]; return n; }
function _iterableToArrayLimit(r, l) { var t = null == r ? null : "undefined" != typeof Symbol && r[Symbol.iterator] || r["@@iterator"]; if (null != t) { var e, n, i, u, a = [], f = !0, o = !1; try { if (i = (t = t.call(r)).next, 0 === l) { if (Object(t) !== t) return; f = !1; } else for (; !(f = (e = i.call(t)).done) && (a.push(e.value), a.length !== l); f = !0); } catch (r) { o = !0, n = r; } finally { try { if (!f && null != t["return"] && (u = t["return"](), Object(u) !== u)) return; } finally { if (o) throw n; } } return a; } }
function _arrayWithHoles(r) { if (Array.isArray(r)) return r; }
/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/wiresheet/rc/wb/controller/ops/LinkOp
 */
define(['baja!', 'lex!baja', 'Promise', 'nmodule/webEditors/rc/wb/baja/linkCheckUtil', 'nmodule/webEditors/rc/wb/commands/LinkCommand', 'nmodule/webEditors/rc/wb/commands/LinkToCommand', 'nmodule/wiresheet/rc/wb/util/wsUtils'], function (baja, lexs, Promise, linkCheckUtil, LinkCommand, LinkToCommand, wsUtils) {
  'use strict';

  var toBajaId = wsUtils.toBajaId,
    boundingBox = wsUtils.boundingBox;
  var _lexs = _slicedToArray(lexs, 1),
    bajaLex = _lexs[0];
  var LINK_ID = 'link';
  var getInvalidDirectionMessage = function getInvalidDirectionMessage(direction, endSlot) {
    if (direction === 'out') {
      return bajaLex.get({
        key: 'linkcheck.cannotLinkSourceToSource',
        args: [endSlot]
      });
    }
    return bajaLex.get({
      key: 'linkcheck.cannotLinkTargetToTarget',
      args: [endSlot]
    });
  };
  var getLinkToMessage = function getLinkToMessage(startEntityText, endEntityText) {
    return bajaLex.get({
      key: 'linkcheck.linkTo',
      args: [startEntityText, endEntityText]
    });
  };

  /**
   * This op allows the user to make a connection from an outgoing connector
   * on one glyph to an incoming connector on another glyph.
   *
   * This op has a BajaScript dependency. It operates by creating Link objects
   * and writing them through to the station.
   *
   * To avoid confusion, in this file we'll use "start" and "end" to refer to
   * where the _user_ drags the mouse, and "source" and "target" to refer to the
   * properties of the actual `Link` we create.
   *
   * @class
   * @alias module:nmodule/wiresheet/rc/wb/controller/ops/LinkOp
   * @implements module:nmodule/wiresheet/rc/typedefs~DragOp
   * @param {Object} params
   * @param {number} params.x starting mouse x, in wixels
   * @param {number} params.y starting mouse y, in wixels
   * @param {module:nmodule/wiresheet/rc/core/ViewModel} params.viewModel
   * @param {baja.Component} params.base a base component for resolving ORDs
   * @param {string} params.entityId the entity that is the origin of the link
   * @param {string} params.connectorDirection the link is going
   * @param {string} params.connectorId the connector within the link origin
   * that we're linking from
   * @param {string} params.connectorDirection `in` if we're going to link into
   * the component under the mouse; `out` if we're linking out from it
   * @param {function} params.setCursor a function to update the visible cursor
   * as we drag around
   * @param {function} params.setToolTip a function to update the display value
   * of the tooltip
   * @param {function} params.cancelToolTip a function to cancel the ToolTip if
   * it is currently displayed
   */
  var LinkOp = function LinkOp(params) {
    var _this = this;
    var origX = params.x,
      origY = params.y,
      viewModel = params.viewModel,
      base = params.base,
      entityId = params.entityId,
      startDirection = params.connectorDirection,
      connectorId = params.connectorId,
      setCursor = params.setCursor,
      setToolTip = params.setToolTip,
      cancelToolTip = params.cancelToolTip;
    var startOrd = toBajaId(entityId);
    var startSlot = connectorId && toBajaId(connectorId);

    /**
     *
     * @param {string} entityId the entity the mouse is over
     * @param {String} connectorId the connector the mouse is over
     * @returns {Promise.<module:nmodule/wiresheet/rc/wb/controller/ops/LinkOp~LinkDragInfo|null>} resolves
     * to info about what's currently dragged by the mouse (or null if not
     * currently over a valid connector)
     */
    var getCurrentDragInfo = function getCurrentDragInfo(entityId, connectorId) {
      if (!entityId) {
        return Promise.resolve(null);
      }
      var endOrd = toBajaId(entityId);
      return baja.BatchResolve.resolve({
        ords: [startOrd, endOrd],
        base: base
      }).then(function (br) {
        var _br$getTargetObjects = br.getTargetObjects(),
          _br$getTargetObjects2 = _slicedToArray(_br$getTargetObjects, 2),
          start = _br$getTargetObjects2[0],
          end = _br$getTargetObjects2[1];
        return {
          start: start,
          startSlot: startSlot,
          end: end,
          endSlot: connectorId && toBajaId(connectorId)
        };
      });
    };
    var makeLinkCreateInfo = function makeLinkCreateInfo(dragInfo) {
      var start = dragInfo.start,
        startSlot = dragInfo.startSlot,
        end = dragInfo.end,
        endSlot = dragInfo.endSlot;
      switch (startDirection) {
        case 'in':
          return {
            source: end,
            sourceSlot: endSlot,
            target: start,
            targetSlot: startSlot
          };
        case 'out':
          return {
            source: start,
            sourceSlot: startSlot,
            target: end,
            targetSlot: endSlot
          };
      }
    };

    /**
     * Check if the current location is a valid drop point for a drag and drop Link operation.
     *
     * @param {string} entityId the entity the mouse is over
     * @param {string} connectorId the connector the mouse is over
     * @param {string} direction in if linking into the entity; out if linking
     * out from the entity
     * @returns {Promise}
     *
     */
    this.updateDropTarget = function (entityId, connectorId, direction) {
      if (!entityId) {
        cancelToolTip();
        return setCursor('grabbing');
      }
      return getCurrentDragInfo(entityId, connectorId).then(function (dragInfo) {
        if (!dragInfo) {
          return;
        }
        if (direction === startDirection) {
          setToolTip(getInvalidDirectionMessage(direction, dragInfo.endSlot));
          return setCursor('not-allowed');
        }

        // Assuming the drag is over header or footer (don't know how to distinguish)
        var startSlot = dragInfo.startSlot,
          endSlot = dragInfo.endSlot;
        if (!startSlot || !endSlot) {
          // Do nothing and commit
          setCursor(startDirection);
          var _startSlot = dragInfo.startSlot,
            _endSlot = dragInfo.endSlot;
          setToolTip(getLinkToMessage(_startSlot || '*', _endSlot || '*'));
          return;
        }
        var linkCreateInfo = makeLinkCreateInfo(dragInfo);
        return toLinkCheck(linkCreateInfo).then(function (linkCheck) {
          setCursor(linkCheck.isValid() ? startDirection : 'not-allowed');
          if (linkCheck.isValid()) {
            setToolTip(getLinkToMessage(dragInfo.startSlot, dragInfo.endSlot));
          } else {
            var text = bajaLex.get({
              key: 'linkcheck.linkInvalid',
              args: [dragInfo.endSlot, linkCheck.getInvalidReason()]
            });
            setToolTip(text);
          }
        });
      });
    };

    /**
     * When the user drags around, move the link "arrow" appropriately.
     * @param {module:nmodule/wiresheet/rc/typedefs~DragOpState} dragOpState
     * @returns {Promise}
     */
    this.update = function (_ref) {
      var x = _ref.x,
        y = _ref.y,
        entityId = _ref.entityId,
        connectorId = _ref.connectorId,
        connectorDirection = _ref.connectorDirection;
      var end = {
        x: x,
        y: y
      };
      var start = {
        x: origX,
        y: origY
      };
      return viewModel.put(toLinkGlyph(start, end), LINK_ID).then(function () {
        return _this.updateDropTarget(entityId, connectorId, connectorDirection);
      });
    };

    /**
     * When the user commits the op, write a new Link slot through to the
     * station, linking the two selected slots.
     * @param {module:nmodule/wiresheet/rc/typedefs~DragOpState} dragOpState
     * @returns {Promise|*}
     */
    this.commit = function (_ref2) {
      var entityId = _ref2.entityId,
        connectorId = _ref2.connectorId,
        endDirection = _ref2.connectorDirection;
      if (endDirection === startDirection) {
        return;
      } // user goofed
      if (!entityId) {
        return;
      } // user dropped on empty space

      return getCurrentDragInfo(entityId, connectorId).then(function (dragInfo) {
        var linkCreateInfo = makeLinkCreateInfo(dragInfo);
        var source = linkCreateInfo.source,
          sourceSlot = linkCreateInfo.sourceSlot,
          target = linkCreateInfo.target,
          targetSlot = linkCreateInfo.targetSlot;
        var cmd;
        if (!sourceSlot || !targetSlot) {
          cmd = new LinkCommand({
            subject: source,
            sourceSlotName: sourceSlot,
            object: target,
            targetSlotName: targetSlot,
            direction: 'to'
          });
        } else {
          cmd = new LinkToCommand({
            subject: source,
            sourceSlotName: sourceSlot,
            object: target,
            targetSlotName: targetSlot
          });
        }
        return cmd.invoke();
      });
    };

    /**
     * Removes any link glyph from the view model.
     * @returns {Promise}
     */
    this.destroy = function () {
      return viewModel.del(LINK_ID);
    };
  };

  /**
   * @param {baja.LinkCheck~LinkCreateInfo} linkCreateInfo
   * @returns {Promise.<baja.LinkCheck>}
   */
  function toLinkCheck(linkCreateInfo) {
    return linkCheckUtil.checkLink(linkCreateInfo).then(function (linkCheck) {
      return linkCheck && linkCheck.serverLinkCheck || linkCheck;
    });
  }
  function toLinkGlyph(start, end) {
    var layout = boundingBox([start, end]);
    return {
      glyph: {
        type: 'LinkGlyph',
        start: {
          x: start.x - layout.x,
          y: start.y - layout.y
        },
        end: {
          x: end.x - layout.x,
          y: end.y - layout.y
        }
      },
      layout: layout
    };
  }

  /**
   * @typedef module:nmodule/wiresheet/rc/wb/controller/ops/LinkOp~LinkDragInfo
   * @property {baja.Component} start component we're dragging from
   * @property {string} startSlot slot we're dragging from
   * @property {baja.Component} end component we're dropping onto
   * @property {string} endSlot slot we're dropping onto
   */

  return LinkOp;
});
