/*jshint devel: true, browser: true */

/**
 * API Status: **Private**
 * @module nmodule/wiresheet/rc/wb/WebWiresheet
 */
define(['bajaux/commands/Command', 'bajaux/mixin/responsiveMixIn', 'bajaux/mixin/subscriberMixIn', 'nmodule/webEditors/rc/fe/baja/BaseEditor', 'nmodule/webEditors/rc/fe/baja/util/DepthSubscriber', 'nmodule/webEditors/rc/wb/mixin/ContextMenuSupport', 'nmodule/webEditors/rc/wb/mixin/TransferSupport', 'nmodule/wiresheet/rc/wb/WbConstants', 'nmodule/wiresheet/rc/wb/WbWiresheetEnv', 'nmodule/wiresheet/rc/wb/commands/SelectAllCommand', 'nmodule/wiresheet/rc/wb/commands/WsOptionsCommand', 'nmodule/wiresheet/rc/wb/WsOptions', 'css!nmodule/wiresheet/rc/wiresheet'], function (Command, responsiveMixin, subscribable, BaseEditor, DepthSubscriber, ContextMenuSupport, TransferSupport, WbConstants, WbWiresheetEnv, SelectAllCommand, WsOptionsCommand, WsOptions) {
  'use strict';

  var WIXEL = WbConstants.WIXEL;
  var DEFAULT_THUMBNAIL_OFFSET = 5,
    THUMBNAIL_VISIBILITY_THRESHOLD = 0.4;
  var widgetDefaults = function widgetDefaults() {
    return {
      properties: {
        tHandle: {
          value: '',
          hidden: true,
          readonly: true,
          "transient": true,
          parameter: true
        },
        tSlot: {
          value: '',
          hidden: true,
          readonly: true,
          "transient": true,
          parameter: true
        },
        sHandle: {
          value: '',
          hidden: true,
          readonly: true,
          "transient": true,
          parameter: true
        },
        sSlot: {
          value: '',
          hidden: true,
          readonly: true,
          "transient": true,
          parameter: true
        },
        selectedType: {
          value: '',
          hidden: true,
          readonly: true,
          "transient": true,
          parameter: true
        }
      }
    };
  };

  /**
   * bajaux implementation of Web Wiresheet.
   *
   * @class
   * @alias module:nmodule/wiresheet/rc/wb/WebWiresheet
   * @extends module:nmodule/webEditors/rc/fe/baja/BaseEditor
   */
  var WebWiresheet = function WebWiresheet() {
    var _ref = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {},
      theme = _ref.theme;
    BaseEditor.call(this, {
      arguments: arguments,
      defaults: widgetDefaults()
    });
    subscribable(this);
    this.$theme = theme;
    this.$zoomLevel = 1;
    this.$subscriber = new DepthSubscriber(2);
    ContextMenuSupport(this, {
      menuModuleId: 'nmodule/wiresheet/rc/wb/baja/BajaContextMenu'
    });
    TransferSupport(this);
    responsiveMixin(this);
  };
  WebWiresheet.prototype = Object.create(BaseEditor.prototype);
  WebWiresheet.prototype.constructor = WebWiresheet;
  WebWiresheet.prototype.$getSvg = function () {
    return this.jq().find('.ws-root')[0];
  };
  WebWiresheet.prototype.doInitialize = function (dom) {
    var _this = this;
    dom.addClass('WebWiresheet');
    dom.html("\n      <div class=\"ws-content-outer\"> \n        <div class=\"ws-content-inner ws-render-context ws-invisible-lines\" />\n      </div>\n      <div class=\"ws-thumb-container ws-visible-lines\"/>");
    dom.find('.ws-content-outer').on('scroll', function () {
      _this.$scroll();
    });
  };
  WebWiresheet.prototype.doLayout = function () {
    var jq = this.jq(),
      width = jq.width(),
      height = jq.height();
    this.$env && this.$env.viewportChanged({
      width: Math.ceil(width / WIXEL),
      height: Math.ceil(height / WIXEL)
    });
    this.$scroll();
    this.$options && this.$updateThumbnailForOptions();
    this.$updateThumbnailPosition();
    this.$updateThumbnailDisplay();
  };
  WebWiresheet.prototype.doLoad = function (component) {
    var _this2 = this;
    return WsOptions.make().then(function (wsOptions) {
      _this2.$options = wsOptions;
      return WbWiresheetEnv.initialize({
        component: component,
        dom: _this2.jq(),
        contextMenu: _this2.getContextMenu(),
        options: wsOptions,
        selectionParams: {
          tHandle: _this2.properties().getValue('tHandle'),
          tSlot: _this2.properties().getValue('tSlot'),
          sHandle: _this2.properties().getValue('sHandle'),
          sSlot: _this2.properties().getValue('sSlot'),
          selectedType: _this2.properties().getValue('selectedType')
        },
        theme: _this2.$theme
      }).then(function (env) {
        _this2.$env = env;
        var zoomLevel = _this2.$zoomLevel;
        var setZoom = function setZoom(zoom) {
          zoom = zoomLevel = Math.max(zoom, 0.2);
          return _this2.$env.zoom(zoom);
        };
        _this2.getCommandGroup().add(new Command({
          module: 'bajaui',
          lex: 'commands.zoomIn',
          func: function func() {
            return setZoom(zoomLevel + 0.2);
          }
        }), new Command({
          module: 'bajaui',
          lex: 'commands.zoomOut',
          func: function func() {
            return setZoom(zoomLevel - 0.2);
          }
        }), new Command({
          module: 'bajaui',
          lex: 'commands.zoomReset',
          func: function func() {
            return setZoom(1);
          }
        }), new SelectAllCommand(env.$viewModel, env.$selection), new WsOptionsCommand(_this2));
      });
    });
  };
  WebWiresheet.prototype.getContextMenuSelector = function () {
    return '[ws\\:entityid], .ws-content-outer';
  };

  /**
   * Applies user configurable options to the wiresheet
   * 
   * @param {module:nmodule/wiresheet/rc/wb/WsOptions} options 
   */
  WebWiresheet.prototype.applyUserOptions = function (options) {
    this.$options = options;
    this.$updateThumbnailForOptions();
    return this.$env.applyUserOptions(options);
  };
  WebWiresheet.prototype.doDestroy = function () {
    this.jq().removeClass('WebWiresheet');
    var env = this.$env;
    return env && env.release();
  };

  /**
   * @private
   */
  WebWiresheet.prototype.$scroll = function () {
    this.$env && this.$env.scroll();
  };

  /**
   * @private
   */
  WebWiresheet.prototype.$getScrollbarWidth = function () {
    var jq = this.jq(),
      scrollDiv = jq.find('.ws-content-outer')[0];
    return scrollDiv.offsetWidth - scrollDiv.clientWidth;
  };

  /**
   * @private
   */
  WebWiresheet.prototype.$getScrollbarHeight = function () {
    var jq = this.jq(),
      scrollDiv = jq.find('.ws-content-outer')[0];
    return scrollDiv.offsetHeight - scrollDiv.clientHeight;
  };

  /**
   * @private
   */
  WebWiresheet.prototype.$getThumbContainerJq = function () {
    return this.jq().find('.ws-thumb-container');
  };

  /**
   * @private
   */
  WebWiresheet.prototype.$updateThumbnailForOptions = function () {
    this.$updateThumbnailVisibility();
    this.$updateThumbnailPosition();
  };

  /**
   * @private
   */
  WebWiresheet.prototype.$updateThumbnailVisibility = function () {
    var showThumbnail = this.$options.showThumbnail,
      thumbJq = this.$getThumbContainerJq();
    thumbJq.toggleClass('ws-thumb-hidden', !showThumbnail);
  };

  /**
   * @private
   */
  WebWiresheet.prototype.$updateThumbnailPosition = function () {
    var thumbJq = this.$getThumbContainerJq(),
      scrollBarWidth = this.$getScrollbarWidth(),
      scrollBarHeight = this.$getScrollbarHeight(),
      options = this.$options || {},
      clearThumbCssPos = function clearThumbCssPos() {
        thumbJq.css('left', '');
        thumbJq.css('top', '');
        thumbJq.css('right', '');
        thumbJq.css('bottom', '');
      };
    // Defaults to right-top  
    var xDir = 'right',
      yDir = 'top',
      thumbnailX = options.thumbnailX || -1 * DEFAULT_THUMBNAIL_OFFSET,
      thumbnailY = options.thumbnailY || DEFAULT_THUMBNAIL_OFFSET;
    switch (Math.sign(thumbnailX)) {
      case -1:
        xDir = 'right';
        thumbnailX = scrollBarWidth + -1 * thumbnailX;
        break;
      case 1:
        xDir = 'left';
        break;
      default:
        break;
    }
    switch (Math.sign(thumbnailY)) {
      case -1:
        yDir = 'bottom';
        thumbnailY = scrollBarHeight + -1 * thumbnailY;
        break;
      case 1:
        yDir = 'top';
        break;
      default:
        break;
    }
    clearThumbCssPos();
    // finally if the x and y are still less than the default offset reset it
    thumbJq.css(xDir, thumbnailX < DEFAULT_THUMBNAIL_OFFSET ? DEFAULT_THUMBNAIL_OFFSET : thumbnailX);
    thumbJq.css(yDir, thumbnailY < DEFAULT_THUMBNAIL_OFFSET ? DEFAULT_THUMBNAIL_OFFSET : thumbnailY);
    thumbJq.css('transform-origin', xDir + ' ' + yDir);
  };

  /**
   * @private
   */
  WebWiresheet.prototype.$updateThumbnailDisplay = function () {
    var jq = this.jq(),
      wiresheetWidth = jq.width(),
      wiresheetHeight = jq.height(),
      thumbJq = this.$getThumbContainerJq(),
      thumbWidth = thumbJq.width(),
      thumbHeight = thumbJq.height(),
      maxThumbWidth = wiresheetWidth * THUMBNAIL_VISIBILITY_THRESHOLD,
      maxThumbHeight = wiresheetHeight * THUMBNAIL_VISIBILITY_THRESHOLD,
      widthThumbScale = maxThumbWidth / thumbWidth,
      heightThumbScale = maxThumbHeight / thumbHeight;
    if (widthThumbScale >= 1 && heightThumbScale >= 1) {
      thumbJq.css("transform", '');
      return;
    }
    var scaleFactor = Math.min(widthThumbScale, heightThumbScale);
    thumbJq.css("transform", "scale(".concat(scaleFactor, ")"));
  };
  return WebWiresheet;
});
