function _slicedToArray(r, e) { return _arrayWithHoles(r) || _iterableToArrayLimit(r, e) || _unsupportedIterableToArray(r, e) || _nonIterableRest(); }
function _nonIterableRest() { throw new TypeError("Invalid attempt to destructure non-iterable instance.\nIn order to be iterable, non-array objects must have a [Symbol.iterator]() method."); }
function _unsupportedIterableToArray(r, a) { if (r) { if ("string" == typeof r) return _arrayLikeToArray(r, a); var t = {}.toString.call(r).slice(8, -1); return "Object" === t && r.constructor && (t = r.constructor.name), "Map" === t || "Set" === t ? Array.from(r) : "Arguments" === t || /^(?:Ui|I)nt(?:8|16|32)(?:Clamped)?Array$/.test(t) ? _arrayLikeToArray(r, a) : void 0; } }
function _arrayLikeToArray(r, a) { (null == a || a > r.length) && (a = r.length); for (var e = 0, n = Array(a); e < a; e++) n[e] = r[e]; return n; }
function _iterableToArrayLimit(r, l) { var t = null == r ? null : "undefined" != typeof Symbol && r[Symbol.iterator] || r["@@iterator"]; if (null != t) { var e, n, i, u, a = [], f = !0, o = !1; try { if (i = (t = t.call(r)).next, 0 === l) { if (Object(t) !== t) return; f = !1; } else for (; !(f = (e = i.call(t)).done) && (a.push(e.value), a.length !== l); f = !0); } catch (r) { o = !0, n = r; } finally { try { if (!f && null != t["return"] && (u = t["return"](), Object(u) !== u)) return; } finally { if (o) throw n; } } return a; } }
function _arrayWithHoles(r) { if (Array.isArray(r)) return r; }
/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/wiresheet/rc/core/ViewModel
 */
define(['Promise', 'underscore', 'nmodule/js/rc/tinyevents/tinyevents'], function (Promise, _, tinyevents) {
  'use strict';

  ////////////////////////////////////////////////////////////////
  // ViewModel
  ////////////////////////////////////////////////////////////////

  /**
   * ViewModel that will back a Wiresheet.
   *
   * A ViewModel consists of a set of vertices and edges. To compare with the
   * Workbench version, a Component would translate to a vertex and a Link would
   * translate to an edge.
   *
   * @class
   * @alias module:nmodule/wiresheet/rc/core/ViewModel
   */
  var ViewModel = function ViewModel() {
    this.$counter = 0;

    // map from id to a vertex or edge entity
    this.$entities = {};

    // list of triples where subject/object are vertex ids and predicate is edge id
    this.$triples = [];
    tinyevents(this);
  };

  /**
   * @private
   * @returns {Promise}
   */
  ViewModel.prototype.$clear = function () {
    var _this = this;
    return Promise.all([this.getVertices(), this.getEdges()]).then(function (_ref) {
      var _ref2 = _slicedToArray(_ref, 2),
        vertices = _ref2[0],
        edges = _ref2[1];
      _this.$entities = {};
      _this.$triples = [];
      return Promise.all(vertices.map(function (vertex) {
        return _this.emitAndWait('vertexRemoved', vertex);
      }).concat(edges.map(function (edge) {
        return _this.emitAndWait('edgeRemoved', edge);
      })));
    });
  };

  /**
   * @private
   * @returns {string}
   */
  ViewModel.prototype.$getIdProperty = function () {
    return '__id';
  };

  /**
   * Get the entity associated with this id.
   *
   * @param {String|Object} id the id of the entity, or the entity itself if it
   * already exists in the db
   * @returns {Promise.<Object|null>} promise to be resolved with the entity,
   * or null if not found
   */
  ViewModel.prototype.get = function (id) {
    return Promise.resolve(this.$getEntity(id));
  };

  /**
   * Get the id associated with this entity.
   *
   * @param {String|Object} obj the entity whose id you want, or a string id
   * will be returned directly
   * @returns {String}
   */
  ViewModel.prototype.getId = function (obj) {
    if (typeof obj === 'string') {
      return obj;
    }
    if (obj) {
      var idProp = obj[this.$getIdProperty()];
      if (idProp) {
        return idProp;
      }
    }
    return _.findKey(this.$entities, function (val) {
      return val.entity === obj;
    }) || null;
  };

  /**
   * @private
   * @param {object} entity
   * @param {string} type 'vertex' or 'edge'
   * @param {string} [id] optionally specify the desired id
   * @returns {string}
   */
  ViewModel.prototype.$putEntity = function (entity, type, id) {
    var putId = id || this.getId(entity) || "id".concat(this.$counter++);
    entity[this.$getIdProperty()] = putId;
    this.$entities[putId] = {
      type: type,
      entity: entity
    };
    return putId;
  };

  /**
   * @private
   * @param {string} id
   * @returns {object}
   */
  ViewModel.prototype.$getEntity = function (id) {
    var entityId = this.getId(id);
    var obj = entityId && this.$entities[entityId];
    return obj ? obj.entity : null;
  };

  /**
   * Callback when an entity is being overwritten. Override if your ViewModel
   * subclass requires any special behavior when overwriting an existing entity.
   * @private
   * @param {object} existing the entity existing in the database
   * @param {object} newEntity the entity being saved into the database
   * @returns {object} the actual entity that should be saved into the database.
   * By default just returns newEntity directly.
   */
  ViewModel.prototype.$overwrite = function (existing, newEntity) {
    return newEntity;
  };

  /**
   * Adds a new entity to the ViewModel and resolves a unique ID for it.
   *
   * Note that the entity object will be altered by this action (it will receive
   * an ID property), so for now, an entity cannot be stored in two ViewModels
   * at once. Only store single-purpose data objects as entities.
   *
   * Will emit a `vertexAdded` or `vertexUpdated` tinyevent, depending on
   * whether the object is being added for the first time, or replacing an
   * existing value.
   * 
   * Will emit an 'put' event, that can be registered for any processing
   * that needs to be performed at the moment an entity is added to the database,
   * say, call a layout update on an entity.
   *
   * @param {Object} obj the object to be added
   * @param {String} [id] the desired ID. One will be generated if not given.
   * @returns {Promise.<String>} promise to be resolved with the ID of the
   * added entity, after all event handlers have resolved
   */
  ViewModel.prototype.put = function (obj, id) {
    var _this2 = this;
    var existing = this.$getEntity(id || this.getId(obj));
    if (existing) {
      existing = _.extend({}, existing);
    }
    var putId = this.$putEntity(obj, 'vertex', id);
    var newVertex = this.$getEntity(putId);
    if (existing) {
      newVertex = this.$overwrite(existing, newVertex);
    }

    // put will trigger a layout update
    return this.emitAndWait('put', newVertex).then(function () {
      var emit = existing ? _this2.emitAndWait('vertexUpdated', newVertex, existing) : _this2.emitAndWait('vertexAdded', newVertex);
      return emit.then(function () {
        return putId;
      });
    });
  };

  /**
   * Removes an entity from the ViewModel.
   *
   * If a vertex, will emit a `vertexRemoved` tinyevent with the vertex entity.
   * If an edge, will emit an `edgeRemoved` tinyevent with the edge triple.
   *
   * @param {String|Object} obj the entity to remove, or its id
   * @returns {Promise} promise to be resolved when the entity has been removed,
   * after all event handlers have resolved
   */
  ViewModel.prototype.del = function (obj) {
    var _this3 = this;
    var id = this.getId(obj);
    return this.get(id).then(function (entity) {
      if (!entity) {
        return null;
      }
      var entities = _this3.$entities;
      var entityObj = entities[id];
      var isEdge = entityObj.type === 'edge';
      delete entities[id];
      var eventArg = !isEdge && entity,
        edgesRemoved = [];
      _this3.$triples = _this3.$triples.filter(function (triple) {
        if (isEdge && !eventArg && triple.predicate === id) {
          edgesRemoved.push({
            subject: _this3.$getEntity(triple.subject),
            predicate: entity,
            object: _this3.$getEntity(triple.object)
          });
        } else {
          // When a vertext is deleted it removes all its links.
          // Make a list of all these removed links and emit "edgeRemoved" as well.
          if (triple.subject === id) {
            edgesRemoved.push({
              subject: entity,
              predicate: _this3.$getEntity(triple.predicate),
              object: _this3.$getEntity(triple.object)
            });
          } else if (triple.object === id) {
            edgesRemoved.push({
              subject: _this3.$getEntity(triple.subject),
              predicate: _this3.$getEntity(triple.predicate),
              object: entity
            });
          }
        }
        return triple.subject !== id && triple.predicate !== id && triple.object !== id;
      });
      if (isEdge) {
        return Promise.all(edgesRemoved.map(function (edge) {
          return Promise.all([_this3.emit('edgeRemoved', edge), _this3.emit('del', edge)]);
        }));
      } else {
        var promises = [_this3.emitAndWait('vertexRemoved', eventArg), _this3.emit('del', eventArg)];
        edgesRemoved.forEach(function (edge) {
          promises.push(_this3.emitAndWait('edgeRemoved', edge), _this3.emitAndWait('del', edge));
        });
        return Promise.all(promises);
      }
    });
  };

  /**
   * Creates or updates a link between a subject and an object in the ViewModel.
   *
   * Since a subject and object can have multiple links between them, if you
   * need to update link data, you must currently retrieve the current actual
   * link data, update it, and pass the same object (or an exact copy) back into
   * this function.
   *
   * Will emit an `edgeAdded` or an `edgeUpdated` tinyevent with the edge
   * triple, as appropriate.
   * 
   * Will emit an 'put' event, that can be registered for any processing
   * that needs to be performed at the moment an entity (a link in this case) is added to the database,
   * say, call a layout update on an entity.
   *
   * @param {Object} params
   * @param {String|Object} params.subject
   * @param {String|Object} params.object
   * @param {Object} params.predicate object to store as the link data
   * @returns {Promise.<String>} promise to be resolved with the ID of the
   * newly added link, after all event handlers have resolved - or rejected if
   * subject/object could not be found
   */
  ViewModel.prototype.link = function (_ref3) {
    var _this4 = this;
    var subject = _ref3.subject,
      object = _ref3.object,
      predicate = _ref3.predicate;
    var triples = this.$triples;
    var subjectId = this.getId(subject);
    var objectId = this.getId(object);
    var predicateId = this.getId(predicate);
    subject = this.$getEntity(subjectId);
    object = this.$getEntity(object);
    if (!subject) {
      return Promise.reject(new Error("subject id \"".concat(subjectId, "\" not found")));
    }
    if (!object) {
      return Promise.reject(new Error("object id \"".concat(objectId, "\" not found")));
    }
    var existing = this.$getEntity(predicateId);
    if (existing) {
      predicate = this.$overwrite(existing, predicate);
    }
    var linkId = this.$putEntity(predicate, 'edge');
    if (!existing) {
      triples.push({
        subject: subjectId,
        predicate: linkId,
        object: objectId
      });
    } else {
      var triple = triples.find(function (x) {
        return x.predicate === predicateId;
      });
      if (triple) {
        triple.subject = subjectId;
        triple.object = objectId;
      }
    }
    var newTriple = {
      subject: this.$getEntity(subjectId),
      predicate: predicate,
      object: this.$getEntity(objectId)
    };

    // put will trigger a layout update
    return this.emitAndWait('put', newTriple).then(function () {
      var event = existing ? 'edgeUpdated' : 'edgeAdded';
      return _this4.emitAndWait(event, newTriple).then(function () {
        return linkId;
      });
    });
  };

  /**
   * Retrieve all vertices from the view model.
   *
   * @returns {Promise.<Array.<Object>>} promise to be resolved with an array
   * of all vertices, unsorted
   */
  ViewModel.prototype.getVertices = function () {
    return Promise.resolve(Object.values(this.$entities).filter(function (e) {
      return e.type === 'vertex';
    }).map(function (e) {
      return e.entity;
    }));
  };

  /**
   * Retrieve view model triples, optionally filtered by
   * subject/predicate/object ID.
   *
   * @param {Object} [query]
   * @param {Object|String} [query.subject] the desired subject, or ID of same
   * @param {Object|String} [query.predicate] the desired predicate, or ID of
   * same
   * @param {Object|String} [query.object] the desired object, or ID of same
   * @returns {Promise.<Array.<Object>>} promise to be resolved with an unsorted
   * array of matching triples, each with `subject`, `predicate`, `object`
   * properties
   */
  ViewModel.prototype.getEdges = function () {
    var _this5 = this;
    var _ref4 = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {},
      subject = _ref4.subject,
      predicate = _ref4.predicate,
      object = _ref4.object;
    var subjectId = subject && this.getId(subject);
    var predicateId = predicate && this.getId(predicate);
    var objectId = object && this.getId(object);
    if (subject && !subjectId || predicate && !predicateId || object && !objectId) {
      //TODO: warn - s/p/o not in db
      return Promise.resolve([]);
    }
    var results = this.$triples.filter(function (triple) {
      return (!subject || triple.subject === subjectId) && (!predicate || triple.predicate === predicateId) && (!object || triple.object === objectId);
    });
    var edges = [];
    results.forEach(function (triple) {
      var subject = _this5.$getEntity(subjectId || triple.subject);
      var predicate = _this5.$getEntity(predicateId || triple.predicate);
      var object = _this5.$getEntity(objectId || triple.object);
      if (subject && predicate && object) {
        edges.push({
          subject: subject,
          predicate: predicate,
          object: object
        });
      }
    });
    return Promise.resolve(edges);
  };

  /**
   * Retrieve all vertex and edge objects from the database.
   * @returns {Promise.<Array.<Object>>} promise to be resolved with an unsorted
   * array of all vertex and edge objects.
   */
  ViewModel.prototype.getAllEntities = function () {
    return Promise.all([this.getVertices(), this.getEdges()]).then(function (_ref5) {
      var _ref6 = _slicedToArray(_ref5, 2),
        vs = _ref6[0],
        es = _ref6[1];
      return vs.concat(_.pluck(es, 'predicate'));
    }).then(_.compact);
  };
  return ViewModel;
});
