/**
 * @license Copyright 2012, Tridium, Inc. All Rights Reserved.
 */

/**
 * @fileOverview Label class
 * 
 * @author Gareth Johnson
 * @version 0.0.2.0
 */

/**
 * @private
 * @module mobile/px/widgets/Label
 */
define(['baja!', 'jquery', 'jquerymobile', 'Promise', 'underscore', 'mobile/px/util.px', 'mobile/px/widgets/Widget'], function (baja, $, jqm, Promise, _, pxUtil, Widget) {
  "use strict";

  var getHalign = pxUtil.getHalign,
    getValign = pxUtil.getValign,
    updateImage = pxUtil.updateImage,
    updateImagePosition = pxUtil.updateImagePosition,
    updateText = pxUtil.updateText;
  var LABEL_HTML = "<div class='widgetLabel'>" + "<div class='widgetInnerOuter'>" + "<div class='widgetLabelInner'>" + "<span class='bajaImage'/>" + "<span class='labelText'></span>" + "</div>" + "</div>" + "</div>";

  /**
   * The Widget that represents Niagara's 'bajaui:Label' Type.
   *
   * @private
   * @class
   * @alias module:mobile/px/widgets/Label
   * @extends module:mobile/px/widgets/Widget
   */
  var Label = function Label() {
    Widget.apply(this, arguments);
    this.$blinkTicket = baja.clock.expiredTicket;
    this.$imgDom = null;
    this.$textDom = null;
    this.$textDomShown = true;
    this.$imgDomShown = true;
    this.$innerDom = null;
  };
  Label.prototype = Object.create(Widget.prototype);
  Label.prototype.constructor = Label;

  /**
   * Update the DOM associated with this Widget.
   *
   * @param {Function} hasUpdate called to test whether a 
   *                               given Property can update the DOM.
   */
  Label.prototype.doUpdate = function (hasUpdate) {
    var that = this,
      dom = that.$dom,
      imgDom = that.$imgDom,
      textDom = that.$textDom,
      innerDom = that.$innerDom;

    // Blink
    if (hasUpdate("visible") || hasUpdate("blink")) {
      if (that.getBlink() && that.getVisible()) {
        if (that.$blinkTicket.isExpired()) {
          // Create the blink timer
          that.$blinkTicket = baja.clock.schedulePeriodically(function () {
            var visible = dom.css("visibility") === "visible";
            visible = !visible;
            dom.css("visibility", visible ? "visible" : "hidden");
          }, 1000);
        }
      } else {
        // Cancel any outstanding ticket
        that.$blinkTicket.cancel();
        if (that.getVisible()) {
          // Show the element
          dom.css("visibility", "visible");
        }
      }
    }

    // Font
    if (hasUpdate("font")) {
      that.applyCssFromSlot("font", textDom);
    }

    // Horizontal alignment
    if (hasUpdate("halign")) {
      var halign = getHalign(that);
      dom.css("text-align", halign);
      innerDom.css("text-align", halign);
      imgDom.css("text-align", halign);
    }

    // Vertical alignment
    if (hasUpdate("valign")) {
      var valign = getValign(that);
      dom.css("vertical-align", valign);
      imgDom.css("vertical-align", valign);
      innerDom.css("vertical-align", valign);
      textDom.css("vertical-align", valign);
    }

    // Background color
    if (hasUpdate("background")) {
      that.applyCssFromSlot("background", {
        dom: dom,
        colorSelector: "background",
        imageSelector: "background"
      });
    }

    // Text color
    if (hasUpdate("foreground")) {
      that.applyCssFromSlot("foreground", {
        dom: dom,
        colorSelector: "color"
      });
    }
    if (hasUpdate('enabled')) {
      dom.toggleClass('ui-state-disabled', !that.get('enabled'));
    }

    // if any of these properties change the label layout, we must update
    // image position
    return Promise.all([(hasUpdate('text') || hasUpdate('textFormat')) && updateText(that), hasUpdate('image') && updateImage(that), hasUpdate('textToIconAlign'), hasUpdate('textIconGap')]).then(function (updates) {
      return _.some(updates) && updateImagePosition(that, innerDom);
    });
  };

  /**
   * Append the Widget's DOM content onto the specified DOM element.
   *
   * @param {Object} dom DOM element to append content too.
   */
  Label.prototype.load = function (dom) {
    var _this = this;
    var label = this.$dom = $(LABEL_HTML);
    this.$textDom = label.find(".labelText");
    this.$imgDom = label.find(".bajaImage");
    this.$innerDom = label.find(".widgetLabelInner");
    if (!this.isAbsolutelyPositioned()) {
      label.css("padding", "7px 7px");
    }
    label.appendTo(dom);
    label.on('click', function () {
      _this.handleEvent('click');
      return false;
    });
  };
  return Label;
});
