/**
 * @license Copyright 2012, Tridium, Inc. All Rights Reserved.
 */

/**
 * @fileOverview Button class
 * 
 * @author Gareth Johnson
 * @version 0.0.2.0
 */

/**
 * @private
 * @module mobile/px/widgets/Button
 */
define(['baja!', 'jquery', 'jquerymobile', 'Promise', 'underscore', 'mobile/px/util.px', 'mobile/px/widgets/Widget'], function (baja, $, jqm, Promise, _, pxUtil, Widget) {
  "use strict";

  var getHalign = pxUtil.getHalign,
    getValign = pxUtil.getValign,
    updateImage = pxUtil.updateImage,
    updateImagePosition = pxUtil.updateImagePosition,
    updateText = pxUtil.updateText;

  /**
   * The Widget that represents Niagara's 'bajaui:Button' Type.
   *
   * @private
   * @class
   * @alias module:mobile/px/widgets/Button
   * @extends module:mobile/px/widgets/Widget
   */
  var Button = function Button() {
    Widget.apply(this, arguments);
    this.$blinkTicket = baja.clock.expiredTicket;
    this.$textDom = null;
    this.$buttonDom = null;
    this.$innerDom = null;
    this.$textDomShown = true;
    this.$imgDomShown = true;
    this.$pressed = false;
  };
  Button.prototype = Object.create(Widget.prototype);
  Button.prototype.constructor = Button;

  /**
   * Called after the Widget has loaded and any enhancements have 
   * been made to the DOM.
   */
  Button.prototype.postLoad = function () {
    var _this = this;
    // After the enhancement, the top level DOM element has changed
    // so we need to set the parent DOM as the base DOM element for the Widget
    var that = this,
      buttonDom = that.$dom,
      dom = buttonDom.parent(),
      textDom = $('<span class="ui-btn-text"></span>'),
      imgDom = $('<span class="bajaImage"/>'),
      innerDom = $('<div class="widgetInnerOuter"></div>');
    innerDom.append(imgDom);
    innerDom.append(textDom);
    innerDom.append(buttonDom.detach());
    dom.append(innerDom);
    dom.addClass("widgetButton");
    dom.on('click vmousedown', function (e) {
      return _this.handleEvent(e.type);
    });

    // Create image DOM element after page has been enhanced

    if (this.isAbsolutelyPositioned()) {
      innerDom.css({
        "display": "table-cell",
        "width": "100%"
      });
      dom.css('padding', '4px 7px');
    } else {
      dom.css('padding', '');
    }
    that.$dom = dom;
    that.$buttonDom = buttonDom;
    that.$textDom = textDom;
    that.$imgDom = imgDom;
    that.$innerDom = innerDom;
  };

  /**
   * Update the DOM associated with this Widget.
   *
   * @param {Function} hasUpdate called to test whether a 
   *                               given Property can update the DOM.
   */
  Button.prototype.doUpdate = function (hasUpdate) {
    // Need to get parent when manipulating this in jQuery Mobile
    var that = this,
      dom = that.$dom,
      innerDom = that.$innerDom,
      buttonDom = that.$buttonDom,
      textDom = that.$textDom,
      imgDom = that.$imgDom;

    // Font
    if (hasUpdate("font")) {
      that.applyCssFromSlot('font', textDom);
    }

    // Horizontal alignment
    if (hasUpdate("halign")) {
      var halign = getHalign(that);
      dom.css("text-align", halign);
      imgDom.css("text-align", halign);
    }

    // Vertical alignment
    if (hasUpdate("valign")) {
      var valign = getValign(that);
      dom.css("vertical-align", valign);
      innerDom.css("vertical-align", valign);
      imgDom.css("vertical-align", valign);
    }

    // Enabled  
    if (hasUpdate("enabled")) {
      buttonDom.button(that.isEnabled() ? "enable" : "disable");
    }

    // Visibility (need to hide parent)
    if (hasUpdate("visible")) {
      dom.toggle(that.get('visible'));
    }

    // Blink
    if (hasUpdate("blink")) {
      if (that.getBlink()) {
        if (that.$blinkTicket.isExpired()) {
          // Create the blink timer
          that.$blinkTicket = baja.clock.schedulePeriodically(function () {
            var wasVisible = innerDom.css("visibility") === "visible",
              isVisible = !wasVisible;
            innerDom.css("visibility", isVisible ? "visible" : "hidden");
          }, 1000);
        }
      } else {
        that.$blinkTicket.cancel();
        innerDom.css("visibility", "visible");
      }
    }

    // Background color
    if (hasUpdate("background")) {
      that.applyCssFromSlot("background", {
        dom: dom,
        colorSelector: "background",
        imageSelector: "background"
      });
    }

    // Text color
    if (hasUpdate("foreground")) {
      that.applyCssFromSlot("foreground", {
        dom: dom,
        colorSelector: "color"
      });
    }

    // if any of these properties change the label layout, we must update
    // image position
    return Promise.all([(hasUpdate("text") || hasUpdate("textFormat")) && updateText(that), hasUpdate("image") && updateImage(that), hasUpdate("textToIconAlign"), hasUpdate("textIconGap")]).then(function (updates) {
      return _.some(updates) && updateImagePosition(that, innerDom);
    });
  };

  /**
   * Append the Widget's DOM content onto the specified DOM element.
   *
   * @param {JQuery} dom DOM element to append content to.
   */
  Button.prototype.load = function (dom) {
    var buttonDom = this.$dom = $('<input type="button" data-theme="a">');
    buttonDom.appendTo(dom);
  };
  Button.prototype.doHandleEvent = function (eventName) {
    var _this2 = this;
    switch (eventName) {
      case 'vmousedown':
        this.$pressed = true;
        $(document).one('vmouseup', function () {
          _this2.$pressed = false;
          _this2.handleEvent('vmouseup');
        });
        break;
    }
  };
  Button.prototype.isPressed = function () {
    return this.$pressed;
  };
  return Button;
});
