/**
 * @license Copyright 2012, Tridium, Inc. All Rights Reserved.
 */

/**
 * @fileOverview Font class
 * 
 * @author Gareth Johnson
 * @version 0.0.2.0
 */
define(['baja!', 'mobile/px/gx/gxStyles'], function (baja, gxStyles) {
  "use strict";

  var strictArg = baja.strictArg,
    fontCache = {},
    notCurrentlySupported = "Not currently supported";

  /**
   * @class Font.
   * 
   * This is currently a partial implementation of Niagara's 'gx:Font' Type.
   * 
   * When creating a Simple, always use the 'make' method instead of creating a new Object.
   *
   * @name Font
   * @extends baja.Simple
   */
  function Font(obj) {
    var that = this;
    baja.callSuper(Font, that, arguments);
    that.$fontName = strictArg(obj.fontName, String);
    that.$size = strictArg(obj.size || 0, Number);
    that.$bold = strictArg(obj.bold || false, Boolean);
    that.$italic = strictArg(obj.italic || false, Boolean);
    that.$underline = strictArg(obj.underline || false, Boolean);
    that.$str = strictArg(obj.str || "", String);
  }
  baja.subclass(Font, baja.Simple);

  /**
   * Make a Font.
   */
  Font.make = function (obj) {
    throw new Error(notCurrentlySupported);
  };

  /**
   * Make a Font.
   */
  Font.prototype.make = function (obj) {
    return Font.make.apply(Font, arguments);
  };

  /**
   * Decode a Font from a String
   *
   * @param {String} str
   * @return {Font}
   */
  Font.prototype.decodeFromString = function (str) {
    // Attempt to get from the Cache
    if (fontCache.hasOwnProperty(str)) {
      return fontCache[str];
    }

    // Decode font
    var res,
      bold = false,
      italic = false,
      underline = false,
      size,
      fontName,
      i,
      font,
      regex = /(bold|italic|underline)? ?(bold|italic|underline)? ?(bold|italic|underline)? ?([0-9\.\,]+)pt (.+)/;
    res = regex.exec(str);
    if (!res || res && res.length !== 6) {
      throw new Error("Invalid font: " + str);
    }

    // Decode regular expression
    for (i = 0; i < res.length; ++i) {
      if (!res[i]) {
        continue;
      }
      if (i === 1 || i === 2 || i === 3) {
        if (res[i] === "bold") {
          bold = true;
        } else if (res[i] === "italic") {
          italic = true;
        } else if (res[i] === "underline") {
          underline = true;
        }
      } else if (i === 4) {
        size = Number(res[4]);
      } else if (i === 5) {
        fontName = res[5];
      }
    }

    // Create the font
    font = new Font({
      fontName: fontName,
      size: size,
      bold: bold,
      italic: italic,
      underline: underline,
      str: str
    });

    // Store the font in the Cache for later reference
    fontCache[str] = font;
    return font;
  };

  /**
   * Encode the Font to a String
   *
   * @return {String}
   */
  Font.prototype.encodeToString = function () {
    if (this.$str) {
      return this.$str;
    }
    var s = "";
    if (this.bold) {
      s += "bold ";
    }
    if (this.italic) {
      s += "italic ";
    }
    if (this.underline) {
      s += "underline ";
    }
    s += String(this.size) + "pt ";
    s += this.fontName;
    this.$str = s;
    return s;
  };

  /**
   * Default Font instance
   */
  Font.DEFAULT = new Font({
    fontName: "sans-serif",
    size: 12
  });

  /**
   * Null Font instance
   */
  fontCache["null"] = Font.NULL = new Font({
    fontName: "0",
    str: "null"
  });

  /**
   * Return the font name.
   *
   * @return {String}
   */
  Font.prototype.getFontName = function () {
    return this.$fontName;
  };

  /**
   * Return the size.
   *
   * @return {Number}
   */
  Font.prototype.getSize = function () {
    return this.$size;
  };

  /**
   * Return true if the font is bold.
   *
   * @return {Boolean}
   */
  Font.prototype.getBold = function () {
    return this.$bold;
  };

  /**
   * Return true if the font is italic.
   *
   * @return {Boolean}
   */
  Font.prototype.getItalic = function () {
    return this.$italic;
  };

  /**
   * Return true if the font is underline.
   *
   * @return {Boolean}
   */
  Font.prototype.getUnderline = function () {
    return this.$underline;
  };
  Font.prototype.getCssString = function () {
    // TODO: Support for HTML5 web fonts?
    // Lazily create the CSS for the font
    if (this.$fontCss === undefined) {
      var fcss = "";
      if (this !== Font.NULL) {
        if (this.$italic) {
          fcss += "italic ";
        }
        if (this.$bold) {
          fcss += "bold ";
        }
        fcss += this.$size + "px " + "\"" + this.$fontName + "\", serif";

        // Create CSS for changing the font
        this.$fontCss = fcss;
      }
    }
    return this.$fontCss;
  };

  /**
   * Update an HTML DOM element.
   *
   * @param dom the DOM Object to update
   */
  Font.prototype.update = function (dom) {
    gxStyles.applyFont(this, dom);
  };
  return Font;
});
