/**
 * @license Copyright 2012, Tridium, Inc. All Rights Reserved.
 */

/**
 * @fileOverview SpectrumBinding class
 * 
 * @author Gareth Johnson
 * @version 0.0.2.0
 */
define(['baja!', 'mobile/px/bindings/Binding', 'mobile/px/gx/Brush', 'mobile/px/gx/Color'], function (baja, Binding, Brush, Color) {
  "use strict";

  var callSuper = baja.callSuper;

  /**
   * @class SpectrumBinding.
   * 
   * A Component that represents a Niagara 'kitPx:SpectrumBinding' Type.
   *
   * @name SpectrumBinding
   * @extends Binding
   */
  function SpectrumBinding() {
    callSuper(SpectrumBinding, this, arguments);
  }
  baja.subclass(SpectrumBinding, Binding);

  /**
   * Return true if the Property is overridden.
   *
   * @param {baja.Property} prop the Property to test.
   * @return {Boolean} return true if the Property is overridden.
   */
  SpectrumBinding.prototype.isOverridden = function (prop) {
    if (prop.getName() === this.getWidgetProperty()) {
      return true;
    }

    // Call super method
    return callSuper("isOverridden", SpectrumBinding, this, arguments);
  };
  function solveSpectrumColor(spectrumBinding, value) {
    var red,
      blue,
      green,
      alpha,
      mRed,
      bRed,
      mGreen,
      bGreen,
      mBlue,
      bBlue,
      mAlpha,
      bAlpha,
      mid = spectrumBinding.getSetpoint(),
      delta = spectrumBinding.getExtent() / 2.0,
      lowColor = spectrumBinding.getLowColor(),
      midColor = spectrumBinding.getMidColor(),
      highColor = spectrumBinding.getHighColor();

    // solve for the color using a linear equation y = mx + b,
    // the y axis is the value being monitored, and the x
    // axis is the color (red, green, or blue)
    if (value < mid) {
      // handle current over boundary
      if (value < mid - delta) {
        return lowColor;
      }
      mRed = (midColor.getRed() - lowColor.getRed()) / delta;
      bRed = midColor.getRed() - mRed * mid;
      red = mRed * value + bRed;
      mGreen = (midColor.getGreen() - lowColor.getGreen()) / delta;
      bGreen = midColor.getGreen() - mGreen * mid;
      green = mGreen * value + bGreen;
      mBlue = (midColor.getBlue() - lowColor.getBlue()) / delta;
      bBlue = midColor.getBlue() - mBlue * mid;
      blue = mBlue * value + bBlue;
      mAlpha = (midColor.getAlpha() - lowColor.getAlpha()) / delta;
      bAlpha = midColor.getAlpha() - mAlpha * mid;
      alpha = mAlpha * value + bAlpha;
    } else {
      // handle current over boundary
      if (value > mid + delta) {
        return highColor;
      }
      mRed = (highColor.getRed() - midColor.getRed()) / delta;
      bRed = midColor.getRed() - mRed * mid;
      red = mRed * value + bRed;
      mGreen = (highColor.getGreen() - midColor.getGreen()) / delta;
      bGreen = midColor.getGreen() - mGreen * mid;
      green = mGreen * value + bGreen;
      mBlue = (highColor.getBlue() - midColor.getBlue()) / delta;
      bBlue = midColor.getBlue() - mBlue * mid;
      blue = mBlue * value + bBlue;
      mAlpha = (highColor.getAlpha() - midColor.getAlpha()) / delta;
      bAlpha = midColor.getAlpha() - mAlpha * mid;
      alpha = mAlpha * value + bAlpha;
    }
    return Color.make({
      red: red,
      green: green,
      blue: blue,
      alpha: alpha
    });
  }

  /**
   * Called when the binding is updated.
   */
  SpectrumBinding.prototype.update = function () {
    // Load the set point value
    var widget = this.getWidget();
    if (!this.isBound()) {
      return;
    }
    var prop = widget.getSlot(this.getWidgetProperty());
    if (prop !== null) {
      // Get target value
      var targetVal = this.$target.getObject();

      // convert target value
      var newVal = Number.getNumberFromINumeric(targetVal);

      // Solve the color
      var color = solveSpectrumColor(this, newVal);
      if (prop.getType().is("gx:Brush")) {
        widget.set({
          slot: prop,
          value: Brush.make({
            color: color
          })
        });
      }
      if (prop.getType().is("gx:Color")) {
        widget.set({
          slot: prop,
          value: color
        });
      }
    }

    // Call super
    return callSuper("update", SpectrumBinding, this, arguments);
  };
  return SpectrumBinding;
});
