/**
 * @copyright 2019 Tridium, Inc. All Rights Reserved.
 * @author Sai Komaravolu
 */

/* eslint-env browser */

define(['baja!', 'bajaux/Widget', 'Promise', 'nmodule/webEditors/rc/fe/fe', 'nmodule/nss/rc/model/Severity', 'css!nmodule/nss/rc/nss'], function (baja, Widget, Promise, fe, Severity) {
  'use strict';

  var exports = {};

  /**
   * Returns a blank severity object.
   * @static
   * @param {string} [displayName=params.name] severity display name
   * @param {string[]} [icon=[]] array of strings that define an icon for this severity level
   * @param {string} name severity name
   * @param {number} priority severity priority
   * @throws {Error} if name or priority is missing
   */
  exports.newSeverity = function (displayName, icon, name, priority) {
    return new Severity({
      displayName: displayName,
      icon: icon,
      name: name,
      priority: priority
    });
  };

  /**
   * Sorts the items by priority.
   * @static
   * @param {module:nmodule/nss/rc/model/Item[]} itemsToSort is list of items of type Item
   * @returns {module:nmodule/nss/rc/model/Item[]} after sorting by priority
   */
  exports.sortByPriority = function (itemsToSort) {
    return itemsToSort.sort(function (item1, item2) {
      return item2.getSeverity().getPriority() - item1.getSeverity().getPriority();
    });
  };

  /**
   * Filters item by the input severity.
   * @static
   * @param {module:nmodule/nss/rc/model/Item} itemsToFilter is list of items of type Item
   * @param {module:nmodule/nss/rc/model/Severity[]} severity the criteria to filter by name
   * @returns {module:nmodule/nss/rc/model/Item[]} after sorting by priority
   */
  exports.filterBySeverity = function (itemsToFilter, severity) {
    return itemsToFilter.filter(function (locItem) {
      return locItem.getSeverity().getName() === severity.getName();
    });
  };

  /**
   * @static
   * @param {module:nmodule/nss/rc/model/Item} itemsToFilter is list of items of type Item
   * @param {string} severityName the criteria to filter by name
   * @returns {module:nmodule/nss/rc/model/Item[]} filtered by severityName
   */
  exports.filterBySeverityName = function (itemsToFilter, severityName) {
    return itemsToFilter.filter(function (locItem) {
      return locItem.getSeverity().getName() === severityName;
    });
  };

  /**
   * Gets the baja object associated with the class in dom.
   * @param {JQuery} jq - Input dom to find the class
   * @param {string} className - Class to search.
   * @return {*}
   */
  exports.getWidgetForClass = function (jq, className) {
    return Widget["in"](jq.find(className));
  };
  exports.getTypeAndCountForSeverity = function (countsBySeverityArr) {
    var typeVsCountArr = [];
    countsBySeverityArr.forEach(function (countBySev) {
      typeVsCountArr.push({
        type: countBySev.severity.getName(),
        value: countBySev.count
      });
    });
    return typeVsCountArr;
  };

  /**
   * @returns {Promise.<object>} resolves to default color configuration for
   * security dashboard donut charts. Use as `color` Property on
   * `DonutChartWidget`.
   */
  exports.getSeverityColors = function () {
    //TODO: from lexicon
    return Promise.resolve({
      'securityStatusInfo': '#ccc',
      'securityStatusOK': '#7eb338',
      'securityStatusWarning': '#ffc627',
      'securityStatusAlert': '#ee3224'
    });
  };

  /**
   * If the ORD can be linked to, perform a hyperlink; otherwise does nothing.
   * @param {string|baja.Ord} ord
   * @param {string|baja.Ord} baseOrd
   */
  exports.hyperlink = function (ord, baseOrd) {
    if (exports.isLinkable(ord)) {
      ord = baja.Ord.make(ord);
      if (exports.isLinkable(baseOrd)) {
        ord = baja.Ord.make({
          base: baseOrd,
          child: ord
        });
      }
      ord = ord.normalize();
      if (window.niagara && window.niagara.env) {
        window.niagara.env.hyperlink(ord);
      } else {
        window.location.assign(ord.toUri());
      }
    }
  };

  /**
   * @param {string|baja.Ord} ord
   * @returns {boolean} true if the ORD can take us somewhere (not empty or default)
   */
  exports.isLinkable = function (ord) {
    return !!ord && baja.Ord.make(ord) !== baja.Ord.DEFAULT;
  };
  return exports;
});
