/*
Advanced widget state
=====================

A dynamic spandrel widget has two "sources of truth" that define how the widget
will render out. These two sources of truth are the two arguments to the
spandrel function: the loaded value, and the state of the Widget itself.

Usually, these values are sufficient to correctly render the widget. But
sometimes, the widget needs more information than can be conveyed by these
values. (Please see the bajaux tutorials in Doc Developer for example use
cases). As of 4.12, the widget state can now hold custom values.

In this example, just an Ord is not sufficient to hold the state of our widget:
we want to blank out the string editor when Null is selected, but also remember
what the previous entered value was, so the user doesn't lose their work if they
check and uncheck Null. Since an Ord doesn't have a field for "the previously
entered value," we need another place to hold that information.

So we "bind" the string editor to an ordString state key. This way, whenever
that editor is modified, its new value is propagated to the ordString state
value and not lost. On a render, we use the state value instead of the actual
loaded baja.Ord as the source of truth for the string editor. (We also set the
string editor to disabled to prevent unwanted updates while Null is checked.)

Notice the "lax" keyword on the string editor. By default, spandrel will not
overwrite a modified editor to prevent user changes from being discarded. But in
this case, when the null checkbox is changed, we want to change the value of the
string editor whether it's modified or not. The "lax" keyword allows
user-entered changes on a widget to be overwritten.

toState() and fromState() are then handy shortcuts for this style of widget. On
loading a value, we set the initial state values accordingly; and then when
reading a value out, we use the state instead.

Point this example at a slot of type baja:Ord, such as
station:|slot:/SpandrelExamples/Simples/ord.
*/

/** @jsx spandrel.jsx */

define([ 'baja!', 'bajaux/spandrel' ], function (baja, spandrel) {

  'use strict';

  return class OrdOrNull extends spandrel((ord, { isNull, ordString, self }) => {
    return (
      <div onUxModify={() => self.rerender()}>
        <any value={isNull ? '' : ordString}
             lax bind spandrelKey="ordString"
             enabled={!isNull} />
        <label>Null:</label>
        <any value={isNull}
             bind spandrelKey="isNull" />
      </div>
    );
  }, { strategy: 'niagara' }) {

    toState(ord) {
      return { isNull: ord.isNull(), ordString: String(ord) };
    }

    fromState({ isNull, ordString }) {
      return isNull ? baja.Ord.DEFAULT : baja.Ord.make(ordString);
    }
  };
});
