/**
 * @copyright 2018 Tridium, Inc. All Rights Reserved.
 */

/**
 * Created by e877411 on 10/20/2016.
 * D3StackedBarChart is designed keeping EquipmentStatus plotting in mind.
 * Each series will be plotted one above another in a single svg.
 * It has namesake y axis and ticks, tick lines and manually placed.
 */

define(['baja!', 'bajaux/Widget', 'bajaux/mixin/subscriberMixIn', 'jquery', 'd3', 'moment', 'nmodule/analytics/rc/util/analyticsUtil', 'lex!analytics', 'nmodule/analytics/rc/chart/types/AnalyticD3BarChart'], function (baja, Widget, subscriberMixIn, $, d3, moment, analyticsUtil, lexicon, AnalyticD3BarChart) {
  "use strict";

  var lex = lexicon[0];

  /**
   * Returns bar height to plot.
   * @param chart
   * @param ser
   * @param data
   * @returns {number}
   */
  function getBarHeight(chart, ser) {
    return chart.availableHeight() / ser.length;
  }

  /**
   * Returns Y Position to set the bar start position.
   * @param barHeight
   * @param index
   * @returns {number}
   */
  function getYPosition(barHeight, index) {
    return barHeight * index;
  }

  /**
   * @constructor
   * Base view for all web analytics charts
   */
  var AnalyticD3StackedBarChart = function AnalyticD3StackedBarChart(params) {
    AnalyticD3BarChart.apply(this, params);
    return this;
  };

  // Inheriting Analytics Base Chart from AnalyticD3BaseChart
  AnalyticD3StackedBarChart.prototype = Object.create(AnalyticD3BarChart.prototype);
  // Setting the constructor
  AnalyticD3StackedBarChart.prototype.constructor = AnalyticD3StackedBarChart;

  /**
   * Render the Stacked Bar Chart.
   * @param chartSettingsCollection
   */
  AnalyticD3StackedBarChart.prototype.draw = function (baseChart, series) {
    var serLength = 0,
      processedSeries = [],
      linesData = [],
      ticksData = [];
    series.map(function (ser) {
      serLength = d3.max([serLength, ser.data.length]);
    });
    // Never go inside if the series does not contain any data.
    if (serLength !== 0) {
      // Find the bar width of every bar that needs to be drawn
      var barWidth = analyticsUtil.getRndIntervalFrmAllSeries(series, baseChart);
      var barHeight = getBarHeight(baseChart, series),
        offsetHeight = 0.05 * barHeight; // Offset to push the bar to some pixels to align with bar height.

      series.map(function (ser, index) {
        var yPosition = getYPosition(barHeight, index);

        // Prepare the horizontal series separation lines data.
        if (index !== 0) {
          linesData.push({
            x: 0,
            y: yPosition
          });
        }

        // Prepare the ticks data
        ticksData.push({
          x: -30,
          y: yPosition + offsetHeight,
          text: lex.get("chart.equipment.op.on")
        });

        // Push off only if user chooses it.
        if (baseChart.getTabConfigDataModel().getOffStatus()) {
          ticksData.push({
            x: -30,
            y: yPosition + barHeight / 2,
            text: lex.get("chart.equipment.op.off")
          });
        }

        // Prepare the series data.
        ser.data = ser.data.map(function (e, idx) {
          e.barWidth = barWidth;
          e.barHeight = e.y ? barHeight * 0.95 : barHeight / 2;
          e.yPosition = e.y ? yPosition + offsetHeight : yPosition + barHeight / 2;
          e.color = ser.color;
          e.unit = ser.unit ? ser.unit : ser.unitSymbol ? ser.unitSymbol : "";
          return e;
        });
        processedSeries.push.apply(processedSeries, ser.data);
      });
    }

    // Draw the bars.
    var svg = baseChart.svg(),
      bar = svg.selectAll('rect').data(processedSeries).enter().append('rect').attr('x', function (d) {
        // sets the x position of the bar
        return baseChart.xScale(d.x);
      }).attr('y', function (d) {
        // sets the y position of the bar
        return d.yPosition;
      }).attr('width', function (d) {
        return d.barWidth;
      }) // sets the width of bar
      .attr('height', function (d) {
        // sets the height of bar
        return d.barHeight;
      }).attr('fill', function (d, i) {
        return d.color;
      });

    // Draw the horizontal series separation lines.
    svg.append("g").selectAll("line").data(linesData).enter().append("line").attr("x1", function (d) {
      return d.x;
    }).attr("y1", function (d) {
      return d.y;
    }).attr("x2", baseChart.availableWidth()).attr("y2", function (d) {
      return d.y;
    }).attr("stroke-width", 2).attr("stroke", "black");
    var inG = svg.append("g").selectAll("text").data(ticksData).enter().append("g");

    // Draw text On or Off or both on Y Axis depending on user choice.
    inG.append("text").attr("class", "label").text(function (d) {
      return d.text;
    }).attr("x", function (d) {
      return d.x;
    }).attr("y", function (d) {
      return d.y;
    }).attr("fill", "gray");

    // Draw the tick lines.
    inG.append("line").attr("x1", function (d) {
      return 0;
    }).attr("y1", function (d) {
      return d.y;
    }).attr("x2", -5).attr("y2", function (d) {
      return d.y;
    }).attr("stroke-width", 1).attr("stroke", "gray");

    // Append a helper line which will be moved for the tooltip.
    svg.append("g").append("line").attr("class", "hLine").attr("x1", 0).attr("x2", 0).attr("y1", baseChart.availableHeight()).attr("y2", baseChart.availableHeight()).attr("stroke", "black");
    // On the master G assign mouse events to move the line.
    d3.selectAll("g.masterG").on("mouseenter", function () {
      var mp = d3.mouse(this);
      d3.selectAll("line.hLine").attr("x1", mp[0]).attr("x2", mp[0]).attr("y1", 0).attr("y2", baseChart.availableHeight()).attr('stroke', "#000000").attr('stroke-width', 2).attr('fill', 'none');
    }).on("mousemove", function () {
      var mp = d3.mouse(this);
      d3.selectAll("line.hLine").attr("x1", mp[0]).attr("x2", mp[0]);
    }).on("mouseleave", function () {
      d3.selectAll("line.hLine").attr("x1", 0).attr("x2", 0).attr("y1", baseChart.availableHeight()).attr("y2", baseChart.availableHeight());
      // Remove tooltip div.
      baseChart.getToolTipDiv().transition().duration(200).style("opacity", 0);
    });

    // debugValue to support the sanity testing, should be enabled through analyticService.
    if (baseChart.isDebugEnabled) {
      d3.selectAll('rect').attr('debugValue', function (d) {
        return (d.recName ? d.recName : d.x) + ";" + d.y + (["null", undefined].contains(d.unit) ? "" : ";" + d.unit);
      });
    }
    return bar;
  };

  /**
   *
   * @param baseChart
   * @param ele
   * @param series
   */
  AnalyticD3StackedBarChart.prototype.toolTip = function (baseChart, ele, series) {
    var tooltipDiv = baseChart.getToolTipDiv();
    function setToolTip(bc, tTipDiv, mp, d) {
      var transform = bc.getTransformation();
      tTipDiv.html(bc.getToolTipHtml(d)).style("left", mp[0] * 0.9 + transform[0] + "px").style("top", mp[1] + transform[1] + "px");
    }
    ele.on("mouseenter", function (d) {
      tooltipDiv.transition().duration(200).style("opacity", 0.9);
      setToolTip(baseChart, tooltipDiv, d3.mouse(this), d);
    }).on("mousemove", function (d) {
      setToolTip(baseChart, tooltipDiv, d3.mouse(this), d);
    });
  };
  return AnalyticD3StackedBarChart;
});
