/**
 * @copyright 2018 Tridium, Inc. All Rights Reserved.
 */

/*jshint browser: true*/

/**
 * API Status: **Private**
 * @module nmodule/analytics/rc/model/AnalyticsBoxTableSeries
 */
define(['baja!', 'underscore', 'nmodule/analytics/rc/analyticsUtil', 'nmodule/webChart/rc/model/BaseSeries', 'nmodule/webChart/rc/model/modelUtil', 'nmodule/webChart/rc/webChartUtil', 'nmodule/webChart/rc/chunkUtil', 'Promise'], function (baja, _, analyticsUtil, BaseSeries, modelUtil, webChartUtil, chunkUtil, Promise) {
  "use strict";

  /**
   * Box Table Series for a chart. This Series displays BoxTable results.
   *
   * @class
   * @extends module:nmodule/webChart/rc/model/BaseSeries
   * @alias module:nmodule/analytics/rc/AnalyticsBoxTableSeries
   */
  var AnalyticsBoxTableSeries = function AnalyticsBoxTableSeries() {
    BaseSeries.apply(this, arguments);
  };
  AnalyticsBoxTableSeries.prototype = new BaseSeries();
  AnalyticsBoxTableSeries.prototype.constructor = AnalyticsBoxTableSeries;

  /**
   * Requires value and timestamp columns to be a chartable BoxSeries.
   * @param boxTable
   * @returns {boolean}
   */
  AnalyticsBoxTableSeries.hasRequiredColumns = function (boxTable) {
    var VALUE_COL = 'value',
      TIME_COL = 'timestamp';
    var valueCol = boxTable.getCol(VALUE_COL),
      timeCol = boxTable.getCol(TIME_COL);
    return valueCol && timeCol;
  };
  /**
   * Load the info for the BoxTable Series.
   */
  AnalyticsBoxTableSeries.prototype.loadInfo = function () {
    var that = this,
      value = that.$value,
      model = that.$model,
      timeZone = model.timeScale().getTimeZone(),
      ord = that.$ord;
    if (that.$loaded || that.$loading) {
      return modelUtil.resolvedPromise(); //nothing to do
    }

    //ensure we start with a refresh set of points
    that.$points.length = 0;
    that.$focusPoints = null;
    that.$samplingPoints = null;
    var VALUE_COL = 'value';
    var valueCol = value.getCol(VALUE_COL);
    that.setupFromValueColumn(valueCol);

    ////setup valueScale now that units are available
    return Promise.join(model.makeValueScale(that), analyticsUtil.matchPreferredTimeRange(model, that, ord, timeZone)).spread(function (valueScale) {
      that.$valueScale = valueScale;
      if (that.$seriesName && that.$seriesName.length > 0) {
        that.$displayName = that.$seriesName;
        that.$displayPath = that.$seriesName;
        return Promise.resolve(true);
      }
      var enterOnce = {
        value: true
      };
      chunkUtil.ajax("/analytics/aquery/boxTable/" + baja.SlotPath.escape(that.$ord), {
        progress: function progress(chunks, progressIndex) {
          updateUnits(chunks, enterOnce, model);
        }
      })["catch"](function () {});
      var pathOrd = analyticsUtil.navDisplayOrd(that.$ord);
      return webChartUtil.rpc("type:webChart:WebChartQueryRpc", "getDisplayPath", String(pathOrd)).then(function (response) {
        that.$displayPath = response.displayPath;
        var split = that.$displayPath.split('/');
        that.$displayName = split[split.length - 1];
      });
    });
  };

  /**
   * Re usable method to update the units data.
   * @param chunks
   * @param enterOnce
   * @param model
   */
  function updateUnits(chunks, enterOnce, model) {
    _.each(chunks, function (obj, index) {
      var jsonObj = JSON.parse(obj);
      if (!(jsonObj.type && jsonObj.type === "err")) {
        if (enterOnce.value) {
          if (model.$valueScales[0]) {
            model.$valueScales[0].$units = baja.$("baja:Unit").decodeFromString(jsonObj.u);
          }
          if (model.$seriesList[0]) {
            model.$seriesList[0].$units = baja.$("baja:Unit").decodeFromString(jsonObj.u);
            enterOnce.value = false;
          }
        }
      }
    });
  }

  /**
   * Load the series data.
   *
   * @returns {Promise}
   */
  AnalyticsBoxTableSeries.prototype.loadData = function () {
    var that = this,
      model = that.$model,
      ord = that.$ord,
      timeRange = model.timeRange();
    if (that.$loaded || that.$loading) {
      return modelUtil.resolvedPromise(); //nothing to do
    }
    that.$loading = true;

    //ensure we start with a refresh set of points
    that.$points.length = 0;
    that.$focusPoints = null;
    that.$samplingPoints = null;
    if (that.$dataOrd) {
      ord = that.$dataOrd;
    }
    var fullOrd = modelUtil.getFullScheduleOrd({
      ord: ord,
      timeRange: timeRange
    });
    var dataUri = "/webChart/query/boxTable/" + baja.SlotPath.escape(fullOrd);
    return modelUtil.chunkData(model, that, dataUri).then(function () {
      that.$loaded = true;
      that.$loading = false;
    });
  };

  /**
   * Sets up the value column of its backing collection.
   *
   * @private
   * @param {Object} valueCol the value column retrieved from a BoxTable
   */
  AnalyticsBoxTableSeries.prototype.setupFromValueColumn = function (valueCol) {
    // Get data to plot
    var that = this,
      facets = valueCol.getFacets();

    // declare instance variables
    that.$recordType = valueCol.getType().toString();
    that.$facets = facets || baja.Facets.DEFAULT;
    that.$units = that.$facets ? that.$facets.get("units", baja.Unit.DEFAULT) : baja.Unit.DEFAULT;
    that.$seriesName = that.$facets.get("seriesName");
  };
  AnalyticsBoxTableSeries.prototype.preparePoint = function (raw) {
    return modelUtil.prepareServletPoint(raw, this.$points);
  };

  /**
   * Display Name accessor.
   *
   * @param {String} [displayName] If specified, this will set the displayName.
   * @returns {String}
   */
  AnalyticsBoxTableSeries.prototype.displayName = function (displayName) {
    var that = this;
    return that.$displayName;
  };

  /**
   * Display Path accessor.
   * @param {String} [displayPath] If specified, this will set the displayPath.
   *
   * @returns {String}
   */
  AnalyticsBoxTableSeries.prototype.displayPath = function (displayPath) {
    var that = this;
    return that.$displayPath;
  };
  return AnalyticsBoxTableSeries;
});
