/*
 * Decompiled with CFR 0.152.
 */
package com.tridium.migrator;

import com.tridium.install.installable.DistributionManifest;
import com.tridium.migrator.BBackupDistMigrator;
import com.tridium.migrator.BBogMigrator;
import com.tridium.migrator.FileStats;
import com.tridium.migrator.InsecurePassPhraseSupplier;
import com.tridium.migrator.LazyConsolePassPhraseSupplier;
import com.tridium.migrator.MigConst;
import com.tridium.migrator.MigrationUtils;
import com.tridium.migrator.MigratorOrdConverter;
import com.tridium.migrator.baja.BServiceContainerConverter;
import com.tridium.nre.security.AESDecryptFunction;
import com.tridium.nre.security.EncryptionKeySource;
import com.tridium.nre.security.io.BogPasswordObjectEncoder;
import com.tridium.sys.Nre;
import com.tridium.sys.module.NModule;
import com.tridium.util.CommandLineArguments;
import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.io.Writer;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.security.AccessController;
import java.text.MessageFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.StringJoiner;
import java.util.TimeZone;
import java.util.function.Supplier;
import java.util.logging.ConsoleHandler;
import java.util.logging.FileHandler;
import java.util.logging.Filter;
import java.util.logging.Handler;
import java.util.logging.Level;
import java.util.logging.LogRecord;
import java.util.logging.Logger;
import java.util.logging.SimpleFormatter;
import java.util.regex.Pattern;
import javax.baja.file.BIFile;
import javax.baja.io.ValueDocDecoder;
import javax.baja.migration.BFileMigrator;
import javax.baja.migration.BIFileMigrator;
import javax.baja.migration.IOrdConverter;
import javax.baja.migration.MigrationException;
import javax.baja.migration.MigratorRegistry;
import javax.baja.nre.platform.RuntimeProfile;
import javax.baja.security.BPassword;
import javax.baja.sys.BComponent;
import javax.baja.sys.Sys;
import javax.baja.util.Lexicon;

public class Migrate {
    private static final String TSTAMP_PATTERN = "yyMMdd_HHmm";
    private static final String MIGRATE_LOG_NAME = "migration";
    private static final String SEL_CONTROLLER = "c";
    private static final String SEL_SUPER = "s";
    private static final String USAGE = "USAGE:                                                                           \n  n4mig [options] <source> [target]                                            \nparameters:                                                                    \n  source          station backup distribution file                             \n  target          station name or filename                                     \noptions:                                                                       \n  -help           print this USAGE information                                 \n  -o              overwrite the target location if it exists                   \n  -log:<level>    specify log level                                            \n    level is                                                                   \n      ALL | FINEST | FINER | FINE | CONFIG | INFO | WARNING | SEVERE | OFF     \n             (level specification is not case-sensitive)                       \n  -t:<template>   migration template to use (available types listed below)     \n  -version        print version info                                           \nadvanced options (not typically needed):                                       \n  -filePassPhrase:<passphrase>                                                 \n       use the <passphrase> for encrypting the migrated station bog file.      \n       NOTE: THIS MAY NOT BE SECURE.  The passphrase can be seen by others, or \n       retrieved by reviewing the shell's command history!                     \n  -keepTemp       do not clean up temp directories after execution             \n  -allowNonDist   skip the check for Niagara distribution file as input source;\n                  use for migrating bog snippets                               \n  -showSystemLogs show messages from all loggers; do not hide non-migrator logs\n\n<source> is required and must be a station backup distribution file.           \n\n[target] is optional.  If it is not specified, then it will be derived from    \nthe name of the selected station from the backup distribution file.            \nIf the supplied target is a relative path, then the result will be  placed     \nrelative to the root of niagara_user_home/stations.                            \n";
    private static final String SHARED = "shared";
    private static final Pattern COMPILE = Pattern.compile("shared", 16);
    private static final int PROGRESS_UPDATE_INTERVAL = 10;
    private static final Lexicon LEX = Lexicon.make((String)"migrator");
    private final Logger log = Logger.getLogger("migration");
    private String logfile;
    private LogFileHandler logFileHandler;
    private LogConsoleHandler consoleHandler;
    private Level logLevel;
    private final CommandLineArguments args;
    private boolean overwrite;
    private boolean showSystemLogs;
    private boolean allowNonDist;
    private Supplier<BPassword> passPhraseSupplier = new LazyConsolePassPhraseSupplier();
    private boolean useFilePassPhrase;
    private BIFile migrationTemplate;
    private final DistributionManifest distManifest;
    private final File source;
    private File target;
    private File targetStationHome;
    private File sharedStationHome;
    private Path sourcePath;
    private ArrayList<String> failures;
    private ArrayList<MigName> withOrds;
    private ArrayList<String> completionMessages;
    private File bogFile;
    private FileStats totalSize;
    private FileStats processedSize;
    private int lastProgress;
    private static AESDecryptFunction passwordDecryptFunction;

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void main(String[] rawArgs) {
        CommandLineArguments args = new CommandLineArguments(rawArgs);
        if (args.hasOption("ver")) {
            Migrate.version();
            return;
        }
        if (args.hasOption("help") || args.hasOption("h")) {
            Migrate.usage();
            return;
        }
        if (args.hasHelpOption()) {
            Migrate.usage();
            return;
        }
        if (args.parameters.length < 1) {
            Migrate.usage();
            return;
        }
        String isMig = AccessController.doPrivileged(() -> System.getProperty("niagara.migration"));
        try {
            System.setProperty("niagara.migration", "true");
            MigrationUtils.cleanMigTempDirs();
            Migrate migrate = new Migrate(args, null, null, null, null);
            migrate.logfile = Migrate.makeLogFileName(migrate.target);
            migrate.initialize();
            try (LogFileHandler ignored = migrate.initLogFileHandler(migrate.logfile);
                 LogConsoleHandler ignored2 = migrate.initConsoleHandler();){
                ignored.preventCompilerWarning();
                ignored2.preventCompilerWarning();
                migrate.log.setUseParentHandlers(false);
                ArrayList<String> migrationFailures = migrate.migrate();
                migrationFailures.forEach(migrate.log::severe);
            }
        }
        catch (Throwable throwable) {
            Logger.getLogger(MIGRATE_LOG_NAME).log(Level.SEVERE, MigrationUtils.errorResult(Logger.getLogger(MIGRATE_LOG_NAME), "Error migrating '" + String.join((CharSequence)" ", rawArgs) + '\'', throwable).get());
        }
        finally {
            if (!args.hasOption("keepTemp")) {
                try {
                    MigrationUtils.cleanMigTempDirs();
                }
                catch (IOException e) {
                    Logger.getLogger(MIGRATE_LOG_NAME).info(LEX.getText("migrate.couldNotCleanTempDir"));
                }
            }
            if (isMig != null) {
                System.setProperty("niagara.migration", isMig);
            } else {
                System.clearProperty("niagara.migration");
            }
        }
        System.exit(0);
    }

    void initialize() {
        this.failures = new ArrayList();
        this.withOrds = new ArrayList();
        this.completionMessages = new ArrayList();
        this.log.log(MigConst.CONSOLE, "\n\n------------------------------------------------------------");
        this.log.log(MigConst.CONSOLE, "Migrating station " + this.source.getName() + " to " + this.target.getName());
    }

    Migrate(CommandLineArguments args, File src, File tgt, DistributionManifest distManifest, Supplier<BPassword> passPhraseSupplier) throws Exception {
        boolean created;
        this.args = args;
        this.distManifest = distManifest;
        if (args.hasOption("o")) {
            this.overwrite = true;
        }
        if (args.hasOption("showSystemLogs")) {
            this.showSystemLogs = true;
        }
        if (args.hasOption("t")) {
            String migTemplate = args.getOption("t");
            try {
                BIFile[] templates = MigrationUtils.getMigStationTemplates();
                switch (migTemplate) {
                    case "c": {
                        this.migrationTemplate = templates[0];
                        break;
                    }
                    case "s": {
                        this.migrationTemplate = templates[1];
                        break;
                    }
                    default: {
                        int ndx = Integer.parseInt(migTemplate);
                        this.migrationTemplate = templates[ndx - 1];
                        break;
                    }
                }
            }
            catch (Exception e) {
                this.migrationTemplate = null;
            }
        } else {
            this.migrationTemplate = null;
        }
        this.setLogLevels(Migrate.parseLogLevel(args));
        if (src != null) {
            this.source = src;
        } else {
            String sourceParam = args.parameters[0];
            this.source = new File(sourceParam);
        }
        if (!this.source.exists()) {
            String message = "Error: No file or folder found for source: \"" + this.source + '\"';
            this.log.severe(message);
            throw new IOException("Source does not exist: \"" + this.source + '\"');
        }
        this.target = tgt != null ? tgt : this.makeTarget(args);
        if (passPhraseSupplier != null) {
            this.passPhraseSupplier = passPhraseSupplier;
        }
        if (this.source.getParentFile() != null && !this.target.getParentFile().exists() && !(created = this.target.getParentFile().mkdirs())) {
            throw new IOException("Could not create one or more parent directories for " + this.target);
        }
        if (args.hasOption("allowNonDist")) {
            if (this.source.toString().endsWith(".dist")) {
                throw new MigrationException(LEX.getText("migrate.nonDist"));
            }
            this.allowNonDist = true;
        }
        if (args.hasOption("filePassPhrase")) {
            this.useFilePassPhrase = true;
            this.passPhraseSupplier = new InsecurePassPhraseSupplier(args.getOption("filePassPhrase"));
        }
    }

    ArrayList<String> migrate() throws Exception {
        MigrationUtils.logInfo("migrate.start", this.source, this.target);
        Optional<String> srcValidation = this.validateSource(this.source, this.log);
        if (srcValidation.isPresent()) {
            this.failures.add(srcValidation.get());
            return this.failures;
        }
        Optional<String> tgtValidation = MigrationUtils.validateTarget(this.target, this.log, this.overwrite);
        if (tgtValidation.isPresent()) {
            this.failures.add(tgtValidation.get());
            return this.failures;
        }
        this.sourcePath = Paths.get(this.source.getAbsolutePath(), new String[0]);
        if (this.logFileHandler != null && this.logFileHandler.getFormatter() instanceof HtmlLogFormatter) {
            ((HtmlLogFormatter)this.logFileHandler.getFormatter()).setState(HtmlLogFormatter.State.MIGRATING);
        }
        this.totalSize = MigrationUtils.getFileStats(this.source);
        this.processedSize = new FileStats();
        this.lastProgress = 0;
        if (this.totalSize.fileCount > 1L) {
            this.log.log(MigConst.CONSOLE, String.format(LEX.getText("migrate.fileStats.start"), this.totalSize.fileCount, this.totalSize.byteCount));
        }
        if (this.source.isDirectory()) {
            this.targetStationHome = this.target;
            this.sharedStationHome = new File(this.target, SHARED);
            this.migrateFolder(this.source, this.target);
        } else {
            this.migrateFile(this.source, this.target);
        }
        if (!this.withOrds.isEmpty() && this.bogFile != null) {
            BogPasswordObjectEncoder bogEncodingInfo = ValueDocDecoder.getBogEncodingInfo((File)this.bogFile);
            ValueDocDecoder.BogDecoderPlugin plugin = new ValueDocDecoder.BogDecoderPlugin(this.bogFile);
            if (bogEncodingInfo != null && bogEncodingInfo.getKeySource() == EncryptionKeySource.external) {
                plugin.setPassPhrase(Optional.of(this.passPhraseSupplier.get()));
            }
            ValueDocDecoder d = new ValueDocDecoder((ValueDocDecoder.IDecoderPlugin)plugin);
            boolean zipped = d.isZipped();
            BComponent c = (BComponent)d.decodeDocument();
            MigratorOrdConverter ordCnv = new MigratorOrdConverter(c);
            for (MigName migNam : this.withOrds) {
                BIFileMigrator mig = migNam.mig;
                MigrationUtils.logInfo("migrate.updateOrds", migNam.fName);
                Optional failureMsg = mig.updateOrds((IOrdConverter)ordCnv, zipped);
                if (!failureMsg.isPresent()) continue;
                this.log.warning(MessageFormat.format(LEX.getText("migrate.updateOrdsFailed"), this.source));
                this.failures.add((String)failureMsg.get());
            }
        }
        if (this.logfile != null) {
            List<String> warningMessages;
            Optional<Object> formatter = Optional.empty();
            if (this.logFileHandler.getFormatter() instanceof HtmlLogFormatter) {
                formatter = Optional.of((HtmlLogFormatter)this.logFileHandler.getFormatter());
            }
            formatter.ifPresent(f -> f.setState(HtmlLogFormatter.State.COMPLETE));
            this.log.log(MigConst.CONSOLE, "\n");
            if (this.failures.isEmpty()) {
                this.log.log(MigConst.CONSOLE, MessageFormat.format(LEX.getText("migrate.completed"), this.source, this.target, this.logfile));
            } else {
                StringBuilder failureList = new StringBuilder();
                for (String s : this.failures) {
                    failureList.append(s).append('\n');
                }
                this.log.log(MigConst.CONSOLE, MessageFormat.format(LEX.getText("migrate.failed"), this.source, this.target));
                this.log.log(MigConst.CONSOLE, MessageFormat.format(LEX.getText("migrate.failureList"), failureList.toString()));
            }
            if (this.useFilePassPhrase) {
                this.completionMessages.add(LEX.getText("migrate.filePassPhraseUsed"));
            }
            if (!this.completionMessages.isEmpty()) {
                this.log.log(MigConst.CONSOLE, '\n' + LEX.getText("migrate.notes"));
                if (this.migrationTemplate != null) {
                    this.log.log(MigConst.CONSOLE, MessageFormat.format(LEX.getText("serviceContainerConverter.templateSelected"), this.migrationTemplate.getFileName()));
                }
                for (String msg : this.completionMessages) {
                    this.log.log(MigConst.CONSOLE, msg);
                }
            }
            if (!(warningMessages = this.logFileHandler.getMessages()).isEmpty()) {
                StringJoiner joiner = new StringJoiner("\n");
                formatter.ifPresent(f -> f.setState(HtmlLogFormatter.State.WARNINGS));
                warningMessages.forEach(joiner::add);
                this.log.log(Level.INFO, joiner.toString());
            }
            formatter.ifPresent(f -> f.setState(HtmlLogFormatter.State.FINISHED));
        }
        return this.failures;
    }

    private File makeTarget(CommandLineArguments args) {
        String targetParam = null;
        if (args.parameters.length > 1) {
            targetParam = args.parameters[1];
        }
        if (targetParam == null) {
            return this.makeTargetFileFromSource(this.source);
        }
        Path tgtPath = Paths.get(targetParam, new String[0]);
        if (tgtPath.isAbsolute()) {
            return tgtPath.toFile();
        }
        return this.makeTargetFileFromName(tgtPath);
    }

    private File makeTargetFileFromSource(File src) {
        if (this.log.isLoggable(Level.FINEST)) {
            this.log.finest("Migrate.makeTargetFileFromSource(): file=" + src + "; name=" + src.getName());
        }
        if (!src.exists()) {
            return null;
        }
        File nuh = Sys.getNiagaraUserHome();
        File stations = new File(nuh, "stations");
        return new File(stations, src.getName());
    }

    private File makeTargetFileFromName(Path tgtPath) {
        if (this.log.isLoggable(Level.FINEST)) {
            this.log.finest("Migrate.makeTargetFileFromName(): tgtPath=" + tgtPath);
        }
        File nuh = Sys.getNiagaraUserHome();
        File stations = new File(nuh, "stations");
        return new File(stations, tgtPath.getFileName().toString());
    }

    private void migrateFolder(File srcFolder, File tgtFolder) {
        MigrationUtils.logInfo("migrate.folder", srcFolder, tgtFolder);
        tgtFolder = this.updateTargetLocation(srcFolder, tgtFolder);
        if (!tgtFolder.mkdirs()) {
            this.log.log(Level.WARNING, MessageFormat.format(LEX.getText("migrate.cannotCreateDirs"), tgtFolder));
            this.failures.add(MessageFormat.format(LEX.getText("migrate.failedFor"), srcFolder) + ':' + MessageFormat.format(LEX.getText("migrate.cannotCreateDirs"), tgtFolder));
            return;
        }
        BIFileMigrator migrator = this.getMigrator(srcFolder, tgtFolder, this.logLevel);
        if (migrator != null) {
            if (this.log.isLoggable(Level.FINE)) {
                this.log.fine("  Migrator type " + migrator.getClass());
            }
            try {
                Optional failureMsg = migrator.migrate();
                if (failureMsg.isPresent()) {
                    this.log.warning(MessageFormat.format(LEX.getText("migrate.failedFor"), srcFolder));
                    this.failures.add((String)failureMsg.get());
                }
                migrator.addCompletionMessage(this.completionMessages);
            }
            catch (Exception e) {
                this.log.log(Level.WARNING, MessageFormat.format(LEX.getText("migrate.exceptionFor"), srcFolder), e);
                this.failures.add("Migration failed for " + srcFolder + ':' + e);
            }
            this.processedSize.add(MigrationUtils.getFileStats(srcFolder));
        } else {
            File[] sourceFiles = srcFolder.listFiles();
            if (sourceFiles != null) {
                for (File file : sourceFiles) {
                    try {
                        File targetChild = new File(tgtFolder, file.getName());
                        if (file.isDirectory()) {
                            this.migrateFolder(file, targetChild);
                            continue;
                        }
                        this.migrateFile(file, targetChild);
                    }
                    catch (Exception e) {
                        this.log.log(Level.WARNING, MessageFormat.format(LEX.getText("migrate.exceptionFor"), file), e);
                        this.failures.add("Migration failed for " + srcFolder + ':' + e);
                    }
                }
            }
        }
        this.updateProgress();
    }

    private void migrateFile(File srcFile, File tgtFile) throws Exception {
        Optional failureMsg;
        tgtFile = this.updateTargetLocation(srcFile, tgtFile);
        BIFileMigrator migrator = this.getMigrator(srcFile, tgtFile, this.logLevel);
        Objects.requireNonNull(migrator);
        MigrationUtils.logInfo("migrate.file", srcFile, tgtFile);
        if (this.log.isLoggable(Level.FINE)) {
            this.log.fine("  Migrator type " + migrator.getClass());
        }
        if ((failureMsg = migrator.migrate()).isPresent()) {
            this.log.severe(MessageFormat.format(LEX.getText("migrate.failedFor"), srcFile));
            this.failures.add((String)failureMsg.get());
        } else {
            if (migrator.mayContainOrds()) {
                this.withOrds.add(new MigName(migrator, srcFile.getName()));
            }
            if (migrator instanceof BBogMigrator && "config.bog".equals(srcFile.getName())) {
                this.bogFile = tgtFile;
            }
            if (tgtFile.equals(this.target) && migrator instanceof BBackupDistMigrator) {
                this.target = ((BBackupDistMigrator)migrator).getTarget();
            }
        }
        migrator.addCompletionMessage(this.completionMessages);
        if (this.migrationTemplate == null && migrator instanceof BBogMigrator) {
            this.migrationTemplate = ((BBogMigrator)migrator).getMigrationTemplate();
        }
        this.processedSize.add(MigrationUtils.getFileStats(srcFile));
        this.updateProgress();
    }

    private File updateTargetLocation(File srcFile, File tgtFile) {
        if (this.allowNonDist) {
            return tgtFile;
        }
        if (srcFile.equals(this.source)) {
            if (this.log.isLoggable(Level.FINE)) {
                this.log.fine("source " + srcFile + " is root source; leaving target at " + tgtFile);
            }
            return tgtFile;
        }
        Path srcPath = Paths.get(srcFile.getAbsolutePath(), new String[0]);
        Path sharedPath = this.sourcePath.relativize(srcPath);
        boolean isProtected = MigrationUtils.isProtected(sharedPath, this.distManifest);
        if (isProtected) {
            if (this.log.isLoggable(Level.FINE)) {
                this.log.fine("Leaving " + srcFile + " in protected_station_home; target folder is " + tgtFile);
            }
            return tgtFile;
        }
        String tgtFolderName = tgtFile.toString();
        String sharedPathStr = tgtFolderName.replace(this.targetStationHome.toString(), "").substring(1);
        if (!sharedPathStr.equals(SHARED) && sharedPathStr.startsWith(SHARED)) {
            sharedPathStr = COMPILE.matcher(sharedPathStr).replaceAll("").substring(1);
        }
        File newTgtFolder = new File(this.sharedStationHome, sharedPathStr);
        MigrationUtils.logInfo("migrate.moveSharedFolder", sharedPathStr, srcFile, newTgtFolder);
        return newTgtFolder;
    }

    private BIFileMigrator getMigrator(File srcFile, File tgtFile, Level logLevel) {
        BIFileMigrator migrator = MigratorRegistry.lookup((File)srcFile);
        if (migrator != null) {
            if (migrator instanceof BBackupDistMigrator && !this.source.getName().endsWith("dist")) {
                MigrationUtils.logInfo("migrate.foundDist", srcFile);
                migrator = new BFileMigrator();
            }
            migrator.initialize(srcFile, tgtFile, this.passPhraseSupplier, this.distManifest);
            migrator.setLogLevel(logLevel);
            if (migrator instanceof BFileMigrator) {
                ((BFileMigrator)migrator).setArgs(this.args);
            }
            if (migrator instanceof BBogMigrator) {
                ((BBogMigrator)migrator).setMigrationTemplate(this.migrationTemplate);
            }
        }
        return migrator;
    }

    List<String> getCompletionMessages() {
        return this.completionMessages;
    }

    private void setLogLevels(Level level) {
        this.logLevel = level;
        this.log.setLevel(this.logLevel);
        Logger.getLogger("migration.registry").setLevel(this.logLevel);
        this.log.config("Log level for migration log set to " + this.logLevel);
        if (!this.showSystemLogs) {
            Logger.getLogger("sys.engine").setLevel(Level.OFF);
            Logger.getLogger("alarm.database").setLevel(Level.OFF);
            try {
                Logger sysXmlLogger = Logger.getLogger("sys.xml");
                WrapperConsoleHandler sysXmlHandler = new WrapperConsoleHandler();
                sysXmlHandler.setFormatter(new LogFormatter());
                sysXmlLogger.addHandler(sysXmlHandler);
                sysXmlLogger.setUseParentHandlers(false);
            }
            catch (Exception e) {
                e.printStackTrace();
            }
        }
    }

    private static Level parseLogLevel(CommandLineArguments args) {
        Level logLevel = Level.INFO;
        if (args.hasOption("log")) {
            logLevel = Level.parse(args.getOption("log").toUpperCase());
        } else if (args.hasOption("l")) {
            logLevel = Level.parse(args.getOption("l").toUpperCase());
        }
        return logLevel;
    }

    private static String makeLogFileName(File target) {
        SimpleDateFormat format = new SimpleDateFormat(TSTAMP_PATTERN);
        format.setTimeZone(TimeZone.getDefault());
        String ts = format.format(new Date());
        return target + "_miglog_" + ts + ".html";
    }

    private LogFileHandler initLogFileHandler(String name) throws IOException {
        try {
            File logfile = new File(name);
            if (logfile.getParentFile() != null && !logfile.getParentFile().exists() && !logfile.getParentFile().mkdirs()) {
                throw new IOException("Could not create parent folder for migration logfile");
            }
            this.logFileHandler = new LogFileHandler(name);
            this.logFileHandler.setFormatter(new HtmlLogFormatter());
            this.log.addHandler(this.logFileHandler);
            this.log.finest("Log handler created for file at " + name);
        }
        catch (Exception e) {
            MigrationUtils.logSevere("migrate.couldNotCreateLog", e, new Object[0]);
            throw e;
        }
        return this.logFileHandler;
    }

    private LogConsoleHandler initConsoleHandler() {
        try {
            this.consoleHandler = new LogConsoleHandler();
            this.consoleHandler.setFormatter(new LogFormatter());
            this.consoleHandler.setFilter(new ConsoleLogFilter());
            this.log.addHandler(this.consoleHandler);
        }
        catch (Exception e) {
            MigrationUtils.logWarning("migrate.couldNotCreateLogHandler", e, new Object[0]);
        }
        return this.consoleHandler;
    }

    public static AESDecryptFunction getPasswordDecryptFunction() {
        return passwordDecryptFunction;
    }

    public static void setPasswordDecryptFunction(AESDecryptFunction value) {
        passwordDecryptFunction = value;
    }

    private void updateProgress() {
        if (this.totalSize.fileCount <= 1L) {
            return;
        }
        int fileProgress = (int)((double)this.processedSize.fileCount / (double)this.totalSize.fileCount * 100.0);
        int byteProgress = (int)((double)this.processedSize.byteCount / (double)this.totalSize.byteCount * 100.0);
        if (fileProgress - this.lastProgress > 10) {
            this.log.log(MigConst.CONSOLE, String.format(LEX.getText("migrate.overall.progress"), this.processedSize.fileCount, this.totalSize.fileCount, fileProgress, this.processedSize.byteCount, this.totalSize.byteCount, byteProgress));
            this.lastProgress = fileProgress;
        }
    }

    private static void version() {
        NModule migrationRt = Nre.getModuleManager().loadModule(MIGRATE_LOG_NAME, RuntimeProfile.rt);
        Migrate.println("  migration-rt.bajaVersion:   " + migrationRt.getBajaVersion());
        Migrate.println("  migration-rt.vendor:        " + migrationRt.getVendor());
        Migrate.println("  migration-rt.vendorVersion: " + migrationRt.getVendorVersion());
        NModule migratorWb = Nre.getModuleManager().loadModule("migrator", RuntimeProfile.wb);
        Migrate.println("  migrator-wb.bajaVersion:   " + migratorWb.getBajaVersion());
        Migrate.println("  migrator-wb.vendor:        " + migratorWb.getVendor());
        Migrate.println("  migrator-wb.vendorVersion: " + migratorWb.getVendorVersion());
    }

    private static void println(String s) {
        System.out.println(s);
    }

    private static void usage() {
        Migrate.println(USAGE);
        BServiceContainerConverter.printStationTemplateTypes();
    }

    private Optional<String> validateSource(File source, Logger log) {
        try {
            if (this.distManifest == null && !source.toString().endsWith(".dist")) {
                if (this.allowNonDist) {
                    log.severe("Source is not a Niagara Backup Distribution file; Assuming Niagara version 3.8.38");
                } else {
                    return Optional.of(LEX.getText("migrate.notBackupDist"));
                }
            }
            return Optional.empty();
        }
        catch (Exception e) {
            String message = MessageFormat.format(LEX.getText("migrate.cannotValidate"), source, "source");
            log.log(Level.SEVERE, message, e);
            return MigrationUtils.errorResult(log, message, e);
        }
    }

    private static class ConsoleLogFilter
    implements Filter {
        private ConsoleLogFilter() {
        }

        @Override
        public boolean isLoggable(LogRecord record) {
            return record.getLoggerName().startsWith(Migrate.MIGRATE_LOG_NAME) && (record.getLevel().intValue() == 820 || record.getLevel().intValue() >= Level.SEVERE.intValue());
        }
    }

    static class HtmlLogFormatter
    extends SimpleFormatter {
        private State state = State.START;
        private State lastState = State.START;
        private final LogFormatter formatter = new LogFormatter();
        private static final Pattern REPLACE_NEWLINE = Pattern.compile("\n", 16);
        private static final String CRLF = "\r\n";

        HtmlLogFormatter() {
        }

        @Override
        public String getHead(Handler h) {
            StringJoiner joiner = new StringJoiner("");
            joiner.add("<html>").add(CRLF).add("<body>").add(CRLF).add("<div id = \"report\">").add(CRLF).add("<p>").add("<a href=\"#miglog\">").add(LEX.getText("migrate.logname")).add("</a><br>").add(CRLF).add("<a href=\"#results\">").add(LEX.getText("migrate.results")).add("</a><br>").add(CRLF).add("</p>").add(CRLF).add("<h2>").add(LEX.getText("migrate.report")).add("</h2>").add(CRLF);
            return joiner.toString();
        }

        @Override
        public String getTail(Handler h) {
            return "</body>\r\n</html>";
        }

        @Override
        public String format(LogRecord record) {
            String msg = "";
            switch (this.state) {
                case START: {
                    msg = REPLACE_NEWLINE.matcher(record.getMessage()).replaceAll("<br>");
                    break;
                }
                case MIGRATING: {
                    StringJoiner joiner = new StringJoiner("");
                    if (this.lastState == State.START) {
                        joiner.add("</div><div id=\"miglog\">").add(CRLF).add("<p><h2>").add(LEX.getText("migrate.logname")).add("</h2></p><pre><code>").add(CRLF);
                    }
                    joiner.add(HtmlLogFormatter.safe(this.formatter.format(record)));
                    msg = joiner.toString();
                    break;
                }
                case COMPLETE: {
                    StringJoiner joiner = new StringJoiner("");
                    if (this.lastState == State.MIGRATING) {
                        joiner.add("</code></pre></div>").add(CRLF).add("<div id=\"results\">").add("<h2>").add(LEX.getText("migrate.results")).add("</h2><p>").add(CRLF);
                    }
                    joiner.add(REPLACE_NEWLINE.matcher(record.getMessage() + "<br>").replaceAll("<br>"));
                    msg = joiner.toString();
                    break;
                }
                case WARNINGS: {
                    StringJoiner joiner = new StringJoiner("");
                    if (this.lastState == State.COMPLETE) {
                        joiner.add("</p><div id =\"warnings\">").add(CRLF).add("<p><h3>").add(LEX.getText("migrate.warnings")).add("</h3></p>").add(CRLF).add("<pre>");
                    }
                    joiner.add(REPLACE_NEWLINE.matcher(record.getMessage()).replaceAll("<br>"));
                    msg = joiner.toString();
                    break;
                }
                case FINISHED: {
                    if (this.lastState == State.COMPLETE) {
                        return "</div>\r\n";
                    }
                    if (this.lastState != State.WARNINGS) break;
                    return "</pre></div></div>\r\n";
                }
            }
            this.lastState = this.state;
            return msg;
        }

        private static String safe(String s) {
            try {
                StringWriter sw = new StringWriter();
                int len = s.length();
                for (int i = 0; i < len; ++i) {
                    HtmlLogFormatter.safe(sw, s.charAt(i));
                }
                return sw.toString();
            }
            catch (IOException e) {
                return s;
            }
        }

        private static void safe(Writer out, int c) throws IOException {
            if (c < 32 || c > 126 || c == 39 || c == 34) {
                if (c == 10) {
                    out.write(10);
                    return;
                }
                if (c == 13) {
                    out.write(13);
                    return;
                }
                if (c == 9) {
                    out.write(9);
                    return;
                }
                out.write("&#x");
                out.write(Integer.toHexString(c));
                out.write(59);
            } else if (c == 60) {
                out.write("&lt;");
            } else if (c == 62) {
                out.write("&gt;");
            } else if (c == 38) {
                out.write("&amp;");
            } else {
                out.write((char)c);
            }
        }

        public void setState(State state) {
            this.state = state;
        }

        public static enum State {
            START,
            MIGRATING,
            COMPLETE,
            WARNINGS,
            FINISHED;

        }
    }

    private static class LogFormatter
    extends SimpleFormatter {
        private LogFormatter() {
        }

        @Override
        public String format(LogRecord record) {
            String f = "%2$s\n";
            if (record.getLevel() != Level.INFO) {
                f = "%1$s %2$s\n";
            }
            if (record.getLevel() == Level.SEVERE) {
                f = "!!" + f;
            }
            f = record.getLoggerName().equals(Migrate.MIGRATE_LOG_NAME) && record.getLevel() != MigConst.CONSOLE ? "\n*** " + f : "  " + f;
            String thrown = "";
            if (record.getThrown() != null) {
                StringWriter sw = new StringWriter();
                PrintWriter pw = new PrintWriter(sw);
                pw.println();
                record.getThrown().printStackTrace(pw);
                pw.close();
                thrown = sw.toString();
                if (record.getLevel().intValue() >= Level.WARNING.intValue()) {
                    f = f + ":\n%3$s";
                }
            }
            return String.format(f, record.getLevel().getLocalizedName(), this.formatMessage(record), thrown);
        }
    }

    static class WrapperConsoleHandler
    extends ConsoleHandler
    implements Closeable {
        @Override
        public void publish(LogRecord record) {
            String msg = MessageFormat.format(LEX.getText("migrate.wrapperMsg"), record.getLoggerName(), this.getFormatter().format(record));
            Logger.getLogger(Migrate.MIGRATE_LOG_NAME).log(Level.WARNING, msg);
        }
    }

    static class LogConsoleHandler
    extends ConsoleHandler
    implements Closeable {
        void preventCompilerWarning() {
        }
    }

    static class LogFileHandler
    extends FileHandler
    implements Closeable {
        List<String> warningMessages = new ArrayList<String>();

        public LogFileHandler(String filename) throws IOException {
            super(filename);
        }

        void preventCompilerWarning() {
        }

        @Override
        public void publish(LogRecord record) {
            if (record.getLoggerName().equals(Migrate.MIGRATE_LOG_NAME) && record.getLevel().intValue() >= Level.WARNING.intValue()) {
                this.warningMessages.add(record.getMessage());
            }
            super.publish(record);
        }

        public List<String> getMessages() {
            return this.warningMessages;
        }
    }

    static final class MigName {
        final BIFileMigrator mig;
        final String fName;

        MigName(BIFileMigrator m, String fn) {
            this.mig = m;
            this.fName = fn;
        }
    }
}

