/* eslint-disable promise/catch-or-return,no-unused-vars,promise/no-return-wrap */
/**
 * @copyright 2018 Tridium, Inc. All Rights Reserved.
 */

/*jshint browser: true */
/**
 * API Status: **Private**
 * @module nmodule/analytics/rc/report/fe/AnalyticNavTree
 */
define(['baja!', 'jquery', 'Promise', 'underscore', 'bajaux/events', 'bajaux/dragdrop/dragDropUtils', 'nmodule/webEditors/rc/fe/fe', 'nmodule/webEditors/rc/fe/baja/BaseEditor', 'nmodule/webChart/rc/fe/color/ColorEditor', 'nmodule/analytics/rc/report/util/AnalyticGroupTreeNode', 'hbs!nmodule/analytics/rc/report/templates/UxAnalyticNavTree', 'nmodule/analytics/rc/report/util/reportWidgetEvents', 'nmodule/webEditors/rc/wb/menu/CommandGroupContextMenu', 'nmodule/webEditors/rc/wb/mixin/ContextMenuSupport', 'nmodule/webEditors/rc/wb/menu/menuAgents', 'lex!analytics', 'baja!analytics:AnalyticNodeGroup', 'css!nmodule/analytics/rc/report/styles/uxstyles', 'css!nmodule/webEditors/rc/wb/tree/NavTreeStyle'], function (baja, $, Promise, _, events, dragDropUtils, fe, BaseEditor, ColorEditor, AnalyticGroupTreeNode, uxAnalyticNavTree, reportWidgetEvents, CommandGroupContextMenu, addContextMenuSupport, menuAgents, lexs, types) {
  'use strict';

  var DESTROY_EVENT = events.DESTROY_EVENT,
    LOAD_EVENT = events.LOAD_EVENT,
    MODIFY_EVENT = events.MODIFY_EVENT,
    MENU_MODULE_ID = 'nmodule/analytics/rc/report/util/analyticGroupMenuAgent',
    ACTIVATED_EVENT = 'navTree:activated',
    lex = lexs[0];

  ////////////////////////////////////////////////////////////////
  // Exports
  ////////////////////////////////////////////////////////////////

  /**
   * Allows for displaying, expanding, and collapsing a single node in a tree.
   *
   * A single `AnalyticNavTree` instance will show a group's color selector
   * and display value,
   * an expand/collapse button, and a list containing the node's children.
   *
   * The kid list will initially be collapsed.
   *
   * @class
   * @alias module:nmodule/analytics/rc/report/fe/AnalyticNavTree
   * @extends module:nmodule/webEditors/rc/fe/baja/BaseEditor
   * @param {Object} [params]
   * @param {Boolean} [params.expanded=false] set to true if this tree should
   * be immediately expanded on first load
   * @param {Boolean} [params.loadKids=true] set to false if this tree should
   * not attempt to load its children before it is expanded
   * @param {Boolean} [params.enableHoverPreload=false] set to true if child
   * nodes should start to preload when the mouse hovers over the expand
   * button (to shave a bit of time off child node expansion)
   */
  var AnalyticNavTree = function AnalyticNavTree(params) {
    var that = this;
    BaseEditor.apply(that, arguments);
    that.$registerContextMenu();
  };
  AnalyticNavTree.prototype = Object.create(BaseEditor.prototype);
  AnalyticNavTree.prototype.constructor = AnalyticNavTree;

  /**
   * Return true if this nav tree is currently selected/highlighted by the user.
   *
   * @returns {Boolean}
   */
  AnalyticNavTree.prototype.isSelected = function () {
    return this.$selected;
  };

  /**
   * Indicates that this node has been "activated" by the user, say by double-
   * clicking it. In contrast with "selected", say by a single click. Triggers
   * `AnalyticNavTree.ACTIVATED_EVENT`.
   */
  AnalyticNavTree.prototype.activate = function () {
    this.trigger(ACTIVATED_EVENT);
  };

  /**
   * Sets up initial HTML for the nav tree node.
   * @param {jQuery} dom
   */
  AnalyticNavTree.prototype.doInitialize = function (dom) {
    var that = this;
    dom.addClass("navEditor");
    dom.html(uxAnalyticNavTree());
    dom.on([DESTROY_EVENT, LOAD_EVENT].join(' '), '.editor', false);
    dom.on(MODIFY_EVENT, '.editor', function () {
      that.setModified(true);
      return false;
    });
    dom.children('.group-content').on('dragover dragenter', function (e) {
      if (that.value().isDropTarget()) {
        e.preventDefault();
      }
    }).on('drop', function (e) {
      var node = that.value();
      if (node.isDropTarget()) {
        var clipboard = e.originalEvent.dataTransfer;
        dragDropUtils.fromClipboard(clipboard).then(function (envelope) {
          return envelope.toValues();
        }).then(function (values) {
          node.doDrop(values).then(function (groupNode) {
            // Trigger an event
            that.jq().trigger(reportWidgetEvents.RE_RENDER_NODES_REQUEST, {
              data: groupNode
            });
            // Set modified to true
            that.setModified(true);
          });
        });
        return false;
      }
    });
    dom.children('button').on('click', function (event) {
      event.stopPropagation();
      var p;
      if (that.$isExpanded()) {
        p = that.collapse();
      } else {
        p = that.expand();
      }
    });
    that.jq().on(reportWidgetEvents.RE_RENDER_NODES_REQUEST, function (event, data) {
      that.$loadNodes(data.data, data.meta);
      that.setModified(true);
    });
    return BaseEditor.prototype.doInitialize.apply(this, arguments);
  };

  /**
   *
   * @param {module:nmodule/webEditors/rc/wb/tree/TreeNode} value
   * @returns {Promise}
   */
  AnalyticNavTree.prototype.doLoad = function (value) {
    if (!(value instanceof AnalyticGroupTreeNode)) {
      return Promise.reject(new Error('AnalyticGroupTreeNode required'));
    }
    var group = value.getComponent();
    var prop = this.properties();
    var that = this,
      displayElement = that.$getDisplayElement(),
      colorElement = that.$getColorSelectionElement();
    displayElement.text(baja.SlotPath.unescape(group.getDisplayName()));
    displayElement.attr('guid', group.getGuid());
    var showColorElement = prop.getValue("showColorSelector");
    var pNodeList = that.$loadNodes(value);
    var cePromise = Promise.resolve();
    if (showColorElement) {
      cePromise = fe.buildFor({
        dom: colorElement,
        value: group.getSeriesColor(),
        type: ColorEditor,
        formFactor: 'mini'
      }).then(function (ed) {
        ed.jq().on(events.MODIFY_EVENT, function (evt) {
          that.setModified(true);
        });
        return Promise.resolve(ed);
      });
    }
    var expand = that.expand();
    return Promise.join(cePromise, expand, pNodeList, function (fePromiseColor, fePromiseExpand, gePromiseNodes) {
      that.$colorEditor = fePromiseColor;
      that.registerNodeElementEvents();
    });
  };

  /**
   * Register right click event handler
   */
  AnalyticNavTree.prototype.registerNodeElementEvents = function () {
    // Now give a right click flag
    var that = this;
    var nodes = $('.node-list > li', that.jq());
    _.each([nodes], function (elem) {
      that.registerRightClickOnElem(elem);
      that.registerMouseEvents(elem);
    });
  };

  /**
   * Register the right-click on individual elements
   */
  AnalyticNavTree.prototype.registerRightClickOnElem = function (elem) {
    var that = this;
    elem.mousedown(function (event) {
      switch (event.which) {
        case 3:
          that.$clickedElement = $(this);
          break;
        default:
          that.$clickedElement = false;
      }
    });
  };
  /**
   * Register the right-click on individual elements
   */
  AnalyticNavTree.prototype.registerMouseEvents = function (elem) {};
  /**
   *  Returns the 'AnalyticNodeGroup' with the components
   *  underneath
   */
  AnalyticNavTree.prototype.doRead = function () {
    var groupNode = this.value(),
      comp = groupNode.getComponent();
    if (this.$colorEditor) {
      return this.$colorEditor.read().then(function (color) {
        comp.setSeriesColor(color);
        return Promise.resolve(comp);
      });
    } else {
      return Promise.resolve(comp);
    }
  };
  AnalyticNavTree.prototype.expand = function () {
    this.$getKidsListElement().show();
    return this.$updateButton(true);
  };
  AnalyticNavTree.prototype.collapse = function () {
    this.$getKidsListElement().hide();
    return this.$updateButton(false);
  };
  AnalyticNavTree.prototype.getSubject = function (dom) {
    var that = this;
    if (that.$clickedElement) {
      if (that.$clickedElement.hasClass('context-menu')) {
        return [this.value()];
      } else {
        // It has to be LI
        var index = that.$clickedElement.prop("gid") || 0;
        return [this.value().getKid(index)];
      }
    } else {
      return [];
    }
  };
  /**
   *
   */
  AnalyticNavTree.prototype.getContextMenuSelector = function () {
    return ".node-list > li";
  };
  /**
   * Removes the `NavTree` CSS class, unsubscribes for events from the
   * backing `TreeNode`, and destroys all child editors. Essentially, wipes
   * out everything from here down.
   * @returns {*}
   */
  AnalyticNavTree.prototype.doDestroy = function () {
    this.jq().remove();
  };

  //Private methods
  /**
   * Load all the child nodes
   * @param value
   */
  AnalyticNavTree.prototype.$loadNodes = function (value, renameInfo) {
    var that = this;
    var t = that.$getKidsListElement(),
      pr = that.properties();
    t.empty();
    var groupComponent = value.getComponent(),
      grpDispElem = that.$getDisplayElement();
    grpDispElem.html(baja.SlotPath.unescape(groupComponent.getDisplayName()));
    if (renameInfo && renameInfo.oldName) {
      grpDispElem.prop("guid", renameInfo.oldName);
    }
    var kids = value.getKids();
    // Now Add the children
    kids.then(function (subNodes) {
      if (subNodes.length === 0) {
        var dispOrd = pr.getValue("dispOrd");
        grpDispElem.attr('guid', groupComponent.getGuid());
        if (dispOrd === false) {
          // Indicate the node/group is not there.
          grpDispElem.addClass('no-ord');
          var childNodes = groupComponent.getSlots().is("analytics:AnalyticNode").toValueArray();
          dispOrd = lex.get({
            key: "report.noord.err",
            args: [childNodes.length === 1 ? childNodes[0].getNodePath() : ""]
          });
        }
        grpDispElem.attr("title", dispOrd);
      }
      for (var i = 0; i < subNodes.length; i++) {
        var subNode = subNodes[i];
        var kidEl = $('<li>');
        var c = subNode.getComponent();
        kidEl.html(baja.SlotPath.unescape(c.getDisplayName()));
        var slotPath = c.getSlotPath();
        kidEl.prop("slotPath", slotPath);
        var title = subNode.getNavNodeDisplay();
        if (title === false) {
          // Indicate the node/group is not there.
          kidEl.addClass('no-ord');
          title = lex.get({
            key: "report.noord.err",
            args: [slotPath]
          });
        }
        kidEl.prop("title", title);
        kidEl.prop("gid", i);
        $('<button class=" subNode expand collapsed" type="button"></button>').prependTo(kidEl);
        kidEl.appendTo(t);
        that.registerRightClickOnElem(kidEl);
      }
      return Promise.resolve(t);
    });
    $(t).on("click", "button.subNode", function (e) {
      e.stopPropagation();
      $(this).toggleClass("collapsed");
      $(this).toggleClass("expanded");
    });
  };
  /**
   *  Register the context menu agent
   */
  AnalyticNavTree.prototype.$registerContextMenu = function () {
    var that = this,
      menuAgentModuleId = MENU_MODULE_ID;
    var reg1 = menuAgents.register("analytics:AnalyticNode", menuAgentModuleId);
    Promise.all([reg1]).then(function () {
      addContextMenuSupport(that);
    });
  };
  /**
   * Updates the CSS classes and disabled status of the expand/collapse button
   * to reflect whether the tree can be expanded or collapsed.
   *
   * @private
   * @returns {Promise}
   */
  AnalyticNavTree.prototype.$updateButton = function (expand) {
    var button = this.$getButton();
    button.toggleClass('expanded', expand).toggleClass('collapsed', !expand);
  };
  AnalyticNavTree.prototype.$isExpanded = function () {
    var button = this.$getButton();
    return button.hasClass('expanded');
  };
  /**
   * Return the element for the expand/collapse button.
   * @private
   * @returns {jQuery}
   */
  AnalyticNavTree.prototype.$getButton = function () {
    return this.jq().children('button');
  };

  /**
   * Return the element to load the node's display value.
   * @private
   * @returns {jQuery}
   */
  AnalyticNavTree.prototype.$getDisplayElement = function () {
    return this.jq().children('.group-content').children('.group-display-name');
  };

  /**
   * Return the element to load the node's icon.
   * @private
   * @returns {jQuery}
   */
  AnalyticNavTree.prototype.$getColorSelectionElement = function () {
    return this.jq().children('.group-content').children('.group-color');
  };

  /**
   * Return the <ul> element that holds the list items to hold `NavTree`s for
   * the child nodes.
   * @private
   * @returns {jQuery}
   */
  AnalyticNavTree.prototype.$getKidsListElement = function () {
    return this.jq().children('ul');
  };
  /**
   * Reset the "guid" prop of display element
   */
  AnalyticNavTree.prototype.resetGuid = function () {
    var guidElem = this.$getDisplayElement();
    guidElem.prop("guid", false);
  };
  return AnalyticNavTree;
});
