/**
 * @copyright 2018 Tridium, Inc. All Rights Reserved.
 */
define(['baja!', 'bajaux/Widget', 'bajaux/mixin/subscriberMixIn', 'jquery', 'd3', 'underscore', 'nmodule/analytics/rc/util/analyticsUtil', 'nmodule/analytics/rc/chart/base/AnalyticsBaseChart', 'nmodule/analytics/rc/chart/base/AnalyticsTabConfigBaseModel'], function (baja, Widget, subscriberMixIn, $, d3, _, analyticsUtil, AnalyticsBaseChart, AnalyticsTabConfigBaseModel) {
  "use strict";

  /**
   * @constructor
   * Base view for all web analytics charts
   */
  var AnalyticD3BaseChart = function AnalyticD3BaseChart(params) {
    AnalyticsBaseChart.call(this, $.extend({}, params));
    subscriberMixIn(this);
    this.minValue = baja.Float.MAX_VALUE.valueOf();
    this.maxValue = baja.Float.MIN_VALUE.valueOf();
    this.minDuration = baja.Float.MAX_VALUE.valueOf();
    this.maxDuration = baja.Float.MIN_VALUE.valueOf();
    this.$svg = undefined;
    this.textSize = 0;
    this.legendIconSize = 0;
    this.generalIconSize = 0;
    this.supportedResolutions = [100, 300, 500, 700, 900, 1100, 1300];
    this.supportedTickCounts = [1, 2, 3, 4, 5, 6, 10];
    this.resolutionScale = d3.scale.threshold().domain(this.supportedResolutions).range(this.supportedTickCounts);
    this.textScale = d3.scale.linear().domain([100, 1300]).range([8, 25]);
    this.legendIconScale = d3.scale.linear().domain([100, 1300]).range([8, 30]);
    this.generalIconScale = d3.scale.linear().domain([100, 1300]).range([8, 30]);
    this.$textLengthLimit = 20;
  };
  // Inheriting Analytics Base Chart from AnalyticsBaseChart
  AnalyticD3BaseChart.prototype = Object.create(AnalyticsBaseChart.prototype);
  // Setting the constructor
  AnalyticD3BaseChart.prototype.constructor = AnalyticD3BaseChart;

  /**
   *
   * @returns {undefined|*}
   */
  AnalyticD3BaseChart.prototype.initSvg = function () {
    var that = this;
    var outerDivWidth = that.getChartPaneWidth();
    var outerDivHeight = that.getChartPaneHeight();
    var parentNode = that.jq();
    var margin = that.chartMargins();

    // Define SVG chart
    this.$svg = d3.select(parentNode.find('.chartArea')[0]).append("div").classed("svg-container", true).attr("style", "height:" + outerDivHeight + "px;").append("svg").attr("preserveAspectRatio", "xMinYMin meet").attr("viewBox", "0 0 " + outerDivWidth + " " + outerDivHeight).classed("svg-content-responsive", true).append("g").attr("class", "masterG").attr("transform", "translate(" + margin.left + "," + margin.top + ")");
    return this.$svg;
  };

  /**
   * Returns the chart div for the tool tip.
   */
  AnalyticD3BaseChart.prototype.getToolTipDiv = function () {
    return d3.select(this.jq().find('.hoverToolTip').get(0));
  };

  /**
   * Returns the SVG element defined for the chart.
   * @returns {undefined|*}
   */
  AnalyticD3BaseChart.prototype.svg = function () {
    return this.$svg;
  };

  /**
   * Instantiates the X Axis for the chart and returns the svg element of the x axis.
   */
  AnalyticD3BaseChart.prototype.initXAxis = function () {
    var that = this;
    var xAxis = that.getXAxis();
    if (xAxis === undefined) {
      return;
    }
    function xLabel(e) {
      return that.getXAxisLabel.apply(that, e);
    }
    return that.$svg.append("g").attr("class", "x axis").attr("transform", "translate(0," + that.availableHeight() + ")").call(xAxis).append("text").attr("class", "label").attr("x", that.getLabelPositionX()).attr("y", 40).attr("text-anchor", "middle").text(xLabel);
  };

  /**
   * Instantiates the Y axis and returns the svg element of the y axis.
   */
  AnalyticD3BaseChart.prototype.initYAxis = function () {
    var that = this;
    var yAxis = that.getYAxis();
    if (yAxis === undefined) {
      return;
    }
    function yLabel(e) {
      return that.getYAxisLabel.apply(that, e);
    }
    return that.$svg.append("g").attr("class", "y axis").call(that.getYAxis()).append("text").attr("class", "label").attr("y", -60).attr("x", -(that.availableHeight() / 2)).attr("dy", ".71em").attr("text-anchor", "middle").attr("transform", "rotate(-90)").text(yLabel);
  };

  /**
   * Returns the position for the X Axis label.
   * @returns {number}
   */
  AnalyticsBaseChart.prototype.getLabelPositionX = function () {
    return this.availableWidth() / 2;
  };

  /**
   * Open for the children to override.
   * Should return the X axis label.
   * @param text
   */
  AnalyticD3BaseChart.prototype.getXAxisLabel = function (text) {
    throw new Error("Not Implemented");
  };

  /**
   * Open for the children to override.
   * Should return the Y Axis label.
   * @param text
   */
  AnalyticD3BaseChart.prototype.getYAxisLabel = function (text) {
    throw new Error("Not Implemented");
  };

  /**
   * Will draw the chart, children can override this method to do extra processing before the draw of chart.
   * @param chartSettingsCollection
   */
  AnalyticD3BaseChart.prototype.reDraw = function (chartSettingsCollection) {
    this.draw(chartSettingsCollection);
  };

  /**
   * Will actually draw the chart.
   * @param chartSettingsCollection
   */
  AnalyticD3BaseChart.prototype.draw = function (chartSettingsCollection, isRedrawRequest) {
    var that = this;
    that.series = [];
    if (chartSettingsCollection.length > 0) {
      that.processData(chartSettingsCollection, isRedrawRequest);
    } else {
      return;
    }

    // Append X axis at the bottom
    that.initXAxis();
    // Append Y axis to left
    that.initYAxis();
    if (that.getSelectedChartType() === undefined) {
      that.setSelectedChartType(that.supportedCharts()[0]);
    }
    var chartType = new (that.getSelectedChartType())();
    that.$chart = chartType.draw(this, that.series);
    chartType.toolTip(this, that.$chart, that.series);
  };

  /**
   * Get the transformation coordinates for the tooltip.
   * @returns {number[]}
   */
  AnalyticD3BaseChart.prototype.getTransformation = function () {
    return this.svg().attr("transform").replace(/(translate\()(\d+.?\d+,\d+.?\d+)(\))$/, "$2").split(",").map(function (e) {
      return isNaN(parseFloat(e)) ? 0 : parseFloat(e);
    }) || [0, 0];
  };

  /**
   * Builds the Y Axis at the last stage, appends the format and other final things.
   * @param yScale
   * @returns {*|Number|{}}
   */
  AnalyticD3BaseChart.prototype.buildYAxis = function (yScale) {
    var yAxis = d3.svg.axis().scale(yScale).orient("left").tickFormat(function (d) {
      return analyticsUtil.standardizeInput(d);
    }).ticks(this.getTickCountForYAxis());
    return yAxis;
  };

  /**
   * Builds the X Axis at the last stage, appends the format and other final things.
   * @param xScale
   * @returns {*|Number|{}}
   */
  AnalyticD3BaseChart.prototype.buildXAxis = function (xScale) {
    var xAxis = d3.svg.axis().scale(xScale).orient("bottom").ticks(this.getTickCountForXAxis());
    return xAxis;
  };

  /**
   * Return the tick count for X Axis.
   * @returns {*}
   */
  AnalyticD3BaseChart.prototype.getTickCountForXAxis = function () {
    return this.resolutionScale(this.availableWidth());
  };

  /**
   * Returns the tick count for Y Axis.
   * @returns {*}
   */
  AnalyticD3BaseChart.prototype.getTickCountForYAxis = function () {
    return this.resolutionScale(this.availableHeight());
  };

  /**
   * Applies the Axis tick's text size.
   */
  AnalyticD3BaseChart.prototype.applyAxisTickTextSize = function () {
    this.svg().select(".x.axis").selectAll("text").style("font-size", this.getTabConfigDataModel().getFontSize() + "px");
    this.svg().select(".y.axis").selectAll("text").style("font-size", this.getTabConfigDataModel().getFontSize() + "px");
  };

  /**
   * Returns the legend's icon size.
   * @returns {*|number}
   */
  AnalyticD3BaseChart.prototype.getLegendIconSize = function () {
    return this.legendIconSize;
  };

  /**
   * Setter for legend icon size.
   * @param legendIconSize
   */
  AnalyticD3BaseChart.prototype.setLegendIconSize = function (legendIconSize) {
    this.legendIconSize = legendIconSize;
  };

  /**
   * Returns the general icon size.
   * @returns {*|number}
   */
  AnalyticD3BaseChart.prototype.getGeneralIconSize = function () {
    return this.generalIconSize;
  };

  /**
   * Returns the general icon size applicable for all charts.
   * @param generalIconSize
   */
  AnalyticD3BaseChart.prototype.setGeneralIconSize = function (generalIconSize) {
    this.generalIconSize = generalIconSize;
  };

  /**
   * Returns the set text scale for the base chart.
   * @returns {*}
   */
  AnalyticD3BaseChart.prototype.getTextScale = function () {
    return this.textScale;
  };

  /**
   * Returns the legend icon scale for the base chart.
   * @returns {*}
   */
  AnalyticD3BaseChart.prototype.getLegendIconScale = function () {
    return this.legendIconScale;
  };

  /**
   * Returns the general icon scale applied on the base chart.
   * @returns {*}
   */
  AnalyticD3BaseChart.prototype.getGeneralIconScale = function () {
    return this.generalIconScale;
  };

  /**
   * Returns true if the trimmed width and height is greater than 300.
   * @returns {boolean}
   */
  AnalyticD3BaseChart.prototype.showCommandBar = function () {
    if (this.availableHeight() > 300 && this.availableWidth() > 300) {
      return true;
    }
    return false;
  };

  /**
   * Returns the text length limit for the chart. Used by relative contribution chart.
   * @returns {number}
   */
  AnalyticD3BaseChart.prototype.getTextLengthLimit = function () {
    return this.$textLengthLimit;
  };

  /**
   * Base model for configuration.
   * Children should override this method and define their tab model. Then only their configuration will be seen.
   * @returns {*}
   */
  AnalyticD3BaseChart.prototype.getTabModelType = function () {
    return AnalyticsTabConfigBaseModel;
  };
  AnalyticD3BaseChart.prototype.getFormattedTooltipValue = function (value) {
    return analyticsUtil.standardizeInput(value);
  };
  AnalyticD3BaseChart.prototype.getBisectData = function (xValue) {
    var that = this;
    var bisectX = d3.bisector(function (d) {
      return d.x;
    }).left;
    var cIndexs = [];
    that.series.map(function (d, index) {
      var cIndex = bisectX(d.data, xValue, 0);
      if (d.data[cIndex] && xValue >= d.data[0].x - 3600010 && xValue <= d.data[d.data.length - 1].x) {
        cIndexs.push({
          i: index,
          cInd: cIndex,
          data: d.data[cIndex].x,
          d: Math.abs(d.data[cIndex].x - xValue)
        });
      }
    });

    // var minObj = {};
    return _.min(cIndexs, function (cObj) {
      return cObj.d;
    });
    // if (cIndexs.length !== 0) {
    //   minObj = cIndexs[0];
    //   cIndexs.map(function (d) {
    //     if (minObj.d > d.d) {
    //         minObj = d;
    //     }
    //   });
    //   return minObj;
    // }
  };
  AnalyticD3BaseChart.prototype.getBoundary = function () {
    var that = this;
    var minMax = [];
    that.series.map(function (d) {
      minMax.push.apply(minMax, d3.extent(d.data, function (e) {
        return e.x;
      }));
    });
    return d3.extent(minMax);
  };
  return AnalyticD3BaseChart;
});
