/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Danesh Kamal
 */

/**
 * API Status: **Private**
 * @module nmodule/history/rc/fe/RolloverValueEditor
 */
define(['baja!', 'bajaux/events', 'Promise', 'nmodule/webEditors/rc/fe/baja/ComplexCompositeEditor', 'css!nmodule/history/rc/history'], function (baja, events, Promise, ComplexCompositeEditor) {
  'use strict';

  var DESTROY_EVENT = events.DESTROY_EVENT,
    DISABLE_EVENT = events.DISABLE_EVENT,
    ENABLE_EVENT = events.ENABLE_EVENT,
    INITIALIZE_EVENT = events.INITIALIZE_EVENT,
    LOAD_EVENT = events.LOAD_EVENT,
    MODIFY_EVENT = events.MODIFY_EVENT,
    READONLY_EVENT = events.READONLY_EVENT,
    WRITABLE_EVENT = events.WRITABLE_EVENT;

  /**
   * Editor for working with `history:RolloverValue` values.
   * @class
   * @extends module:nmodule/webEditors/rc/fe/baja/ComplexCompositeEditor
   * @alias module:nmodule/history/rc/fe/RolloverValueEditor
   */
  var RolloverValueEditor = function RolloverValueEditor(params) {
    ComplexCompositeEditor.apply(this, arguments);
  };
  RolloverValueEditor.prototype = Object.create(ComplexCompositeEditor.prototype);
  RolloverValueEditor.prototype.constructor = RolloverValueEditor;

  /**
   * Initialize editors for the `unspecified` and `value` slots.
   *
   * @returns {Array}
   */
  RolloverValueEditor.prototype.getSlotFilter = function () {
    return [{
      slot: 'unspecified',
      properties: {
        trueText: 'null',
        falseText: 'null'
      }
    }, {
      slot: 'value',
      readonly: true
    }];
  };

  /**
   * Get the boolean editor for the 'null' checkbox.
   *
   * @private
   * @returns {module:nmodule/webEditors/rc/fe/baja/BooleanEditor}
   */
  RolloverValueEditor.prototype.$getNullEditor = function () {
    return this.getBuilder().getEditorFor('unspecified');
  };

  /**
   * Get the editor for the rollover numeric value.
   *
   * @private
   * @returns {module:nmodule/webEditors/rc/fe/baja/NumericEditor}
   */
  RolloverValueEditor.prototype.$getValueEditor = function () {
    return this.getBuilder().getEditorFor('value');
  };

  /**
   * Update the readonly status of the 'null' checkbox and the value editor as
   * appropriate.
   *
   * @param {Boolean} readonly
   * @returns {Promise}
   */
  RolloverValueEditor.prototype.$updateReadonly = function (readonly) {
    var nullEd = this.$getNullEditor(),
      valueEd = this.$getValueEditor();
    return nullEd && nullEd.read().then(function (unspecified) {
      return Promise.all([nullEd.setReadonly(readonly), valueEd.setReadonly(!(!unspecified && !readonly))]);
    });
  };

  /**
   * Arm event handlers to toggle the value editor's readonly status when
   * 'null' is checked and unchecked.
   *
   * @param {jQuery} dom
   */
  RolloverValueEditor.prototype.doInitialize = function (dom) {
    var that = this;
    dom.on([DESTROY_EVENT, DISABLE_EVENT, ENABLE_EVENT, INITIALIZE_EVENT, LOAD_EVENT, READONLY_EVENT, WRITABLE_EVENT].join(' '), '.editor', false);
    dom.on(MODIFY_EVENT, '.key-unspecified', function (e, ed) {
      that.$updateReadonly(that.isReadonly());
      that.setModified(true);
      return false;
    });
  };

  /**
   * Load in the `history:RolloverValue` and ensure the readonly status of the
   * value editor is correct.
   *
   * @param {baja.Struct} rolloverValue a `history:RolloverValue` instance
   * @returns {Promise}
   */
  RolloverValueEditor.prototype.doLoad = function (rolloverValue) {
    var that = this;
    return ComplexCompositeEditor.prototype.doLoad.apply(this, arguments).then(function () {
      return that.$updateReadonly(that.isReadonly());
    });
  };

  /**
   * Update the readonly status of the 'null' checkbox and the value editor as
   * appropriate. The value editor will always be readonly if 'null' is checked.
   *
   * @param {Boolean} readonly
   * @returns {Promise}
   */
  RolloverValueEditor.prototype.doReadonly = function (readonly) {
    return this.$updateReadonly(readonly);
  };
  return RolloverValueEditor;
});
