/*
 * Copyright 2005 Tridium, Inc. All Rights Reserved.
 */

 /* eslint-env browser */
 /*global hx: false, colorChooser: false */

/**
 * HistoryChartBuilder javascript support.
 *
 * @author    Andy Frank
 * @creation  15 Feb 06
 * @version   $Revision$ $Date$
 * @since     Baja 1.0
 */

var chartBuilder = new HistoryChartBuilder();
function HistoryChartBuilder() {
  'use strict';
////////////////////////////////////////////////////////////////
// Attributes
////////////////////////////////////////////////////////////////

  this.selected = [];
  this.temp = null; // used to hold row values temp.

  this.defaultColors = 
  [
    "#3b3ec5",  // blue
    "#b43030",  // red
    "#2aa22a",  // green
    "#9078b4",  // purple
    "#d3962d",  // orange
    "#4bacc6",  // cyan
    "#9dbb61",  // yellowish
    "#456287",  // dark bluish
    "#800000",  // dark red
    "#388194",  // dark cyan
    "#604c78",  // dark purple
    "#000080"   // dark blue
   ];

////////////////////////////////////////////////////////////////
// Methods
////////////////////////////////////////////////////////////////
  
  /**
   * Submit request to build chart.
   * 
   * @param {String} path - the HxOp path for the BHxHistoryChartBuilder object
   */
  this.build = function (path) {
    if (chartBuilder.selected.length === 0) {
      hx.showDialog("No histories selected.");
      
    } else {
      var formPath = "";
      if (path.length > 0) { formPath = path + "."; }
      // Push model into form
      var size = chartBuilder.selected.length;
      hx.setFormValue(formPath + "count", size);
            
      for (var i = 0; i < size; i++) {
        var row = chartBuilder.selected[i];
        hx.setFormValue(formPath + i + ".id", row.id);
        hx.setFormValue(formPath + i + ".displayName", row.displayName);
        hx.setFormValue(formPath + i + ".color", row.color);
        hx.setFormValue(formPath + i + ".type", row.type);
      }
      
      // TODO, correct for path, eventId
      hx.fireEvent(path, "event0");
    }         
  };
  
  /**
   * Add the history to the selection list. 
   */
  this.add = function (id, displayName) {
    chartBuilder.selected.push({ id: id, displayName: displayName, color: "#f00", type: "chart:LineChart" });
    chartBuilder.assignColor(
      chartBuilder.selected[chartBuilder.selected.length - 1]);
    chartBuilder.rebuildSelectList();
  };

  /**
   * Move this node up.
   */
  this.moveUp = function (index) {
    if (index === 0) { return; }
    
    var temp = chartBuilder.selected[index - 1];
    chartBuilder.selected[index - 1] = chartBuilder.selected[index];
    chartBuilder.selected[index] = temp;
    chartBuilder.rebuildSelectList();
  };

  /**
   * Move this node down.
   */
  this.moveDown = function (index) {
    if (index === chartBuilder.selected.length - 1) { return; }
    
    var temp = chartBuilder.selected[index + 1];
    chartBuilder.selected[index + 1] = chartBuilder.selected[index];
    chartBuilder.selected[index] = temp;
    chartBuilder.rebuildSelectList();
  };
  
  /**
   * Remove this node.
   */
  this.remove = function (index) {
    chartBuilder.selected.splice(index, 1);
    chartBuilder.rebuildSelectList();
  };
  
  /**
   * Clear selection list.
   */
  this.clear = function () {
    chartBuilder.selected.length = 0;
    chartBuilder.rebuildSelectList();
  };
  
  /**
   * Rebuild the SelectionList.
   */
  this.rebuildSelectList = function () {
    var table = document.getElementById("selectionList");
    var tbody = table.firstChild;
    
    // Nuke existing rows
    while (tbody.childNodes.length > 0) { tbody.removeChild(tbody.firstChild); }
      
    // Add rows
    for (var i = 0; i < chartBuilder.selected.length; i++) {
      var row = chartBuilder.selected[i];
      
      // Swatch
      var tr = document.createElement("tr");
      var td = document.createElement("td");
      td.appendChild(chartBuilder.makeSwatch(row));
      tr.appendChild(td);
  
      // HistoryID
      td = document.createElement("td");
      td.innerHTML = row.displayName;
      tr.appendChild(td);
      
      // Chart Type
      td = document.createElement("td");
      td.appendChild(chartBuilder.makeSelect(row));
      tr.appendChild(td);
      
      // Controls
      td = document.createElement("td");
      td.width = "100%";
      td.align = "right";      
      var mu = "chartBuilder.moveUp(" + i + ")";
      var md = "chartBuilder.moveDown(" + i + ")";
      var rm = "chartBuilder.remove(" + i + ")";
      td.appendChild(chartBuilder.img("moveUp.png",   mu));
      td.appendChild(chartBuilder.img("moveDown.png", md));
      td.appendChild(chartBuilder.img("delete.png",   rm));
      tr.appendChild(td);
      
      tbody.appendChild(tr);
    }
  };

  /**
   * Convience to build color swatch.
   */
  this.makeSwatch = function (row) {
    var div = document.createElement("div");
    div.style.width  = "16px";
    div.style.height = "16px";
    div.style.background = row.color;
    div.style.border = "1px solid black";
    div.style.cursor = "pointer";
    div.onclick = function () { 
      chartBuilder.temp = row;
      colorChooser.show(chartBuilder.chooseColor); 
    };
    return div;
  };
  
  /**
   * Convience to build chart type dropdown.
   */
  this.makeSelect = function (row) {
    var select = document.createElement("select");
    select.onchange = function () { row.type = this.value; };
    
    select.appendChild(this.option(row, "chart:LineChart", "Line Chart"));
    select.appendChild(this.option(row, "chart:AreaChart", " Area Chart"));
    select.appendChild(this.option(row, "chart:BarChart",  " Bar Chart"));
    select.appendChild(this.option(row, "chart:StackedBarChart", " Stacked Bar Chart"));
    select.appendChild(this.option(row, "chart:DiscreteLineChart", " Discrete Line Chart"));
    select.appendChild(this.option(row, "chart:DiscreteAreaChart", " Discrete Area Chart"));
    select.appendChild(this.option(row, "chart:PieChart", " Pie Chart"));
    
    return select;    
  }; 
  
  /**
   * Convience to add option tag.
   */
  this.option = function (row, type, text) {
    var option = document.createElement("option");
    option.value = type;
    if (row.type === option.value) { option.selected = "selected"; }
    option.innerHTML = text;
    return option;  
  };
   
  /**
   * Convience to build an img element.
   */
  this.img = function (src, event) {
    var span = document.createElement("span");
    span.innerHTML = "<img " +
      "src='/ord?module://history/com/tridium/history/ui/icons/" + src +
      "' style='cursor:pointer;' onclick='" + event + "' />";
    return span;
  };
  
  /**
   * Assign next color to this item.
   */
  this.assignColor = function (item) {
    var color = item.color;
    var def = chartBuilder.defaultColors;
    var sel = chartBuilder.selected;
    
    for (var i = 0; i < def.length; i++) {
      color = def[i];
      var dup = false;
      
      // Walk through item list to see if brush is being used
      for (var j = 0; j < sel.length; j++) {
        var testItem = sel[j];
        if (item === testItem) { continue; }
                
        if (color === testItem.color) {
          dup = true;
          break;
        }
      }
      
      if (!dup) { break; }
    }
    
    item.color = color;
  };  
  
////////////////////////////////////////////////////////////////
// Events
////////////////////////////////////////////////////////////////
  
  /**
   * Handler for ColorChooser.
   */
  this.chooseColor = function (color) {
    chartBuilder.temp.color = color;
    chartBuilder.rebuildSelectList();
  };

////////////////////////////////////////////////////////////////
// Tree Expansion
////////////////////////////////////////////////////////////////

  function findClasses(elem) {
    return (elem.className || "").split(" "); 
  }

  function hasClass(elem, cname) {
    var classes = findClasses(elem),
        i;

    for (i = 0; i < classes.length; ++i) {
      if (classes[i] === cname) {
        return true;
      }
    }
    return false;    
  }

  function removeClass(elem, cname) {
    var classes = findClasses(elem),
        i;

    for (i = 0; i < classes.length; ++i) {
      if (classes[i] === cname) {
        classes.splice(i, 1);
        break;
      }
    }

    elem.className = classes.join(" ");  
  }

  function addClass(elem, cname) {
    var classes;
    
    if (!hasClass(elem, cname)) {
      classes = findClasses(elem);
      classes.push(cname);
      elem.className = classes.join(" "); 
    } 
  }

  /**
   * Invoked whenever a tree expander is clicked upon.
   *
   * @param target The DOM element that was originally clicked on
   *               to expand or collapse the tree.
   */
  this.onTreeExpanderClick = function (target) {
    var parent = target.parentElement,
        treeNodeHidden = "treeNodeHidden";

    if (parent) {
      if (!hasClass(parent, treeNodeHidden)) {
        addClass(parent, treeNodeHidden);
      } else {
        removeClass(parent, treeNodeHidden);
      }
    }
  };
}  
