/*
 * Copyright 2015 Tridium, Inc. All Rights Reserved.
 */
package javax.baja.security;

import java.security.AccessController;
import java.security.PrivilegedAction;
import java.util.Arrays;
import java.util.concurrent.atomic.AtomicReference;

import javax.baja.authn.BAuthenticationScheme;
import javax.baja.authn.BPasswordAuthenticationScheme;
import javax.baja.nre.annotations.Facet;
import javax.baja.nre.annotations.Generated;
import javax.baja.nre.annotations.NiagaraAction;
import javax.baja.nre.annotations.NiagaraProperty;
import javax.baja.nre.annotations.NiagaraType;
import javax.baja.space.BComponentSpace;
import javax.baja.sys.Action;
import javax.baja.sys.BAbsTime;
import javax.baja.sys.BBoolean;
import javax.baja.sys.BComplex;
import javax.baja.sys.BComponent;
import javax.baja.sys.BFacets;
import javax.baja.sys.BIUnlinkable;
import javax.baja.sys.BInteger;
import javax.baja.sys.BRelTime;
import javax.baja.sys.BValue;
import javax.baja.sys.BasicContext;
import javax.baja.sys.Context;
import javax.baja.sys.Flags;
import javax.baja.sys.IPropertyValidator;
import javax.baja.sys.Localizable;
import javax.baja.sys.LocalizableRuntimeException;
import javax.baja.sys.Property;
import javax.baja.sys.Sys;
import javax.baja.sys.Type;
import javax.baja.sys.Validatable;
import javax.baja.user.BPasswordStrength;
import javax.baja.user.BUser;
import javax.baja.user.BUserService;
import javax.baja.util.CannotValidateException;
import javax.baja.util.Queue;

import com.tridium.nre.security.SecretChars;
import com.tridium.sys.schema.Fw;
import com.tridium.user.BUserPasswordConfiguration;

/**
 * @author Tom Duffy
 * @creation 8/15/2014
 * @since Niagara 4.0
 */
@NiagaraType
@NiagaraProperty(
  name = "password",
  type = "BPassword",
  defaultValue = "BPassword.DEFAULT",
  flags = Flags.OPERATOR,
  facets = {
    @Facet(name = "BFacets.FIELD_EDITOR", value = "\"wbutil:UserPasswordFE\""),
    @Facet(name = "BFacets.UX_FIELD_EDITOR", value = "\"webEditors:UserPasswordEditor\"")
  },
  override = true
)
@NiagaraProperty(
  name = "passwordConfig",
  type = "BUserPasswordConfiguration",
  defaultValue = "new BUserPasswordConfiguration()"
)
/*
 Update the password of this user.
 @since Niagara 4.14
 */
@NiagaraAction(
  name = "updatePassword",
  parameterType = "BUserPasswordChangeParams",
  defaultValue = "new BUserPasswordChangeParams()",
  flags = Flags.OPERATOR,
  facets = @Facet("BFacets.make(BFacets.SECURITY, BBoolean.TRUE)")
)
public class BPasswordAuthenticator
  extends BPasswordCache
  implements IPropertyValidator, BIUnlinkable
{
//region /*+ ------------ BEGIN BAJA AUTO GENERATED CODE ------------ +*/
//@formatter:off
/*@ $javax.baja.security.BPasswordAuthenticator(3498233400)1.0$ @*/
/* Generated Wed May 10 02:25:31 EDT 2023 by Slot-o-Matic (c) Tridium, Inc. 2012-2023 */

  //region Property "password"

  /**
   * Slot for the {@code password} property.
   * @see #getPassword
   * @see #setPassword
   */
  @Generated
  public static final Property password = newProperty(Flags.OPERATOR, BPassword.DEFAULT, BFacets.make(BFacets.make(BFacets.FIELD_EDITOR, "wbutil:UserPasswordFE"), BFacets.make(BFacets.UX_FIELD_EDITOR, "webEditors:UserPasswordEditor")));

  //endregion Property "password"

  //region Property "passwordConfig"

  /**
   * Slot for the {@code passwordConfig} property.
   * @see #getPasswordConfig
   * @see #setPasswordConfig
   */
  @Generated
  public static final Property passwordConfig = newProperty(0, new BUserPasswordConfiguration(), null);

  /**
   * Get the {@code passwordConfig} property.
   * @see #passwordConfig
   */
  @Generated
  public BUserPasswordConfiguration getPasswordConfig() { return (BUserPasswordConfiguration)get(passwordConfig); }

  /**
   * Set the {@code passwordConfig} property.
   * @see #passwordConfig
   */
  @Generated
  public void setPasswordConfig(BUserPasswordConfiguration v) { set(passwordConfig, v, null); }

  //endregion Property "passwordConfig"

  //region Action "updatePassword"

  /**
   * Slot for the {@code updatePassword} action.
   * Update the password of this user.
   * @since Niagara 4.14
   * @see #updatePassword(BUserPasswordChangeParams parameter)
   */
  @Generated
  public static final Action updatePassword = newAction(Flags.OPERATOR, new BUserPasswordChangeParams(), BFacets.make(BFacets.SECURITY, BBoolean.TRUE));

  /**
   * Invoke the {@code updatePassword} action.
   * Update the password of this user.
   * @since Niagara 4.14
   * @see #updatePassword
   */
  @Generated
  public void updatePassword(BUserPasswordChangeParams parameter) { invoke(updatePassword, parameter, null); }

  //endregion Action "updatePassword"

  //region Type

  @Override
  @Generated
  public Type getType() { return TYPE; }
  @Generated
  public static final Type TYPE = Sys.loadType(BPasswordAuthenticator.class);

  //endregion Type

//@formatter:on
//endregion /*+ ------------ END BAJA AUTO GENERATED CODE -------------- +*/

  public BPasswordAuthenticator()
  {
  }

  public BPasswordAuthenticator(BPassword value)
  {
    setPassword(value);
  }

  ////////////////////////////////////////////////////////////////
// Component
////////////////////////////////////////////////////////////////

  @Override
  public final Object fw(int x, Object a, Object b, Object c, Object d)
  {
    switch(x)
    {
      case Fw.CHANGED:
        BComponentSpace space = getComponentSpace();
        if (space == null || space.fireDirectCallbacks())
        {
          fwChanged((Property)a, (Context)b);

        }
        break;
    }
    return super.fw(x, a, b, c, d);
  }

  @Override
  public void started()
  {
    if (isInUserService())
    {
      convertToPbkdf2Password();
    }
  }

  @Override
  public void changed(Property property, Context context)
  {
    if (!isRunning())
    {
      return;
    }
    if (property == passwordConfig)
    {
      BComplex parent = getParent();
      if (parent instanceof BComponent)
      {
        parent.asComponent().changed(getPropertyInParent(), context);
      }

    }
    if(property.getType() == BPassword.TYPE){
      invalidUserSessions((BUser) getParent());
    }
  }

  private void fwChanged(Property prop, Context cx)
  {
    if (isInUserService())
    {
      if (password.equals(prop) && cx != pwConverted)
      {
        // if using pbkdf2, convert the users password
        convertToPbkdf2Password();
      }
    }
  }

  /**
   * Check if the specified password meets password requirements. Checks will
   * only be performed on reversible passwords, calls with an irreversible password
   * will skip chacks and return successfully.
   *
   * @param newPassword the new password to check, must be reversible
   * @param context the context
   * @throws LocalizableRuntimeException if validation fails
   * @throws CannotValidateException if validation fails
   * @since Niagara 4.10
   */
  public void checkPassword(BPassword newPassword, Context context)
  {
    BComplex parent = getParent();
    if (parent instanceof BUser)
    {
      BUser user = (BUser)parent;
      // lease to depth of user password config
      user.lease(1);
      BAuthenticationScheme scheme = user.getAuthenticationScheme();
      if(scheme instanceof BPasswordAuthenticationScheme)
      {
        // lease to depth of global password config
        scheme.lease(1);
        checkPassword(user, (BPasswordAuthenticationScheme) scheme, getPasswordConfig(), newPassword, context);
      }
    }
  }

  /**
   * Check if the specified password meets password requirements. Checks will
   * only be performed on reversible passwords, calls with an irreversible password
   * will skip chacks and return successfully.
   *
   * @param user the user the password is for, or null if a new user
   * @param scheme the authentication scheme for the user, or null if not available
   * @param config the users UserPasswordConfiguration, or null if not available
   * @param newPassword the new password to check, must be reversible
   * @param context the context
   * @throws LocalizableRuntimeException if validation fails
   * @throws CannotValidateException if validation fails
   * @since Niagara 4.10
   */
  public static void checkPassword(BUser user,
                                   BPasswordAuthenticationScheme scheme,
                                   BUserPasswordConfiguration config,
                                   BPassword newPassword,
                                   Context context)
  {
    if (!newPassword.getPasswordEncoder().isReversible())
    {
      return;
    }

    try
    {
      BPasswordStrength strength = scheme.getGlobalPasswordConfiguration().getPasswordStrength();
      try (SecretChars passChars = AccessController.doPrivileged((PrivilegedAction<SecretChars>) newPassword::getSecretChars))
      {
        AtomicReference<Localizable> messageRef = new AtomicReference<>();
        if (!strength.isPasswordValid(passChars.get(), messageRef::set))
        {
          throw new LocalizableRuntimeException("baja", messageRef.get().toString(context));
        }
      }
      if (user != null && scheme.isDuplicatePassword(newPassword, user))
      {
        throw new LocalizableRuntimeException("baja", "user.strongPassword.alreadyUsed");
      }
      if (config != null)
      {
        config.changeIntervalCheck(scheme.getGlobalPasswordConfiguration());
      }
    }
    catch (LocalizableRuntimeException e)
    {
      throw e;
    }
    catch (Exception e)
    {
      throw new CannotValidateException(e);
    }
  }

  private boolean isInUserService()
  {
    if(getParent() != null)
    {
      BValue parent = getParent().getParent();
      return parent instanceof BUserService;
    }
    else
    {
      return false;
    }
  }

  // expired password should no longer keep a user from logging in, but they will be forced
  // to change their password. Return true, expiration will be checked in Tuner/NiagaraAuthenticator
  @Override
  public boolean canLogin()
  {
    return true;
  }

  @Override
  public boolean requiresCredentialsReset()
  {
    return getPasswordConfig().getForceResetAtNextLogin();
  }

  public boolean convertToPbkdf2Password()
  {
    BPassword oldPw = getPassword();
    if (oldPw.getPasswordEncoder().isReversible())
    {
      getPasswordConfig().passwordModified();
      BPassword newPw = BPassword.make(AccessController.doPrivileged((PrivilegedAction<String>)oldPw::getValue), BPbkdf2HmacSha256PasswordEncoder.ENCODING_TYPE);
      set("password", newPw, pwConverted);
      return true;
    }
    return false;
  }

  /**
   * Update the Password for this user.
   * @since Niagara 4.14
   * 
   * @throws LocalizableRuntimeException
   */
  public final void doUpdatePassword(BUserPasswordChangeParams passwordChange, Context cx)
  {
    if (isLockOut)
    {
      if (lockOutTime.compareTo(BAbsTime.now()) > 0)
      {
        throw new LocalizableRuntimeException("baja", "user.password.change.maxBadAttemptLockOut", new String[]{ BInteger.make(MAX_BAD_ATTEMPT_CURRENT_PASSWORD).toString(cx), LOCKOUT_PERIOD_UPDATE_PASSWORD_FAIL.toString(cx) });
      }
      else
      {
        isLockOut = false;
        lockOutTime = BAbsTime.make();
      }
    }

    BPassword userPassword = getPassword();

    if (!userPassword.validate(passwordChange.getCurrentPassword()))
    {
      updatePasswordFailed();
      throw new LocalizableRuntimeException("baja", "user.password.change.currentPasswordNotMatching");
    }

    UpdatePasswordActionContext upCx = new UpdatePasswordActionContext(cx);
    set(BPasswordCache.password, passwordChange.getNewPassword(), upCx);
    resetLockoutProperties();   
  }

  private void resetLockoutProperties()
  {
    isLockOut = false;
    lockOutTime= BAbsTime.make();
    if (updatePasswordFailTimes != null)
    {
      updatePasswordFailTimes.dequeue();
    }
  }

  /**
   * This method should be called whenever an update
   * password fails on this user for bad current
   * password attempts.
   * @since Niagara 4.14
   */
  private void updatePasswordFailed()
  {
    updatePasswordFailTimes = getUpdatePasswordFailTimes();
    // first, enqueue current failure time.
    BAbsTime now = BAbsTime.now();
    updatePasswordFailTimes.enqueue(now);

    // now remove all failure times that are outside the lock-out window.
    BAbsTime startOfWindow = now.subtract(LOCKOUT_WINDOW_UPDATE_PASSWORD_FAIL);
    while (((BAbsTime) updatePasswordFailTimes.peek()).isBefore(startOfWindow))
    {
      updatePasswordFailTimes.dequeue();
    }

    // The fail queue now contains the number of failures within the
    // lock out window. If that number is greater than or equal to the 
    // maximum allowed attempts, then lock the user action.
    if (updatePasswordFailTimes.size() >= MAX_BAD_ATTEMPT_CURRENT_PASSWORD)
    {
      isLockOut = true;
      lockOutTime = now.add(LOCKOUT_PERIOD_UPDATE_PASSWORD_FAIL);
    }
  }

  private Queue getUpdatePasswordFailTimes()
  {
    if (updatePasswordFailTimes == null)
    {
      updatePasswordFailTimes = new Queue();
    }
    return updatePasswordFailTimes;
  }

  public static final Context pwConverted = new BasicContext()
  {
    @Override
    public boolean equals(Object obj) { return this == obj; }
    @Override
    public int hashCode() { return toString().hashCode(); }
    @Override
    public String toString() { return "Context.pwConverted"; }
  };

  //region IPropertyValidator

  @Override
  public IPropertyValidator getPropertyValidator(Property[] properties, Context context)
  {
    return this;
  }

  @Override
  public IPropertyValidator getPropertyValidator(Property property, Context context)
  {
    return this;
  }

  @Override
  public void validateSet(Validatable validatable, Context context)
  {
    if (Arrays.asList(validatable.getModifiedProperties()).contains(password))
    {
      doValidateSet(context, (BPassword)validatable.getProposedValue(password));
    }
  }

  @Override
  public void validateSet(BComplex instance, Property property, BValue newValue, Context context)
  {
    if (property.equals(password))
    {
      doValidateSet(context, (BPassword)newValue);
    }
  }

  private void doValidateSet(Context context, BPassword password)
  {
    if (!(context instanceof UpdatePasswordActionContext) && !getPermissions(context).hasAdminWrite())
    {
      throw new LocalizableRuntimeException("baja", "user.password.change.permission");
    }
    else if (context != pwConverted)
    {
      checkPassword(password, context);
    }
  }
  //endregion IPropertyValidator

  private static class UpdatePasswordActionContext
    extends BasicContext
  {
    private UpdatePasswordActionContext(Context cx)
    {
      super(cx);
    }

    @Override
    public String toString()
    {
      return "UpdatePasswordActionContext";
    }
  }

  private Queue updatePasswordFailTimes;
  private static final BRelTime LOCKOUT_WINDOW_UPDATE_PASSWORD_FAIL = BRelTime.makeSeconds(AccessController.doPrivileged((PrivilegedAction<Integer>)
    () -> Integer.getInteger("niagara.updatePassword.lockoutWindowUpdatePasswordFail", 60)));
  private static final BRelTime LOCKOUT_PERIOD_UPDATE_PASSWORD_FAIL = BRelTime.makeSeconds(AccessController.doPrivileged((PrivilegedAction<Integer>)
    () -> Integer.getInteger("niagara.updatePassword.lockoutPeriodUpdatePasswordFail", 300)));
  private static final int MAX_BAD_ATTEMPT_CURRENT_PASSWORD = AccessController.doPrivileged((PrivilegedAction<Integer>)
    () -> Integer.getInteger("niagara.updatePassword.maxBadAttemptCurrentPassword", 5));

  private boolean isLockOut;
  private BAbsTime lockOutTime = BAbsTime.make();

  public static final BPasswordAuthenticator DEFAULT = new BPasswordAuthenticator();
}
