/*
 * Copyright 2002 Tridium, Inc. All Rights Reserved.
 */
package javax.baja.security;

import javax.baja.sys.BAbsTime;
import javax.baja.sys.Clock;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Date;

/**
 * AuditEvent is the event sent to the system Auditor
 * when users make component modifications and invoke
 * actions.
 *
 * @author    Brian Frank
 * @creation  4 Apr 02
 * @version   $Revision: 7$ $Date: 8/2/07 10:20:58 AM EDT$
 * @since     Baja 1.0 
 */
public class AuditEvent
{

////////////////////////////////////////////////////////////////
// Operation Constants
////////////////////////////////////////////////////////////////

  public static final String CHANGED        = "Changed";
  public static final String ADDED          = "Added";
  public static final String REMOVED        = "Removed";
  public static final String RENAMED        = "Renamed";
  public static final String REORDERED      = "Reordered";
  public static final String FLAGS_CHANGED  = "Flags Changed";
  public static final String FACETS_CHANGED = "Facets Changed";
  public static final String RECATEGORIZED  = "Recategorized";
  public static final String INVOKED        = "Invoked";
  /**
   * Login
   *
   * @since Niagara 3.3
   */
  public static final String LOGIN          = "Login";

  /**
   * Logout
   *
   * @since Niagara 3.3
   */
  public static final String LOGOUT         = "Logout";

  /**
   * Login Failure
   *
   * @since Niagara 3.3
   */
  public static final String LOGIN_FAILURE  = "Login Failure";

  /**
   * Logout due to Inactivity
   *
   * @since Niagara 4.0
   */
  public static final String TIMEOUT        = "Logout (Timeout)";

////////////////////////////////////////////////////////////////
// Constructor
////////////////////////////////////////////////////////////////

  public AuditEvent(String operation, String target, String slotName, String oldValue, 
                    String value, String userName)
  {
    this.operation = operation;
    this.target = target;
    this.slotName = slotName;
    this.oldValue = oldValue;
    this.value = value;
    this.userName = userName;
    this.timestamp = Clock.time();
  }                    
                    
////////////////////////////////////////////////////////////////
// Access
////////////////////////////////////////////////////////////////

  /**
   * Get event operation.
   */
  public String getOperation() { return operation; }                 

  /**
   * Get a string describing source object.
   */
  public String getTarget() { return target; }
  
  /**
   * Get slot name of event or null if not applicable.
   */
  public String getSlotName() { return slotName; }
  
  /**
   * Get old value string or null if not applicable.
   */
  public String getOldValue() { return oldValue; }

  /**
   * Get value string or null if not applicable.
   */
  public String getValue() { return value; }
  
  /**
   * Get user name of user who made the modification.
   */
  public String getUserName() { return userName; }
  
  /**
   * Get the timestamp of the event.
   */
  public BAbsTime getTimestamp() { return timestamp; }

////////////////////////////////////////////////////////////////
// Formatting
////////////////////////////////////////////////////////////////  

  public String toString()
  {
    StringBuilder s = new StringBuilder();
    s.append('[').append(format.format(new Date(timestamp.getMillis())))
      .append("][").append(userName)
      .append("][").append(target).append("] ")
      .append(toSummaryString());
    return s.toString();
  }
  
  /**
   * Get a summary string which doesn't include the 
   * user or timestamp.
   */
  public String toSummaryString()
  {
    StringBuilder s = new StringBuilder();
    if (operation.equals(CHANGED))
    {
      s.append(slotName).append(": ").append(oldValue).append(" -> ").append(value);
    }
    else if(operation.equals(ADDED))
    {
      s.append(slotName).append(": added");
    }      
    else if (operation.equals(REMOVED))
    {
      s.append(slotName).append(": removed");
    }
    else if (operation.equals(RENAMED))
    {
      s.append(slotName).append(": renamed " + value);
    }
    else if (operation.equals(REORDERED))
    {
      s.append("reordered");
    }
    else if (operation.equals(FLAGS_CHANGED))
    {
      s.append(slotName).append(": flags ").append(oldValue).append(" -> ").append(value);
    }
    else if (operation.equals(RECATEGORIZED))
    {
      s.append("recategorized ").append(oldValue).append(" -> ").append(value);
    }
    else if (operation.equals(INVOKED))
    {
      s.append(slotName).append('(');
      if (value != null) s.append(value);
      s.append(')');
    }
    else if (operation.equals(LOGIN))
    {
      s.append("login");//.append(userName).append(" -> ").append(target);
    }
    else if (operation.equals(LOGOUT))
    {
      s.append("logout");//.append(userName).append(" -> ").append(target);
    }
    else if (operation.equals(LOGIN_FAILURE))
    {
      s.append("login failure ").append(value);//.append(" for ").append(userName).append(" -> ").append(target);
    }
    else if (operation.equals(TIMEOUT))
    {
      s.append("browser timeout");
    }
    else
    {
      s.append(operation); 
      if (slotName != null) s.append(";slotName=" + slotName);
      if (oldValue != null) s.append(";oldValue=" + oldValue);
      if (value != null)    s.append(";value=" + value);
    }
    return s.toString();
  }

////////////////////////////////////////////////////////////////
// Attributes
////////////////////////////////////////////////////////////////  

  private DateFormat format = new SimpleDateFormat("HH:mm:ss dd-MMM-yy");

  private String operation;
  private String target;
  private String slotName;
  private String oldValue;
  private String value;
  private String userName;
  private BAbsTime timestamp;
       
}
