/*
 * Copyright 2017, Tridium, Inc. All Rights Reserved.
 */
package javax.baja.bacnet.export;

import static javax.baja.bacnet.enums.BBacnetErrorClass.property;
import static javax.baja.bacnet.enums.BBacnetErrorClass.services;
import static javax.baja.bacnet.enums.BBacnetErrorCode.other;
import static javax.baja.bacnet.enums.BBacnetErrorCode.propertyIsNotA_List;
import static javax.baja.bacnet.enums.BBacnetErrorCode.unknownProperty;
import static javax.baja.bacnet.enums.BBacnetErrorCode.valueOutOfRange;
import static javax.baja.bacnet.enums.BBacnetPropertyIdentifier.logBuffer;
import static javax.baja.bacnet.enums.BBacnetPropertyIdentifier.presentValue;
import static javax.baja.bacnet.export.BacnetDescriptorUtil.findOrAddLocalPoint;
import static javax.baja.bacnet.export.BacnetDescriptorUtil.findOrAddRemotePoint;
import static javax.baja.bacnet.export.BacnetDescriptorUtil.isLocalDevice;
import static javax.baja.bacnet.export.BacnetDescriptorUtil.makeAddListElementError;
import static javax.baja.bacnet.export.BacnetDescriptorUtil.makeRemoveListElementError;

import java.util.ArrayList;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.baja.alarm.BAlarmClass;
import javax.baja.alarm.BAlarmService;
import javax.baja.alarm.BAlarmTransitionBits;
import javax.baja.alarm.BIAlarmSource;
import javax.baja.alarm.ext.BAlarmSourceExt;
import javax.baja.alarm.ext.BAlarmTimestamps;
import javax.baja.alarm.ext.BFaultAlgorithm;
import javax.baja.alarm.ext.BLimitEnable;
import javax.baja.alarm.ext.BNotifyType;
import javax.baja.alarm.ext.BOffnormalAlgorithm;
import javax.baja.alarm.ext.fault.BEnumFaultAlgorithm;
import javax.baja.alarm.ext.fault.BOutOfRangeFaultAlgorithm;
import javax.baja.alarm.ext.offnormal.BBooleanChangeOfStateAlgorithm;
import javax.baja.alarm.ext.offnormal.BBooleanCommandFailureAlgorithm;
import javax.baja.alarm.ext.offnormal.BEnumChangeOfStateAlgorithm;
import javax.baja.alarm.ext.offnormal.BEnumCommandFailureAlgorithm;
import javax.baja.alarm.ext.offnormal.BFloatingLimitAlgorithm;
import javax.baja.alarm.ext.offnormal.BNumericChangeOfStateAlgorithm;
import javax.baja.alarm.ext.offnormal.BOutOfRangeAlgorithm;
import javax.baja.alarm.ext.offnormal.BStringChangeOfStateAlgorithm;
import javax.baja.alarm.ext.offnormal.BStringChangeOfStateFaultAlgorithm;
import javax.baja.bacnet.BBacnetNetwork;
import javax.baja.bacnet.BacnetConst;
import javax.baja.bacnet.BacnetException;
import javax.baja.bacnet.alarm.BBacnetStatusAlgorithm;
import javax.baja.bacnet.datatypes.BBacnetArray;
import javax.baja.bacnet.datatypes.BBacnetBitString;
import javax.baja.bacnet.datatypes.BBacnetDateTime;
import javax.baja.bacnet.datatypes.BBacnetDeviceObjectPropertyReference;
import javax.baja.bacnet.datatypes.BBacnetEventParameter;
import javax.baja.bacnet.datatypes.BBacnetListOf;
import javax.baja.bacnet.datatypes.BBacnetObjectIdentifier;
import javax.baja.bacnet.datatypes.BBacnetObjectPropertyReference;
import javax.baja.bacnet.datatypes.BBacnetPropertyStates;
import javax.baja.bacnet.datatypes.BBacnetTimeStamp;
import javax.baja.bacnet.datatypes.BBacnetUnsigned;
import javax.baja.bacnet.enums.BBacnetBinaryPv;
import javax.baja.bacnet.enums.BBacnetErrorClass;
import javax.baja.bacnet.enums.BBacnetErrorCode;
import javax.baja.bacnet.enums.BBacnetEventState;
import javax.baja.bacnet.enums.BBacnetEventType;
import javax.baja.bacnet.enums.BBacnetNotifyType;
import javax.baja.bacnet.enums.BBacnetObjectType;
import javax.baja.bacnet.enums.BBacnetPropertyIdentifier;
import javax.baja.bacnet.enums.BBacnetReliability;
import javax.baja.bacnet.io.AsnException;
import javax.baja.bacnet.io.ChangeListError;
import javax.baja.bacnet.io.ErrorException;
import javax.baja.bacnet.io.ErrorType;
import javax.baja.bacnet.io.OutOfRangeException;
import javax.baja.bacnet.io.PropertyReference;
import javax.baja.bacnet.io.PropertyValue;
import javax.baja.bacnet.io.RangeData;
import javax.baja.bacnet.io.RangeReference;
import javax.baja.bacnet.io.RejectException;
import javax.baja.bacnet.point.BBacnetProxyExt;
import javax.baja.bacnet.util.BacnetBitStringUtil;
import javax.baja.bacnet.util.PropertyInfo;
import javax.baja.control.BBooleanPoint;
import javax.baja.control.BControlPoint;
import javax.baja.control.BEnumPoint;
import javax.baja.control.BNumericPoint;
import javax.baja.control.BPointExtension;
import javax.baja.control.BStringPoint;
import javax.baja.control.ext.BAbstractProxyExt;
import javax.baja.naming.BOrd;
import javax.baja.naming.SlotPath;
import javax.baja.nre.annotations.Facet;
import javax.baja.nre.annotations.Generated;
import javax.baja.nre.annotations.NiagaraProperty;
import javax.baja.nre.annotations.NiagaraType;
import javax.baja.security.PermissionException;
import javax.baja.spy.SpyWriter;
import javax.baja.status.BIStatus;
import javax.baja.status.BStatus;
import javax.baja.status.BStatusBoolean;
import javax.baja.sys.BAbsTime;
import javax.baja.sys.BBoolean;
import javax.baja.sys.BComplex;
import javax.baja.sys.BComponent;
import javax.baja.sys.BEnum;
import javax.baja.sys.BEnumRange;
import javax.baja.sys.BFacets;
import javax.baja.sys.BInteger;
import javax.baja.sys.BLink;
import javax.baja.sys.BNumber;
import javax.baja.sys.BObject;
import javax.baja.sys.BRelTime;
import javax.baja.sys.BString;
import javax.baja.sys.BValue;
import javax.baja.sys.Context;
import javax.baja.sys.DuplicateSlotException;
import javax.baja.sys.Flags;
import javax.baja.sys.Property;
import javax.baja.sys.ServiceNotFoundException;
import javax.baja.sys.Sys;
import javax.baja.sys.Type;
import javax.baja.util.BFormat;
import javax.baja.util.BTypeSpec;

import com.tridium.bacnet.BacUtil;
import com.tridium.bacnet.ObjectTypeList;
import com.tridium.bacnet.alarm.BBacnetChangeOfDiscreteValueAlgorithm;
import com.tridium.bacnet.asn.AsnUtil;
import com.tridium.bacnet.asn.NErrorType;
import com.tridium.bacnet.asn.NReadPropertyResult;
import com.tridium.bacnet.history.BBacnetTrendLogAlarmSourceExt;
import com.tridium.bacnet.history.BBacnetTrendLogRemoteExt;
import com.tridium.bacnet.history.BIBacnetTrendLogExt;
import com.tridium.bacnet.services.confirmed.ReadRangeAck;

/**
 * BBacnetEventEnrollmentDescriptor exposes a Niagara event to Bacnet.
 *
 * @author Sandipan Aich on 5/4/2017
 * @since Niagara 3 Bacnet 1.0
 */
@NiagaraType
@NiagaraProperty(
  name = "eventEnrollmentOrd",
  type = "BOrd",
  defaultValue = "BOrd.DEFAULT",
  flags = Flags.DEFAULT_ON_CLONE
)
@NiagaraProperty(
  name = "objectId",
  type = "BBacnetObjectIdentifier",
  defaultValue = "BBacnetObjectIdentifier.make(BBacnetObjectType.EVENT_ENROLLMENT)",
  flags = Flags.DEFAULT_ON_CLONE
)
@NiagaraProperty(
  name = "objectName",
  type = "String",
  defaultValue = "",
  flags = Flags.DEFAULT_ON_CLONE
)
@NiagaraProperty(
  name = "description",
  type = "String",
  defaultValue = "",
  flags = Flags.DEFAULT_ON_CLONE
)
@NiagaraProperty(
  name = "typeOfEvent",
  type = "BBacnetEventType",
  defaultValue = "BBacnetEventType.none",
  flags = Flags.DEFAULT_ON_CLONE | Flags.READONLY
)
@NiagaraProperty(
  name = "notifyTypeId",
  type = "BNotifyType",
  defaultValue = "BNotifyType.alarm",
  flags = Flags.DEFAULT_ON_CLONE | Flags.HIDDEN | Flags.READONLY,
  deprecated = true
)
@NiagaraProperty(
  name = "objectPropertyReference",
  type = "BBacnetDeviceObjectPropertyReference",
  defaultValue = "new BBacnetDeviceObjectPropertyReference()",
  flags = Flags.DEFAULT_ON_CLONE | Flags.READONLY | Flags.HIDDEN
)
@NiagaraProperty(
  name = "notificationClassId",
  type = "int",
  defaultValue = "0",
  flags = Flags.DEFAULT_ON_CLONE
)
/*
 Reliability of the Event Enrollment object to perform its monitoring function as described in
 135-2012 12.12.21. Does not reflect the reliability of the monitored object or the result of the
 fault algorithm, if one is in use. Those other items are reflected in the BACnet Reliability
 property as read through a BACnet request.
 */
@NiagaraProperty(
  name = "reliability",
  type = "BBacnetReliability",
  defaultValue = "BBacnetReliability.configurationError",
  flags = Flags.DEFAULT_ON_CLONE | Flags.READONLY
)
@NiagaraProperty(
  name = "eventParameter",
  type = "BBacnetEventParameter",
  defaultValue = "new BBacnetEventParameter()",
  flags = Flags.READONLY | Flags.HIDDEN
)
/*
 @since Niagara 4.14u2
 @since Niagara 4.15u1
 */
@NiagaraProperty(
  name = "notifyType",
  type = "BBacnetNotifyType",
  defaultValue = "BBacnetNotifyType.alarm",
  facets = @Facet("BacUtil.makeBacnetNotifyTypeFacets()")
)
public class BBacnetEventEnrollmentDescriptor
  extends BBacnetEventSource
  implements BacnetPropertyListProvider
{
//region /*+ ------------ BEGIN BAJA AUTO GENERATED CODE ------------ +*/
//@formatter:off
/*@ $javax.baja.bacnet.export.BBacnetEventEnrollmentDescriptor(529595486)1.0$ @*/
/* Generated Tue Apr 08 11:52:29 CDT 2025 by Slot-o-Matic (c) Tridium, Inc. 2012-2025 */

  //region Property "eventEnrollmentOrd"

  /**
   * Slot for the {@code eventEnrollmentOrd} property.
   * @see #getEventEnrollmentOrd
   * @see #setEventEnrollmentOrd
   */
  @Generated
  public static final Property eventEnrollmentOrd = newProperty(Flags.DEFAULT_ON_CLONE, BOrd.DEFAULT, null);

  /**
   * Get the {@code eventEnrollmentOrd} property.
   * @see #eventEnrollmentOrd
   */
  @Generated
  public BOrd getEventEnrollmentOrd() { return (BOrd)get(eventEnrollmentOrd); }

  /**
   * Set the {@code eventEnrollmentOrd} property.
   * @see #eventEnrollmentOrd
   */
  @Generated
  public void setEventEnrollmentOrd(BOrd v) { set(eventEnrollmentOrd, v, null); }

  //endregion Property "eventEnrollmentOrd"

  //region Property "objectId"

  /**
   * Slot for the {@code objectId} property.
   * @see #getObjectId
   * @see #setObjectId
   */
  @Generated
  public static final Property objectId = newProperty(Flags.DEFAULT_ON_CLONE, BBacnetObjectIdentifier.make(BBacnetObjectType.EVENT_ENROLLMENT), null);

  /**
   * Get the {@code objectId} property.
   * @see #objectId
   */
  @Generated
  public BBacnetObjectIdentifier getObjectId() { return (BBacnetObjectIdentifier)get(objectId); }

  /**
   * Set the {@code objectId} property.
   * @see #objectId
   */
  @Generated
  public void setObjectId(BBacnetObjectIdentifier v) { set(objectId, v, null); }

  //endregion Property "objectId"

  //region Property "objectName"

  /**
   * Slot for the {@code objectName} property.
   * @see #getObjectName
   * @see #setObjectName
   */
  @Generated
  public static final Property objectName = newProperty(Flags.DEFAULT_ON_CLONE, "", null);

  /**
   * Get the {@code objectName} property.
   * @see #objectName
   */
  @Generated
  public String getObjectName() { return getString(objectName); }

  /**
   * Set the {@code objectName} property.
   * @see #objectName
   */
  @Generated
  public void setObjectName(String v) { setString(objectName, v, null); }

  //endregion Property "objectName"

  //region Property "description"

  /**
   * Slot for the {@code description} property.
   * @see #getDescription
   * @see #setDescription
   */
  @Generated
  public static final Property description = newProperty(Flags.DEFAULT_ON_CLONE, "", null);

  /**
   * Get the {@code description} property.
   * @see #description
   */
  @Generated
  public String getDescription() { return getString(description); }

  /**
   * Set the {@code description} property.
   * @see #description
   */
  @Generated
  public void setDescription(String v) { setString(description, v, null); }

  //endregion Property "description"

  //region Property "typeOfEvent"

  /**
   * Slot for the {@code typeOfEvent} property.
   * @see #getTypeOfEvent
   * @see #setTypeOfEvent
   */
  @Generated
  public static final Property typeOfEvent = newProperty(Flags.DEFAULT_ON_CLONE | Flags.READONLY, BBacnetEventType.none, null);

  /**
   * Get the {@code typeOfEvent} property.
   * @see #typeOfEvent
   */
  @Generated
  public BBacnetEventType getTypeOfEvent() { return (BBacnetEventType)get(typeOfEvent); }

  /**
   * Set the {@code typeOfEvent} property.
   * @see #typeOfEvent
   */
  @Generated
  public void setTypeOfEvent(BBacnetEventType v) { set(typeOfEvent, v, null); }

  //endregion Property "typeOfEvent"

  //region Property "notifyTypeId"

  /**
   * Slot for the {@code notifyTypeId} property.
   * @see #getNotifyTypeId
   * @see #setNotifyTypeId
   */
  @Deprecated
  @Generated
  public static final Property notifyTypeId = newProperty(Flags.DEFAULT_ON_CLONE | Flags.HIDDEN | Flags.READONLY, BNotifyType.alarm, null);

  /**
   * Get the {@code notifyTypeId} property.
   * @see #notifyTypeId
   */
  @Deprecated
  @Generated
  public BNotifyType getNotifyTypeId() { return (BNotifyType)get(notifyTypeId); }

  /**
   * Set the {@code notifyTypeId} property.
   * @see #notifyTypeId
   */
  @Deprecated
  @Generated
  public void setNotifyTypeId(BNotifyType v) { set(notifyTypeId, v, null); }

  //endregion Property "notifyTypeId"

  //region Property "objectPropertyReference"

  /**
   * Slot for the {@code objectPropertyReference} property.
   * @see #getObjectPropertyReference
   * @see #setObjectPropertyReference
   */
  @Generated
  public static final Property objectPropertyReference = newProperty(Flags.DEFAULT_ON_CLONE | Flags.READONLY | Flags.HIDDEN, new BBacnetDeviceObjectPropertyReference(), null);

  /**
   * Get the {@code objectPropertyReference} property.
   * @see #objectPropertyReference
   */
  @Generated
  public BBacnetDeviceObjectPropertyReference getObjectPropertyReference() { return (BBacnetDeviceObjectPropertyReference)get(objectPropertyReference); }

  /**
   * Set the {@code objectPropertyReference} property.
   * @see #objectPropertyReference
   */
  @Generated
  public void setObjectPropertyReference(BBacnetDeviceObjectPropertyReference v) { set(objectPropertyReference, v, null); }

  //endregion Property "objectPropertyReference"

  //region Property "notificationClassId"

  /**
   * Slot for the {@code notificationClassId} property.
   * @see #getNotificationClassId
   * @see #setNotificationClassId
   */
  @Generated
  public static final Property notificationClassId = newProperty(Flags.DEFAULT_ON_CLONE, 0, null);

  /**
   * Get the {@code notificationClassId} property.
   * @see #notificationClassId
   */
  @Generated
  public int getNotificationClassId() { return getInt(notificationClassId); }

  /**
   * Set the {@code notificationClassId} property.
   * @see #notificationClassId
   */
  @Generated
  public void setNotificationClassId(int v) { setInt(notificationClassId, v, null); }

  //endregion Property "notificationClassId"

  //region Property "reliability"

  /**
   * Slot for the {@code reliability} property.
   * Reliability of the Event Enrollment object to perform its monitoring function as described in
   * 135-2012 12.12.21. Does not reflect the reliability of the monitored object or the result of the
   * fault algorithm, if one is in use. Those other items are reflected in the BACnet Reliability
   * property as read through a BACnet request.
   * @see #getReliability
   * @see #setReliability
   */
  @Generated
  public static final Property reliability = newProperty(Flags.DEFAULT_ON_CLONE | Flags.READONLY, BBacnetReliability.configurationError, null);

  /**
   * Get the {@code reliability} property.
   * Reliability of the Event Enrollment object to perform its monitoring function as described in
   * 135-2012 12.12.21. Does not reflect the reliability of the monitored object or the result of the
   * fault algorithm, if one is in use. Those other items are reflected in the BACnet Reliability
   * property as read through a BACnet request.
   * @see #reliability
   */
  @Generated
  public BBacnetReliability getReliability() { return (BBacnetReliability)get(reliability); }

  /**
   * Set the {@code reliability} property.
   * Reliability of the Event Enrollment object to perform its monitoring function as described in
   * 135-2012 12.12.21. Does not reflect the reliability of the monitored object or the result of the
   * fault algorithm, if one is in use. Those other items are reflected in the BACnet Reliability
   * property as read through a BACnet request.
   * @see #reliability
   */
  @Generated
  public void setReliability(BBacnetReliability v) { set(reliability, v, null); }

  //endregion Property "reliability"

  //region Property "eventParameter"

  /**
   * Slot for the {@code eventParameter} property.
   * @see #getEventParameter
   * @see #setEventParameter
   */
  @Generated
  public static final Property eventParameter = newProperty(Flags.READONLY | Flags.HIDDEN, new BBacnetEventParameter(), null);

  /**
   * Get the {@code eventParameter} property.
   * @see #eventParameter
   */
  @Generated
  public BBacnetEventParameter getEventParameter() { return (BBacnetEventParameter)get(eventParameter); }

  /**
   * Set the {@code eventParameter} property.
   * @see #eventParameter
   */
  @Generated
  public void setEventParameter(BBacnetEventParameter v) { set(eventParameter, v, null); }

  //endregion Property "eventParameter"

  //region Property "notifyType"

  /**
   * Slot for the {@code notifyType} property.
   * @since Niagara 4.14u2
   * @since Niagara 4.15u1
   * @see #getNotifyType
   * @see #setNotifyType
   */
  @Generated
  public static final Property notifyType = newProperty(0, BBacnetNotifyType.alarm, BacUtil.makeBacnetNotifyTypeFacets());

  /**
   * Get the {@code notifyType} property.
   * @since Niagara 4.14u2
   * @since Niagara 4.15u1
   * @see #notifyType
   */
  @Generated
  public BBacnetNotifyType getNotifyType() { return (BBacnetNotifyType)get(notifyType); }

  /**
   * Set the {@code notifyType} property.
   * @since Niagara 4.14u2
   * @since Niagara 4.15u1
   * @see #notifyType
   */
  @Generated
  public void setNotifyType(BBacnetNotifyType v) { set(notifyType, v, null); }

  //endregion Property "notifyType"

  //region Type

  @Override
  @Generated
  public Type getType() { return TYPE; }
  @Generated
  public static final Type TYPE = Sys.loadType(BBacnetEventEnrollmentDescriptor.class);

  //endregion Type

//@formatter:on
//endregion /*+ ------------ END BAJA AUTO GENERATED CODE -------------- +*/

  //region BComponent

  @Override
  public void started()
    throws Exception
  {
    super.started();

    // Export the point and initialize the local copies.
    oldNotifyType = getNotifyType();
    oldId = getObjectId();
    oldName = getObjectName();
    setReliability(BBacnetReliability.noFaultDetected);

    if (Sys.isStationStarted())
    {
      initialize();
      // Increment the Device object's Database_Revision for created object.
      BBacnetNetwork.localDevice().incrementDatabaseRevision();
    }
  }

  @Override
  public void stationStarted() throws Exception
  {
    super.stationStarted();

    // Must wait for stationStarted to ensure that any pointDescriptors referenced by this
    // eventEnrollmentDescriptor are exported first.
    initialize();
  }

  private void initialize()
  {
    BBacnetNetwork.localDevice().export(this);

    BPointExtension pointExt = (BPointExtension) getObject();
    updateEventParameters(pointExt);

    // Initialize fields based on the associated point ext, if it can be resolved.
    getEventEnable(pointExt);
    getNotificationClass(pointExt);
    updateEventMessageTextsConfig(pointExt);
    updateEventAlgorithmInhibitInfo(pointExt);
    updateTimeDelayToNormal(pointExt);
  }

  @Override
  public void changed(Property p, Context cx)
  {
    super.changed(p, cx);

    if (!isRunning())
    {
      return;
    }

    if (p.equals(objectId))
    {
      BLocalBacnetDevice local = BBacnetNetwork.localDevice();
      local.unexport(oldId, oldName, this);
      checkConfiguration();
      oldId = getObjectId();

      try
      {
        ((BComponent)getParent()).rename(getPropertyInParent(), getObjectId().toString(nameContext));
      }
      catch (DuplicateSlotException e)
      {
        // ignore this
      }

      if (configOk)
      {
        local.incrementDatabaseRevision();
      }
    }
    else if (p.equals(objectName))
    {
      BLocalBacnetDevice local = BBacnetNetwork.localDevice();
      local.unexport(oldId, oldName, this);
      checkConfiguration();
      oldName = getObjectName();
      if (configOk)
      {
        local.incrementDatabaseRevision();
      }
    }
    else if (p.equals(eventEnrollmentOrd))
    {
      pointExt = null;
      BLocalBacnetDevice local = BBacnetNetwork.localDevice();
      local.exportByOrd(this);
    }
    else if (p.equals(notifyType))
    {
      if (getNotifyType() == BBacnetNotifyType.ackNotification)
      {
        logger.warning("Invalid Notify Type for " + this);
        setNotifyType(oldNotifyType);
      }
      else
      {
        oldNotifyType = getNotifyType();
      }
    }
  }

  @Override
  public void stopped()
    throws Exception
  {
    BLocalBacnetDevice local = BBacnetNetwork.localDevice();
    local.unexport(oldId, oldName, this);
    oldId = null;
    oldName = null;
    pointExt = null;

    // Increment the Device object's Database_Revision for deleted object.
    if (local.isRunning())
    {
      local.incrementDatabaseRevision();
    }

    super.stopped();
  }

  @Override
  public String toString(Context context)
  {
    return getObjectName() + " [" + getObjectId() + ']';
  }

  //endregion

  //region BIBacnetExportObject

  /**
   * Get the {@link BPointExtension} that {@link #eventEnrollmentOrd} points to. The extension's
   * parent should be the same as pointed to by {@link #objectPropertyReference}. Return null if the
   * eventEnrollmentOrd cannot be resolved to a BPointExtension, the extension has no parent, or the
   * extension's parent is not consistent with the objectPropertyReference.
   */
  @Override
  public BObject getObject()
  {
    BPointExtension pointExt = this.pointExt;
    if (pointExt == null)
    {
      pointExt = resolvePointExt();
      if (pointExt == null)
      {
        // Could not resolve the eventEnrollmentOrd
        return null;
      }
    }

    // Check that the extension has a parent
    BComponent target = (BComponent) pointExt.getParent();
    if (target == null)
    {
      if (logger.isLoggable(Level.FINE))
      {
        logger.fine(getObjectId() + ": associated PointExt (" + pointExt.getSlotPath() + ") has no parent");
      }
      resetDescriptor();
      return null;
    }

    // Resolved extension is consistent with the objectPropertyReference.
    this.pointExt = pointExt;
    return pointExt;
  }

  /**
   * Resolve and return the point extension pointed to by the eventEnrollmentOrd. Returns null if
   * the ord is not resolvable or resolves to something that is not instanceof BPointExtension.
   */
  private BPointExtension resolvePointExt()
  {
    BOrd objectOrd = getEventEnrollmentOrd();
    if (objectOrd.isNull())
    {
      if (logger.isLoggable(Level.FINE))
      {
        logger.fine(getObjectId() + ": eventEnrollmentOrd is null");
      }
      resetDescriptor();
      return null;
    }

    if (logger.isLoggable(Level.FINE))
    {
      logger.fine(getObjectId() + ": resolving eventEnrollmentOrd: " + objectOrd);
    }

    BObject resolved;
    try
    {
      resolved = objectOrd.get(this);
    }
    catch (Exception e)
    {
      logException(
        Level.WARNING,
        new StringBuilder(getObjectId().toString())
          .append(": could not resolve eventEnrollmentOrd: ")
          .append(objectOrd),
        e);
      resetDescriptor();
      return null;
    }

    if (resolved instanceof BAlarmSourceExt || resolved instanceof BBacnetTrendLogAlarmSourceExt)
    {
      return (BPointExtension) resolved;
    }

    if (logger.isLoggable(Level.FINE))
    {
      logger.fine(getObjectId() + ": eventEnrollmentOrd resolved to type " + resolved.getType() +
        " and not instanceof alarm:AlarmSourceExt or bacnet:BacnetTrendLogAlarmSourceExt");
    }
    resetDescriptor();
    return null;
  }

  /**
   * Get {@link #eventEnrollmentOrd}.
   */
  @Override
  public BOrd getObjectOrd()
  {
    return getEventEnrollmentOrd();
  }

  /**
   * Set {@link #eventEnrollmentOrd}.
   */
  @Override
  public void setObjectOrd(BOrd objectOrd, Context cx)
  {
    set(eventEnrollmentOrd, objectOrd, cx);
  }

  @Override
  public void checkConfiguration()
  {
    // quit if fatal fault
    if (isFatalFault())
    {
      setStatus(BStatus.makeFault(getStatus(), true));
      configOk = false;
      return;
    }

    if (!getObjectId().isValid())
    {
      setStatusFaulted("Invalid Object ID");
      return;
    }

    String err = BBacnetNetwork.localDevice().export(this);
    if (err != null)
    {
      duplicate = true;
      setStatusFaulted(err);
      return;
    }

    duplicate = false;
    configOk = true;
    setFaultCause("");
    setStatus(BStatus.ok);
  }

  private void setStatusFaulted(String faultCause)
  {
    setFaultCause(faultCause);
    setStatus(BStatus.makeFault(getStatus(), true));
    configOk = false;
  }

  @Override
  public int[] getPropertyList()
  {
    return BacnetPropertyList.makePropertyList(REQUIRED_PROPS, OPTIONAL_PROPS);
  }

  @SuppressWarnings("AssignmentOrReturnOfFieldWithMutableType")
  public int[] getOptionalProps()
  {
    return OPTIONAL_PROPS;
  }

  @SuppressWarnings("AssignmentOrReturnOfFieldWithMutableType")
  public int[] getRequiredProps()
  {
    return REQUIRED_PROPS;
  }

  @Override
  public PropertyValue readProperty(PropertyReference propertyReference)
    throws RejectException
  {
    return readProperty(propertyReference.getPropertyId(), propertyReference.getPropertyArrayIndex());
  }

  @Override
  public PropertyValue[] readPropertyMultiple(PropertyReference[] propertyReferences) throws RejectException
  {
    ArrayList<PropertyValue> results = new ArrayList<>(propertyReferences.length);
    for (PropertyReference ref : propertyReferences)
    {
      switch (ref.getPropertyId())
      {
        case BBacnetPropertyIdentifier.ALL:
          for (int prop : REQUIRED_PROPS)
          {
            results.add(readProperty(prop, NOT_USED));
          }
          for (int prop : OPTIONAL_PROPS)
          {
            results.add(readProperty(prop, NOT_USED));
          }
          break;

        case BBacnetPropertyIdentifier.OPTIONAL:
          for (int prop : OPTIONAL_PROPS)
          {
            results.add(readProperty(prop, NOT_USED));
          }
          break;

        case BBacnetPropertyIdentifier.REQUIRED:
          for (int prop : REQUIRED_PROPS)
          {
            results.add(readProperty(prop, NOT_USED));
          }
          break;

        default:
          results.add(readProperty(ref.getPropertyId(), ref.getPropertyArrayIndex()));
          break;
      }
    }

    return results.toArray(EMPTY_PROP_VALUE_ARRAY);
  }

  @Override
  public RangeData readRange(RangeReference rangeReference) throws RejectException
  {
    int propertyId = rangeReference.getPropertyId();
    if (!hasProperty(propertyId))
    {
      return new ReadRangeAck(property, unknownProperty);
    }

    return new ReadRangeAck(services, propertyIsNotA_List);
  }

  private static boolean hasProperty(int propertyId)
  {
    for (int id : REQUIRED_PROPS)
    {
      if (id == propertyId)
      {
        return true;
      }
    }

    for (int id : OPTIONAL_PROPS)
    {
      if (id == propertyId)
      {
        return true;
      }
    }

    // Property List is not included in either required or optional so check
    // that last.
    return propertyId == BBacnetPropertyIdentifier.PROPERTY_LIST;
  }

  @Override
  public ErrorType writeProperty(PropertyValue propertyValue) throws BacnetException
  {
    return writeProperty(
      propertyValue.getPropertyId(),
      propertyValue.getPropertyArrayIndex(),
      propertyValue.getPropertyValue(),
      propertyValue.getPriority());
  }

  @Override
  public ChangeListError addListElements(PropertyValue propertyValue)
    throws BacnetException
  {
    int propertyId = propertyValue.getPropertyId();
    if (!hasProperty(propertyId))
    {
      return makeAddListElementError(property, unknownProperty);
    }

    return makeAddListElementError(services, propertyIsNotA_List);
  }

  @Override
  public ChangeListError removeListElements(PropertyValue propertyValue)
    throws BacnetException
  {
    int propertyId = propertyValue.getPropertyId();
    if (!hasProperty(propertyId))
    {
      return makeRemoveListElementError(property, unknownProperty);
    }

    return makeRemoveListElementError(services, propertyIsNotA_List);
  }

  //endregion

  //region BBacnetEventSource

  /**
   * Get the event state associated with the alarm source extension
   */
  @Override
  public BBacnetEventState getEventState()
  {
    BPointExtension pointExt = (BPointExtension) getObject();
    if (pointExt instanceof BAlarmSourceExt)
    {
      return BBacnetEventState.make(((BAlarmSourceExt) pointExt).getAlarmState());
    }

    if (pointExt instanceof BBacnetTrendLogAlarmSourceExt)
    {
      return BBacnetEventState.make(((BBacnetTrendLogAlarmSourceExt) pointExt).getAlarmState());
    }

    if (logger.isLoggable(Level.FINE))
    {
      logger.fine(getObjectId() + ": associated PointExt (" + (pointExt != null ? pointExt.getSlotPath() : null) +
        ") is not an AlarmSourceExt or BacnetTrendLogAlarmSourceExt; returning null for event state");
    }

    return null;
  }

  private BBacnetEventState getEventState(BPointExtension pointExt)
  {
    if (pointExt instanceof BAlarmSourceExt)
    {
      return BBacnetEventState.make(((BAlarmSourceExt) pointExt).getAlarmState());
    }

    if (pointExt instanceof BBacnetTrendLogAlarmSourceExt)
    {
      return BBacnetEventState.make(((BBacnetTrendLogAlarmSourceExt) pointExt).getAlarmState());
    }

    if (logger.isLoggable(Level.FINE))
    {
      logger.fine(getObjectId() + ": associated PointExt (" + (pointExt != null ? pointExt.getSlotPath() : null) +
        ") is not an AlarmSourceExt or BacnetTrendLogAlarmSourceExt; returning BacnetEventState.normal");
    }
    return BBacnetEventState.normal;
  }

  /**
   * Get the parent point of the BPointExtension returned by {@link #getObject()} if that extension
   * is instanceof BAlarmSourceExt. Otherwise, return null if getObject() returns null or the
   * extension is not instanceof BAlarmSourceExt.
   */
  @Override
  public BControlPoint getPoint()
  {
    BPointExtension pointExt = (BPointExtension) getObject();
    if (pointExt instanceof BAlarmSourceExt)
    {
      return pointExt.getParentPoint();
    }

    if (logger.isLoggable(Level.FINE))
    {
      logger.fine(getObjectId() + ": associated PointExt (" + (pointExt != null ? pointExt.getSlotPath() : null) +
        ") is not an AlarmSourceExt; returning null for getPoint");
    }
    return null;
  }

  /**
   * This method returns three flags that separately indicate the acknowledgment state for
   * TO_OFFNORMAL, TO_FAULT, and TO_NORMAL events
   */
  @Override
  public BBacnetBitString getAckedTransitions()
  {
    BAlarmTransitionBits ackedTransitions = getAckedTransitions((BPointExtension) getObject());
    if (ackedTransitions == null)
    {
      if (logger.isLoggable(Level.FINE))
      {
        logger.fine(getObjectId() + ": associated PointExt (" + (pointExt != null ? pointExt.getSlotPath() : null) +
          ") is not an AlarmSourceExt or BacnetTrendLogAlarmSourceExt; returning default ackedTransitions value");
      }
      return null;
    }

    return BacnetBitStringUtil.getBacnetEventTransitionBits(ackedTransitions);
  }

  private static BAlarmTransitionBits getAckedTransitions(BPointExtension pointExt)
  {
    if (pointExt instanceof BAlarmSourceExt)
    {
      return ((BAlarmSourceExt) pointExt).getAckedTransitions();
    }

    if (pointExt instanceof BBacnetTrendLogAlarmSourceExt)
    {
      return ((BBacnetTrendLogAlarmSourceExt) pointExt).getAckedTransitions();
    }

    return null;
  }

  @Override
  public BBacnetTimeStamp[] getEventTimeStamps()
  {
    BPointExtension pointExt = (BPointExtension) getObject();
    if (pointExt instanceof BAlarmSourceExt)
    {
      BAlarmSourceExt alarmExt = (BAlarmSourceExt) pointExt;
      return new BBacnetTimeStamp[] {
        new BBacnetTimeStamp(alarmExt.getLastOffnormalTime()),
        new BBacnetTimeStamp(alarmExt.getLastFaultTime()),
        new BBacnetTimeStamp(alarmExt.getLastToNormalTime())
      };
    }
    else if (pointExt instanceof BBacnetTrendLogAlarmSourceExt)
    {
      BBacnetTrendLogAlarmSourceExt trendAlarmExt = (BBacnetTrendLogAlarmSourceExt) pointExt;
      BAlarmTimestamps toOffnormalTimes = trendAlarmExt.getToOffnormalTimes();
      BAlarmTimestamps toFaultTimes = trendAlarmExt.getToFaultTimes();

      BAbsTime normalTime = toOffnormalTimes.getNormalTime();
      if (normalTime.isBefore(toFaultTimes.getNormalTime()))
      {
        normalTime = toFaultTimes.getNormalTime();
      }

      return new BBacnetTimeStamp[] {
        new BBacnetTimeStamp(toOffnormalTimes.getAlarmTime()),
        new BBacnetTimeStamp(toFaultTimes.getAlarmTime()),
        new BBacnetTimeStamp(normalTime)
      };
    }
    else
    {
      if (logger.isLoggable(Level.FINE))
      {
        logger.fine(getObjectId() + ": associated PointExt (" + (pointExt != null ? pointExt.getSlotPath() : null) +
          ") is not an AlarmSourceExt or BacnetTrendLogAlarmSourceExt; returning default eventTimeStamps");
      }

      // TODO Ensure this is compatible with all calls to this method.
      return null;
    }
  }

  @Override
  public BBacnetBitString getEventEnable()
  {
    BPointExtension pointExt = (BPointExtension) getObject();
    BAlarmTransitionBits alarmEnable = getEventEnable(pointExt);
    return alarmEnable != null ? BacnetBitStringUtil.getBacnetEventTransitionBits(alarmEnable) : null;
  }

  private BAlarmTransitionBits getEventEnable(BPointExtension pointExt)
  {
    BAlarmTransitionBits alarmEnable = null;
    if (pointExt instanceof BAlarmSourceExt)
    {
      alarmEnable = ((BAlarmSourceExt) pointExt).getAlarmEnable();
    }
    else if (pointExt instanceof BBacnetTrendLogAlarmSourceExt)
    {
      alarmEnable = ((BBacnetTrendLogAlarmSourceExt) pointExt).getAlarmEnable();
    }

    if (alarmEnable != null)
    {
      eventEnable = alarmEnable;
    }
    else
    {
      if (logger.isLoggable(Level.FINE))
      {
        logger.fine(getObjectId() + ": associated PointExt (" + (pointExt != null ? pointExt.getSlotPath() : null) +
          ") is not an AlarmSourceExt or BacnetTrendLogAlarmSourceExt");
      }
    }

    return alarmEnable;
  }

  @Override
  public BEnum getEventType()
  {
    updateEventParameters((BPointExtension) getObject());
    return getTypeOfEvent();
  }

  @Override
  public void statusChanged()
  {
    setBacnetStatusFlags(getStatusFlags());
  }

  /**
   *  This method represents four Boolean flags that indicate the general "health" of an object.
   *  These properties are :- <br>
   *  <ul>
   *    <li> IN_ALARM : Logical FALSE (0) if the Event_State property has a value of NORMAL, otherwise logical TRUE (1) </li>
   *    <li> FAULT : Logical TRUE (1) if the Reliability property is present and does not have a value of NO_FAULT_DETECTED, otherwise logical FALSE (0) </li>
   *    <li> OVERRIDDEN : Logical FALSE (0) </li>
   *    <li> OUT_OF_SERVICE : Logical FALSE (0) </li>
   *  </ul>
   * @return status Flags' BitString
   */
  public BBacnetBitString getStatusFlags()
  {
    BPointExtension pointExt = (BPointExtension) getObject();
    if (pointExt == null)
    {
      if (logger.isLoggable(Level.FINE))
      {
        logger.fine(getObjectId() + ": associated PointExt is not an AlarmSourceExt or BacnetTrendLogAlarmSourceExt" +
          "; returning default status flags");
      }
      return STATUS_FLAGS_DEFAULT;
    }

    return BBacnetBitString.make(new boolean[] {
      /* inAlarm */ !BBacnetEventState.isNormal(getEventState(pointExt)),
      // TODO Add support for reliability in the alarm lifecycle of a BACnet object
      /* inFault */ false, // !readReliability().equals(BBacnetReliability.noFaultDetected),
      /* isOverridden */ false,
      /* isOutOfService */ false });
  }

  @Override
  public boolean isValidAlarmExt(BIAlarmSource ext)
  {
    BPointExtension pointExt = (BPointExtension) getObject();
    // TODO Check the offnormal algorithm of the BAlarmSourceExt
    return pointExt instanceof BAlarmSourceExt || pointExt instanceof BBacnetTrendLogAlarmSourceExt;
  }

  @Override
  public boolean isEventInitiationEnabled()
  {
    return true;
  }

  @Override
  public int[] getEventPriorities()
  {
    BBacnetNotificationClassDescriptor nc = getNotificationClass();
    return nc != null ? nc.getEventPriorities() : null;
  }

  @Override
  public BBacnetNotificationClassDescriptor getNotificationClass()
  {
    return getNotificationClass((BPointExtension) getObject());
  }

  private BBacnetNotificationClassDescriptor getNotificationClass(BPointExtension pointExt)
  {
    String alarmClassName = getAlarmClassName(pointExt);
    if (alarmClassName == null)
    {
      // No associated point extension so no alarm class to correlate
      if (logger.isLoggable(Level.FINE))
      {
        logger.fine(getObjectId() + ": cannot retrieve the notification class descriptor-" +
          " associated PointExt (" + (pointExt != null ? pointExt.getSlotPath() : null) +
          ") is not an AlarmSourceExt or BacnetTrendLogAlarmSourceExt");
      }
      return null;
    }

    if (alarmClassName.isEmpty())
    {
      if (logger.isLoggable(Level.FINE))
      {
        logger.fine(getObjectId() + ": pointExt's alarmClassName is an empty string;" +
          " setting notification class ID to unconfigured instance number");
      }
      setNotificationClassId(BBacnetObjectIdentifier.UNCONFIGURED_INSTANCE_NUMBER);
      // TODO Add support for internal reliability evaluation
      //setReliability(BBacnetReliability.configurationError);
      return null;
    }

    BBacnetNotificationClassDescriptor descriptor = findNotificationClass(alarmClassName);
    if (descriptor == null)
    {
      // TODO Add support for internal reliability evaluation
      //setReliability(BBacnetReliability.configurationError);
      return null;
    }

    setNotificationClassId(descriptor.getObjectId().getInstanceNumber());
    return descriptor;
  }

  private BBacnetNotificationClassDescriptor findNotificationClass(String alarmClassName)
  {
    BAlarmService alarmService;
    try
    {
      alarmService = (BAlarmService) Sys.getService(BAlarmService.TYPE);
    }
    catch (ServiceNotFoundException e)
    {
      logException(
        Level.WARNING,
        new StringBuilder(getObjectId().toString())
          .append(": getNotificationClass: could not find the alarm service"),
        e);
      return null;
    }

    BAlarmClass alarmClass = alarmService.lookupAlarmClass(alarmClassName);
    if (alarmClass == null)
    {
      if (logger.isLoggable(Level.FINE))
      {
        logger.fine(getObjectId() + ": getNotificationClass:" +
          " could not find alarm class " + alarmClassName + " in the alarm service");
      }
      return null;
    }

    BIBacnetExportObject descriptor = findDescriptor(alarmClass.getHandleOrd());
    if (descriptor instanceof BBacnetNotificationClassDescriptor)
    {
      if (logger.isLoggable(Level.FINE))
      {
        logger.fine(getObjectId() + ": getNotificationClass:" +
          " found new notification class descriptor for alarm class " + alarmClassName);
      }
      return (BBacnetNotificationClassDescriptor) descriptor;
    }
    else
    {
      if (logger.isLoggable(Level.FINE))
      {
        logger.fine(getObjectId() + ": getNotificationClass:" +
          " could not find a notification class descriptor for alarmClass \"" + alarmClass.getSlotPath() + '\"');
      }
      return null;
    }
  }

  @SuppressWarnings("deprecation")
  @Override
  @Deprecated
  protected void updateAlarmInhibit()
  {
  }

  //endregion

  //region Read Property

  /**
   * Get the value of a property.
   * Subclasses with additional properties override this to check for
   * their properties.  If no match is found, call this superclass
   * method to check these properties.
   *
   * @param pId the requested property-identifier.
   * @param ndx the property array index (-1 if not specified).
   * @return a PropertyValue containing either the encoded value or the error.
   */
  protected PropertyValue readProperty(int pId, int ndx)
  {
    if (ndx >= 0 && !isArray(pId))
    {
      return new NReadPropertyResult(pId, ndx, new NErrorType(BBacnetErrorClass.PROPERTY, BBacnetErrorCode.PROPERTY_IS_NOT_AN_ARRAY));
    }

    if (ndx < NOT_USED)
    {
      return new NReadPropertyResult(pId, ndx, new NErrorType(BBacnetErrorClass.PROPERTY, BBacnetErrorCode.INVALID_ARRAY_INDEX));
    }

    switch (pId)
    {
      case BBacnetPropertyIdentifier.OBJECT_IDENTIFIER:
        return new NReadPropertyResult(pId, ndx, AsnUtil.toAsnObjectId(getObjectId()));
      case BBacnetPropertyIdentifier.OBJECT_NAME:
        return new NReadPropertyResult(pId, ndx, AsnUtil.toAsnCharacterString(getObjectName()));
      case BBacnetPropertyIdentifier.OBJECT_TYPE:
        return new NReadPropertyResult(pId, ndx, AsnUtil.toAsnEnumerated(BBacnetObjectType.EVENT_ENROLLMENT));
      case BBacnetPropertyIdentifier.EVENT_TYPE:
        return new NReadPropertyResult(pId, ndx, AsnUtil.toAsnEnumerated(getEventType()));
      case BBacnetPropertyIdentifier.NOTIFY_TYPE:
        return new NReadPropertyResult(pId, ndx, AsnUtil.toAsnEnumerated(getNotifyType()));
      case BBacnetPropertyIdentifier.EVENT_PARAMETERS:
        return readEventParameters();
      case BBacnetPropertyIdentifier.OBJECT_PROPERTY_REFERENCE:
        return readObjectPropertyReference();
      case BBacnetPropertyIdentifier.EVENT_STATE:
        return readEventState();
      case BBacnetPropertyIdentifier.EVENT_ENABLE:
        return readEventEnable();
      case BBacnetPropertyIdentifier.ACKED_TRANSITIONS:
        return readAckedTransitions();
      case BBacnetPropertyIdentifier.NOTIFICATION_CLASS:
        getNotificationClass();
        return new NReadPropertyResult(pId, ndx, AsnUtil.toAsnUnsigned(getNotificationClassId()));
      case BBacnetPropertyIdentifier.EVENT_TIME_STAMPS:
        return readEventTimeStamps(ndx);
      case BBacnetPropertyIdentifier.EVENT_DETECTION_ENABLE:
        return new NReadPropertyResult(pId, ndx, AsnUtil.toAsnBoolean(getEventDetectionEnable()));
      case BBacnetPropertyIdentifier.STATUS_FLAGS:
        return new NReadPropertyResult(pId, ndx, AsnUtil.toAsnBitString(getStatusFlags()));
      case BBacnetPropertyIdentifier.RELIABILITY:
        // TODO Add support for reliability in the alarm lifecycle of a BACnet object
        return new NReadPropertyResult(pId, ndx, AsnUtil.toAsnEnumerated(BBacnetReliability.noFaultDetected));
      case BBacnetPropertyIdentifier.PROPERTY_LIST:
        return readPropertyList(ndx);
      default:
        return readOptionalProperty(pId, ndx);
    }
  }

  private static String getAlarmClassName(BPointExtension pointExt)
  {
    if (pointExt instanceof BAlarmSourceExt)
    {
      return ((BAlarmSourceExt) pointExt).getAlarmClass();
    }

    if (pointExt instanceof BBacnetTrendLogAlarmSourceExt)
    {
      return ((BBacnetTrendLogAlarmSourceExt) pointExt).getAlarmClass();
    }

    return null;
  }

  private PropertyValue readEventParameters()
  {
    try
    {
      BBacnetEventParameter params = readEventParameters((BPointExtension) getObject());
      return new NReadPropertyResult(
        BBacnetPropertyIdentifier.EVENT_PARAMETERS,
        AsnUtil.toAsn(BacnetConst.ASN_ANY, params));
    }
    catch (EventEnrollmentException e)
    {
      return new NReadPropertyResult(BBacnetPropertyIdentifier.EVENT_PARAMETERS, e.errorType);
    }
  }

  private PropertyValue readObjectPropertyReference()
  {
    BPointExtension pointExt = (BPointExtension) getObject();
    if (pointExt == null)
    {
      if (logger.isLoggable(Level.FINE))
      {
        logger.fine(getObjectId() + ": target alarm ext not configured; falling back to cached objectPropertyReference value");
      }
      return makeObjPropRefPropValue(getObjectPropertyReference());
    }

    BComponent target = (BComponent) pointExt.getParent();
    if (target == null)
    {
      if (logger.isLoggable(Level.FINE))
      {
        logger.fine(getObjectId() + ": target alarm ext has no parent");
      }
      return makeObjPropRefError();
    }

    return getTargetObjPropRef(target);
  }

  private PropertyValue getTargetObjPropRef(BComponent target)
  {
    if (target instanceof BControlPoint)
    {
      return getPointPropRef((BControlPoint) target);
    }
    else if (target instanceof BIBacnetTrendLogExt)
    {
      return getTrendPropRef((BIBacnetTrendLogExt) target);
    }
    else
    {
      if (logger.isLoggable(Level.FINE))
      {
        logger.fine(getObjectId() + ": target alarm ext has no parent");
      }
      return makeObjPropRefError();
    }
  }

  private PropertyValue getPointPropRef(BControlPoint point)
  {
    BAbstractProxyExt proxyExt = point.getProxyExt();
    if (proxyExt instanceof BBacnetProxyExt)
    {
      return makeObjPropRefPropValue(makeRemoteDeviceObjPropRef((BBacnetProxyExt) proxyExt));
    }
    else
    {
      // Must be a local object
      BIBacnetExportObject descriptor = findDescriptor(point.getHandleOrd());
      if (descriptor == null)
      {
        if (logger.isLoggable(Level.FINE))
        {
          logger.fine(getObjectId() + ": alarm ext's target's descriptor not found in local device;" +
            " target slot path: " + point.getSlotPath());
        }
        return makeObjPropRefError();
      }

      // Device ID should be blank for local objects
      BBacnetDeviceObjectPropertyReference objPropRef = new BBacnetDeviceObjectPropertyReference(
        /* objectId */ descriptor.getObjectId(),
        /* propertyId */ presentValue);
      return makeObjPropRefPropValue(objPropRef);
    }
  }

  private PropertyValue getTrendPropRef(BIBacnetTrendLogExt trendLogExt)
  {
    if (trendLogExt instanceof BBacnetTrendLogRemoteExt)
    {
      BBacnetTrendLogRemoteExt trendLogRemoteExt = (BBacnetTrendLogRemoteExt) trendLogExt;
      BBacnetDeviceObjectPropertyReference deviceObjPropRef = new BBacnetDeviceObjectPropertyReference(
        trendLogRemoteExt.getObjectId(),
        trendLogRemoteExt.getPropertyId(),
        trendLogRemoteExt.getArrayIndex(),
        trendLogRemoteExt.getDevice().getObjectId());
      return makeObjPropRefPropValue(deviceObjPropRef);
    }
    else
    {
      BIBacnetExportObject descriptor = findDescriptor(((BComponent) trendLogExt).getHandleOrd());
      if (descriptor == null)
      {
        if (logger.isLoggable(Level.FINE))
        {
          logger.fine(getObjectId() + ": alarm ext's target's descriptor not found in local device;" +
            " target slot path: " + ((BComponent) trendLogExt).getSlotPath());
        }
        return makeObjPropRefError();
      }

      BBacnetDeviceObjectPropertyReference objPropRef = new BBacnetDeviceObjectPropertyReference(
        /* objectId */ descriptor.getObjectId(),
        /* propertyId */ logBuffer,
        /* deviceId */ BBacnetNetwork.localDevice().getObjectId());
      return makeObjPropRefPropValue(objPropRef);
    }
  }

  private static PropertyValue makeObjPropRefPropValue(BBacnetDeviceObjectPropertyReference objPropRef)
  {
    return new NReadPropertyResult(
      BBacnetPropertyIdentifier.OBJECT_PROPERTY_REFERENCE,
      BacnetConst.NOT_USED,
      AsnUtil.toAsn(BacnetConst.ASN_ANY, objPropRef));
  }

  private static PropertyValue makeObjPropRefError()
  {
    return new NReadPropertyResult(
      BBacnetPropertyIdentifier.OBJECT_PROPERTY_REFERENCE,
      new NErrorType(BBacnetErrorClass.PROPERTY, BBacnetErrorCode.OTHER));
  }

  private PropertyValue readEventState()
  {
    BBacnetEventState eventState = null;
    if (getEventDetectionEnable())
    {
      eventState = getEventState();
    }

    if (eventState == null)
    {
      eventState = BBacnetEventState.normal;
    }

    return new NReadPropertyResult(BBacnetPropertyIdentifier.EVENT_STATE, NOT_USED, AsnUtil.toAsnEnumerated(eventState));
  }

  private PropertyValue readEventEnable()
  {
    BBacnetBitString eventEnable = getEventEnable();
    if (eventEnable == null)
    {
      eventEnable = BacnetBitStringUtil.getBacnetEventTransitionBits(this.eventEnable);
    }

    return new NReadPropertyResult(BBacnetPropertyIdentifier.EVENT_ENABLE, NOT_USED, AsnUtil.toAsnBitString(eventEnable));
  }

  private PropertyValue readAckedTransitions()
  {
    if (!getEventDetectionEnable())
    {
      return new NReadPropertyResult(BBacnetPropertyIdentifier.ACKED_TRANSITIONS, NOT_USED, AsnUtil.toAsnBitString(ACKED_TRANS_DEFAULT));
    }

    BAlarmTransitionBits ackedTrans = getAckedTransitions((BPointExtension) getObject());
    if (ackedTrans == null)
    {
      return new NReadPropertyResult(BBacnetPropertyIdentifier.ACKED_TRANSITIONS, NOT_USED, AsnUtil.toAsnBitString(ACKED_TRANS_DEFAULT));
    }

    BAlarmTransitionBits eventTrans = readEventTransition(ackedTrans);
    return new NReadPropertyResult(
      BBacnetPropertyIdentifier.ACKED_TRANSITIONS,
      NOT_USED,
      AsnUtil.toAsnBitString(BacnetBitStringUtil.getBacnetEventTransitionBits(eventTrans)));
  }

  /**
   * Read the value of an optional property.
   * Subclasses with additional properties override this to check for
   * their properties.  If no match is found, call this superclass
   * method to check these properties.
   *
   * @param pId the requested property-identifier.
   * @param ndx the property array index (-1 if not specified).
   * @return a PropertyValue containing either the encoded value or the error.
   */
  protected PropertyValue readOptionalProperty(int pId, int ndx)
  {
    switch (pId)
    {
      case BBacnetPropertyIdentifier.DESCRIPTION:
        return new NReadPropertyResult(pId, ndx, AsnUtil.toAsnCharacterString(getDescription()));
      case BBacnetPropertyIdentifier.EVENT_MESSAGE_TEXTS:
        return readEventMessageTexts(ndx);
      case BBacnetPropertyIdentifier.EVENT_MESSAGE_TEXTS_CONFIG:
        return readEventMessageTextsConfig(ndx);
      case BBacnetPropertyIdentifier.EVENT_ALGORITHM_INHIBIT:
        return readEventAlgorithmInhibit();
      case BBacnetPropertyIdentifier.EVENT_ALGORITHM_INHIBIT_REF:
        return readEventAlgorithmInhibitRef();
      case BBacnetPropertyIdentifier.TIME_DELAY_NORMAL:
        return readTimeDelayNormal();
    }

    return new NReadPropertyResult(pId, ndx, new NErrorType(BBacnetErrorClass.PROPERTY, BBacnetErrorCode.UNKNOWN_PROPERTY));
  }

  private PropertyValue readEventMessageTextsConfig(int ndx)
  {
    updateEventMessageTextsConfig((BPointExtension) getObject());
    return readEventMessageTextsConfig(
      toOffnormalText,
      toFaultText,
      toNormalText,
      ndx);
  }

  private void updateEventMessageTextsConfig(BPointExtension pointExt)
  {
    if (pointExt instanceof BAlarmSourceExt)
    {
      BAlarmSourceExt alarmExt = (BAlarmSourceExt) pointExt;
      toOffnormalText = alarmExt.getToOffnormalText().getFormat();
      toFaultText = alarmExt.getToFaultText().getFormat();
      toNormalText = alarmExt.getToNormalText().getFormat();
    }
    else if (pointExt instanceof BBacnetTrendLogAlarmSourceExt)
    {
      BBacnetTrendLogAlarmSourceExt trendAlarmExt = (BBacnetTrendLogAlarmSourceExt) pointExt;
      toOffnormalText = trendAlarmExt.getToOffnormalText().getFormat();
      toFaultText = trendAlarmExt.getToFaultText().getFormat();
      toNormalText = trendAlarmExt.getToNormalText().getFormat();
    }
  }

  private void updateEventParameters(BPointExtension pointExt)
  {
    try
    {
      readEventParameters(pointExt);
    }
    catch (EventEnrollmentException e)
    {
      logger.log(Level.FINE, getObjectId() + ": exception while updating event parameters", e);
    }
  }

  private BBacnetEventParameter readEventParameters(BPointExtension pointExt)
    throws EventEnrollmentException
  {
    if (pointExt == null)
    {
      if (logger.isLoggable(Level.FINE))
      {
        logger.fine(getObjectId() + ": target alarm ext not configured; falling back to cached eventParameters value");
      }
      return getEventParameter();
    }

    // Extension exists- base the event parameters on the alarm extension's properties
    BBacnetEventParameter eventParam;

    if (pointExt instanceof BBacnetTrendLogAlarmSourceExt)
    {
      BBacnetTrendLogAlarmSourceExt trendAlarmExt = (BBacnetTrendLogAlarmSourceExt) pointExt;
      eventParam = BBacnetEventParameter.makeBufferReady(
        trendAlarmExt.getNotificationThreshold(),
        trendAlarmExt.getLastNotifyRecord());
    }
    else if (pointExt instanceof BAlarmSourceExt)
    {
      BAlarmSourceExt alarmExt = (BAlarmSourceExt) pointExt;
      BOffnormalAlgorithm offnormalAlgorithm = alarmExt.getOffnormalAlgorithm();
      if (offnormalAlgorithm instanceof BBooleanChangeOfStateAlgorithm)
      {
        eventParam = BBacnetEventParameter.makeChangeOfState(
          alarmExt.getTimeDelay(),
          getListOfValues((BBooleanChangeOfStateAlgorithm) offnormalAlgorithm));
      }
      else if (offnormalAlgorithm instanceof BEnumChangeOfStateAlgorithm)
      {
        eventParam = BBacnetEventParameter.makeChangeOfState(
          alarmExt.getTimeDelay(),
          getListOfValues((BEnumChangeOfStateAlgorithm) offnormalAlgorithm));
      }
      else if (offnormalAlgorithm instanceof BNumericChangeOfStateAlgorithm)
      {
        eventParam = BBacnetEventParameter.makeChangeOfState(
          alarmExt.getTimeDelay(),
          getListOfValues((BNumericChangeOfStateAlgorithm) offnormalAlgorithm));
      }
      else if (offnormalAlgorithm instanceof BStringChangeOfStateAlgorithm)
      {
        eventParam = BBacnetEventParameter.makeChangeOfCharacterString(
          alarmExt.getTimeDelay(),
          getListOfValues((BStringChangeOfStateAlgorithm) offnormalAlgorithm));
      }
      else if (offnormalAlgorithm instanceof BBooleanCommandFailureAlgorithm)
      {
        eventParam = BBacnetEventParameter.makeCommandFailure(
          alarmExt.getTimeDelay(),
          getLinkedPropertyReference(
            offnormalAlgorithm,
            BBooleanCommandFailureAlgorithm.feedbackValue,
            BBacnetBinaryPointDescriptor.TYPE));
      }
      else if (offnormalAlgorithm instanceof BEnumCommandFailureAlgorithm)
      {
        eventParam = BBacnetEventParameter.makeCommandFailure(
          alarmExt.getTimeDelay(),
          getLinkedPropertyReference(
            offnormalAlgorithm,
            BEnumCommandFailureAlgorithm.feedbackValue,
            BBacnetMultiStatePointDescriptor.TYPE));
      }
      else if (offnormalAlgorithm instanceof BFloatingLimitAlgorithm)
      {
        BFloatingLimitAlgorithm floatingLimitAlgorithm = (BFloatingLimitAlgorithm) offnormalAlgorithm;
        eventParam = BBacnetEventParameter.makeFloatingLimit(
          alarmExt.getTimeDelay(),
          getLinkedPropertyReference(
            offnormalAlgorithm,
            BFloatingLimitAlgorithm.setpoint,
            BBacnetAnalogPointDescriptor.TYPE),
          (float) floatingLimitAlgorithm.getLowDiffLimit(),
          (float) floatingLimitAlgorithm.getHighDiffLimit(),
          (float) floatingLimitAlgorithm.getDeadband());
      }
      else if (offnormalAlgorithm instanceof BOutOfRangeAlgorithm)
      {
        BOutOfRangeAlgorithm outOfRangeAlgorithm = (BOutOfRangeAlgorithm) offnormalAlgorithm;
        eventParam = BBacnetEventParameter.makeOutOfRange(
          getOutOfRangeEventType(),
          alarmExt.getTimeDelay(),
          outOfRangeAlgorithm.getLowLimit(),
          outOfRangeAlgorithm.getHighLimit(),
          outOfRangeAlgorithm.getDeadband());
      }
      else if (offnormalAlgorithm instanceof BBacnetChangeOfDiscreteValueAlgorithm)
      {
        eventParam = BBacnetEventParameter.makeChangeOfDiscreteValue(alarmExt.getTimeDelay());
      }
      else
      {
        logger.warning(getObjectId() + ": could not construct EventParameters for BAlarmExt offnormalAlgorithm of type " +
          offnormalAlgorithm.getType());
        throw new EventEnrollmentException(
          "alarmExt offnormal algorithm type " + offnormalAlgorithm.getType() + " not supported",
          new NErrorType(BBacnetErrorClass.PROPERTY, BBacnetErrorCode.OTHER));
      }
    }
    else
    {
      logger.warning(getObjectId() + ": could not construct EventParameters for BPointExtension of type " +
        pointExt.getType());
      throw new EventEnrollmentException(
        "pointExt type " + pointExt.getType() + " not supported",
        new NErrorType(BBacnetErrorClass.PROPERTY, BBacnetErrorCode.OTHER));
    }

    setEventParameter(eventParam);
    setTypeOfEvent(BBacnetEventType.make(eventParam.getChoice()));
    return eventParam;
  }

  private BBacnetEventType getOutOfRangeEventType()
    throws EventEnrollmentException
  {
    BBacnetDeviceObjectPropertyReference objPropRef = getObjectPropertyReference();
    BBacnetObjectIdentifier objId = objPropRef.getObjectId();
    int propId = objPropRef.getPropertyId();
    PropertyInfo propInfo = getPropertyInfo(objId.getObjectType(), propId);
    if (propInfo == null)
    {
      throw new EventEnrollmentException(
        "BACnet property information not found for" +
          " object ID: " + objId +
          ", property ID: " + BBacnetPropertyIdentifier.tag(propId),
        new NErrorType(property, other));
    }

    int asnType = propInfo.getAsnType();
    if (asnType == ASN_REAL)
    {
      return BBacnetEventType.outOfRange;
    }
    else if (asnType == ASN_DOUBLE)
    {
      return BBacnetEventType.doubleOutOfRange;
    }
    else if (asnType == ASN_INTEGER)
    {
      return BBacnetEventType.signedOutOfRange;
    }
    else if (asnType == ASN_UNSIGNED || isArraySize(propInfo, objPropRef))
    {
      return BBacnetEventType.unsignedOutOfRange;
    }
    else
    {
      logger.warning(getObjectId() + ": could not construct EventParameters for BAlarmExt OutOfRangeAlgorithm " +
        " because of invalid ASN type " + AsnUtil.getAsnTypeName(asnType));
      throw new EventEnrollmentException(
        "alarmExt outOfRange algorithm type, found invalid ASN type " + AsnUtil.getAsnTypeName(asnType),
        new NErrorType(property, other));
    }
  }

  private BBacnetListOf getListOfValues(BBooleanChangeOfStateAlgorithm offnormalAlgorithm)
    throws EventEnrollmentException
  {
    PropertyInfo propInfo = findPropertyInfo(getObjectPropertyReference());
    BBacnetPropertyStates element;
    if (propInfo.getAsnType() == BacnetConst.ASN_BOOLEAN)
    {
      element = BBacnetPropertyStates.makeBoolean(offnormalAlgorithm.getAlarmValue());
    }
    else if (isBinaryPv(propInfo))
    {
      element = BBacnetPropertyStates.makeBinaryPv(offnormalAlgorithm.getAlarmValue());
    }
    else
    {
      throw new EventEnrollmentException(
        "Boolean change-of-state alarm extension is not supported for the property type: " + propInfo,
        new NErrorType(BBacnetErrorClass.PROPERTY, BBacnetErrorCode.OTHER));
    }

    BBacnetListOf listOfValues = new BBacnetListOf(BBacnetPropertyStates.TYPE);
    listOfValues.addListElement(element, null);
    return listOfValues;
  }

  private BBacnetListOf getListOfValues(BEnumChangeOfStateAlgorithm offnormalAlgorithm)
    throws EventEnrollmentException
  {
    BBacnetDeviceObjectPropertyReference objectPropRef = getObjectPropertyReference();
    PropertyInfo propInfo = findPropertyInfo(objectPropRef);

    BBacnetListOf listOfValues = new BBacnetListOf(BBacnetPropertyStates.TYPE);

    if (propInfo.isEnum())
    {
      BTypeSpec propTypeSpec = BTypeSpec.make(propInfo.getType());
      BObject range = offnormalAlgorithm.getSlotFacets(BEnumChangeOfStateAlgorithm.alarmValues).getFacet(BFacets.RANGE);
      if (range instanceof BEnumRange)
      {
        Type frozenType = ((BEnumRange) range).getFrozenType();
        if (frozenType != null && !frozenType.getTypeSpec().equals(propTypeSpec))
        {
          throw new EventEnrollmentException(
            getObjectId() +
              ": enum change-of-state alarm value enum type \"" + frozenType + '"' +
              " does not match property type \"" + propInfo + '"' +
              " for object property reference: " + objectPropRef,
            new NErrorType(BBacnetErrorClass.PROPERTY, BBacnetErrorCode.OTHER));
        }
      }

      BEnumRange alarmValues = offnormalAlgorithm.getAlarmValues();
      for (int ordinal : alarmValues.getOrdinals())
      {
        listOfValues.addListElement(BBacnetPropertyStates.makeEnum(propTypeSpec, ordinal), null);
      }
    }
    else if (propInfo.getAsnType() == BacnetConst.ASN_UNSIGNED)
    {
      for (int ordinal : offnormalAlgorithm.getAlarmValues().getOrdinals())
      {
        if (ordinal < 0)
        {
          throw new EventEnrollmentException(
            getObjectId() +
              ": enum change-of-state alarm value \"" + ordinal + '"' +
              " for property type \"" + propInfo + '"' +
              " must be greater than or equal to zero",
            new NErrorType(BBacnetErrorClass.PROPERTY, BBacnetErrorCode.OTHER));
        }
        listOfValues.addListElement(BBacnetPropertyStates.makeUnsigned(ordinal), null);
      }
    }
    else if (propInfo.getAsnType() == BacnetConst.ASN_INTEGER)
    {
      for (int ordinal : offnormalAlgorithm.getAlarmValues().getOrdinals())
      {
        listOfValues.addListElement(BBacnetPropertyStates.makeInteger(ordinal), null);
      }
    }
    else
    {
      throw new EventEnrollmentException(
        getObjectId() +
          ": enum change-of-state alarm extension is not supported for property type \"" + propInfo + '"' +
          " for object property reference: " + objectPropRef,
        new NErrorType(BBacnetErrorClass.PROPERTY, BBacnetErrorCode.OTHER));
    }

    return listOfValues;
  }

  private BBacnetListOf getListOfValues(BNumericChangeOfStateAlgorithm offnormalAlgorithm)
    throws EventEnrollmentException
  {
    BBacnetDeviceObjectPropertyReference objectPropRef = getObjectPropertyReference();
    PropertyInfo propInfo = findPropertyInfo(objectPropRef);

    BBacnetListOf listOfValues = new BBacnetListOf(BBacnetPropertyStates.TYPE);
    if (propInfo.getAsnType() == BacnetConst.ASN_UNSIGNED || isArraySize(propInfo, objectPropRef))
    {
      for (int ordinal : offnormalAlgorithm.getAlarmValues().getOrdinals())
      {
        if (ordinal < 0)
        {
          throw new EventEnrollmentException(
            getObjectId() +
              ": numeric change-of-state alarm value \"" + ordinal + '"' +
              " for property type \"" + propInfo + '"' +
              " must be greater than or equal to zero",
            new NErrorType(BBacnetErrorClass.PROPERTY, BBacnetErrorCode.OTHER));
        }
        listOfValues.addListElement(BBacnetPropertyStates.makeUnsigned(ordinal), null);
      }
    }
    else if (propInfo.getAsnType() == BacnetConst.ASN_INTEGER)
    {
      for (int ordinal : offnormalAlgorithm.getAlarmValues().getOrdinals())
      {
        listOfValues.addListElement(BBacnetPropertyStates.makeInteger(ordinal), null);
      }
    }
    else
    {
      throw new EventEnrollmentException(
        "Numeric change-of-state alarm extension is not supported for the property type: " + propInfo,
        new NErrorType(BBacnetErrorClass.PROPERTY, BBacnetErrorCode.OTHER));
    }

    return listOfValues;
  }

  private static BBacnetListOf getListOfValues(BStringChangeOfStateAlgorithm offnormalAlgorithm)
  {
    BBacnetListOf listOfValues = new BBacnetListOf(BString.TYPE);
    listOfValues.addListElement(offnormalAlgorithm.get(BStringChangeOfStateAlgorithm.expression), null);
    return listOfValues;
  }

  private static PropertyInfo findPropertyInfo(BBacnetDeviceObjectPropertyReference objectPropRef)
    throws EventEnrollmentException
  {
    int objectType = objectPropRef.getObjectId().getObjectType();
    int propId = objectPropRef.getPropertyId();
    PropertyInfo propInfo = getPropertyInfo(objectType, propId);
    if (propInfo == null)
    {
      throw new EventEnrollmentException(
        "Property info not found" +
          "; object type: " + BBacnetObjectType.tag(objectType) +
          ", property ID: " + BBacnetPropertyIdentifier.tag(propId),
        new NErrorType(BBacnetErrorClass.PROPERTY, BBacnetErrorCode.OTHER));
    }
    return propInfo;
  }

  private PropertyValue readEventTimeStamps(int ndx)
  {
    BPointExtension pointExt = (BPointExtension) getObject();

    BAbsTime lastOffnormalTime = BAbsTime.DEFAULT;
    BAbsTime lastFaultTime = BAbsTime.DEFAULT;
    BAbsTime lastToNormalTime = BAbsTime.DEFAULT;
    if (pointExt instanceof BAlarmSourceExt)
    {
      BAlarmSourceExt alarmExt = (BAlarmSourceExt) pointExt;
      lastOffnormalTime = alarmExt.getLastOffnormalTime();
      lastFaultTime = alarmExt.getLastFaultTime();
      lastToNormalTime = alarmExt.getLastToNormalTime();
    }
    else if (pointExt instanceof BBacnetTrendLogAlarmSourceExt)
    {
      BBacnetTrendLogAlarmSourceExt trendAlarmExt = (BBacnetTrendLogAlarmSourceExt) pointExt;
      BAlarmTimestamps toOffnormalTimes = trendAlarmExt.getToOffnormalTimes();
      BAlarmTimestamps toFaultTimes = trendAlarmExt.getToFaultTimes();

      lastOffnormalTime = toOffnormalTimes.getAlarmTime();
      lastFaultTime = toFaultTimes.getAlarmTime();

      lastToNormalTime = toOffnormalTimes.getNormalTime();
      if (lastToNormalTime.isBefore(toFaultTimes.getNormalTime()))
      {
        lastToNormalTime = toFaultTimes.getNormalTime();
      }
    }

    return readEventTimeStamps(lastOffnormalTime, lastFaultTime, lastToNormalTime, ndx);
  }

  private BBacnetReliability readReliability()
  {
    BBacnetReliability eventEnrollmentReliability = getReliability();
    if (!eventEnrollmentReliability.equals(BBacnetReliability.noFaultDetected))
    {
      return eventEnrollmentReliability;
    }

    BPointExtension target = (BPointExtension) getObject();
    BComplex parent = target != null ? target.getParent() : null;
    if (parent == null)
    {
      if (logger.isLoggable(Level.FINE))
      {
        logger.fine(getObjectId() + ": reliability set to configurationError because target could not be resolved");
      }
      // TODO Add support for internal reliability evaluation
      //setReliability(BBacnetReliability.configurationError);
      return BBacnetReliability.configurationError;
    }

    if (!(parent instanceof BIStatus))
    {
      if (logger.isLoggable(Level.FINE))
      {
        logger.fine(getObjectId() + ": resolved target is type " + parent.getType()
          + " and not instanceof BIStatus; returning noFaultDetected as reliability value");
      }
      return BBacnetReliability.noFaultDetected;
    }

    BStatus parentStatus = ((BIStatus) parent).getStatus();
    if (parentStatus.isNull())
    {
      return BBacnetReliability.unreliableOther;
    }

    if (parentStatus.isDown() || parentStatus.isStale())
    {
      return BBacnetReliability.communicationFailure;
    }

    if (parentStatus.isFault())
    {
      return BBacnetReliability.monitoredObjectFault;
    }

    // TODO Include fault algorithm?

    return BBacnetReliability.noFaultDetected;
  }

  private PropertyValue readEventAlgorithmInhibit()
  {
    updateEventAlgorithmInhibitInfo((BPointExtension) getObject());
    return new NReadPropertyResult(
      BBacnetPropertyIdentifier.EVENT_ALGORITHM_INHIBIT,
      BacnetConst.NOT_USED,
      AsnUtil.toAsnBoolean(eventAlgorithmInhibit));
  }

  private PropertyValue readEventAlgorithmInhibitRef()
  {
    updateEventAlgorithmInhibitInfo((BPointExtension) getObject());
    return new NReadPropertyResult(
      BBacnetPropertyIdentifier.EVENT_ALGORITHM_INHIBIT_REF,
      BacnetConst.NOT_USED,
      AsnUtil.toAsn(BacnetConst.ASN_ANY, eventAlgorithmInhibitRef));
  }

  private void updateEventAlgorithmInhibitInfo(BPointExtension pointExt)
  {
    if (pointExt instanceof BAlarmSourceExt)
    {
      eventAlgorithmInhibit = ((BAlarmSourceExt) pointExt).getAlarmInhibit().getBoolean();
      updateAlarmInhibitRef(pointExt.getLinks(BAlarmSourceExt.alarmInhibit));
    }
    else if (pointExt instanceof BBacnetTrendLogAlarmSourceExt)
    {
      eventAlgorithmInhibit = ((BBacnetTrendLogAlarmSourceExt) pointExt).getAlarmInhibit().getBoolean();
      updateAlarmInhibitRef(pointExt.getLinks(BBacnetTrendLogAlarmSourceExt.alarmInhibit));
    }
  }

  private void updateAlarmInhibitRef(BLink[] links)
  {
    for (BLink link : links)
    {
      if (!link.isActive() || !link.isEnabled())
      {
        continue;
      }

      BComponent source = link.getSourceComponent();
      if (!(source instanceof BBooleanPoint))
      {
        continue;
      }

      BIBacnetExportObject descriptor = findDescriptor(source.getHandleOrd());
      if (descriptor != null)
      {
        BBacnetObjectPropertyReference newValue = new BBacnetObjectPropertyReference(descriptor.getObjectId());
        if (logger.isLoggable(Level.FINE) && !eventAlgorithmInhibitRef.getObjectId().isConfigured())
        {
          logger.fine(getObjectId() + ": updating unconfigured eventAlgorithmInhibitRef because" +
            " there is a valid link to alarmInhibitRef; new value: " + newValue);
        }
        eventAlgorithmInhibitRef = newValue;
        return;
      }
    }

    if (logger.isLoggable(Level.FINE) && eventAlgorithmInhibitRef.getObjectId().isConfigured())
    {
      logger.fine(getObjectId() + ": setting eventAlgorithmInhibitRef as unconfigured because there" +
        " are no valid links to alarmInhibitRef; old value: " + eventAlgorithmInhibitRef);
    }
    eventAlgorithmInhibitRef = OBJECT_PROP_REF_UNCONFIGURED;
  }

  private PropertyValue readTimeDelayNormal()
  {
    updateTimeDelayToNormal((BPointExtension) getObject());
    return new NReadPropertyResult(
      BBacnetPropertyIdentifier.TIME_DELAY_NORMAL,
      AsnUtil.toAsnUnsigned(timeDelayNormal));
  }

  private void updateTimeDelayToNormal(BPointExtension pointExt)
  {
    if (pointExt instanceof BAlarmSourceExt)
    {
      timeDelayNormal = ((BAlarmSourceExt) pointExt).getTimeDelayToNormal().getSeconds();
    }
  }

  //endregion

  //region Write Property

  @SuppressWarnings("unused")
  protected ErrorType writeProperty(int pId, int ndx, byte[] val, int pri)
  {
    if (ndx >= 0)
    {
      if (!isArray(pId))
      {
        return new NErrorType(BBacnetErrorClass.PROPERTY, BBacnetErrorCode.PROPERTY_IS_NOT_AN_ARRAY);
      }
    }
    else if (ndx < NOT_USED)
    {
      return new NErrorType(BBacnetErrorClass.PROPERTY, BBacnetErrorCode.INVALID_ARRAY_INDEX);
    }

    try
    {
      switch (pId)
      {
        case BBacnetPropertyIdentifier.OBJECT_IDENTIFIER:
        case BBacnetPropertyIdentifier.OBJECT_TYPE:
        case BBacnetPropertyIdentifier.EVENT_TYPE:
        case BBacnetPropertyIdentifier.EVENT_STATE:
        case BBacnetPropertyIdentifier.ACKED_TRANSITIONS:
        case BBacnetPropertyIdentifier.EVENT_TIME_STAMPS:
        case BBacnetPropertyIdentifier.EVENT_MESSAGE_TEXTS:
        case BBacnetPropertyIdentifier.STATUS_FLAGS:
        case BBacnetPropertyIdentifier.RELIABILITY:
        case BBacnetPropertyIdentifier.PROPERTY_LIST:
          if (logger.isLoggable(Level.FINE))
          {
            logger.fine(getObjectId() + ": attempted to write read-only property " + BBacnetPropertyIdentifier.tag(pId));
          }
          return new NErrorType(BBacnetErrorClass.PROPERTY, BBacnetErrorCode.WRITE_ACCESS_DENIED);

        case BBacnetPropertyIdentifier.OBJECT_NAME:
          return BacUtil.setObjectName(this, objectName, val);
        case BBacnetPropertyIdentifier.DESCRIPTION:
          setString(description, AsnUtil.fromAsnCharacterString(val), BLocalBacnetDevice.getBacnetContext());
          return null;

        case BBacnetPropertyIdentifier.NOTIFY_TYPE:
          return writeNotifyType(val);
        case BBacnetPropertyIdentifier.EVENT_PARAMETERS:
          return writeEventParameters(val);
        case BBacnetPropertyIdentifier.OBJECT_PROPERTY_REFERENCE:
          return writeObjectPropertyReference(val);
        case BBacnetPropertyIdentifier.EVENT_ENABLE:
          return writeEventEnable(val);
        case BBacnetPropertyIdentifier.NOTIFICATION_CLASS:
          return writeNotificationClass(val);
        case BBacnetPropertyIdentifier.EVENT_DETECTION_ENABLE:
          setBoolean(BBacnetEventSource.eventDetectionEnable, AsnUtil.fromAsnBoolean(val), BLocalBacnetDevice.getBacnetContext());
          return null;
        case BBacnetPropertyIdentifier.EVENT_MESSAGE_TEXTS_CONFIG:
          return writeMessageTextsConfig(ndx, val);
        case BBacnetPropertyIdentifier.EVENT_ALGORITHM_INHIBIT:
          return writeEventAlgorithmInhibit(val);
        case BBacnetPropertyIdentifier.EVENT_ALGORITHM_INHIBIT_REF:
          return writeEventAlgorithmInhibitRef(val);
        case BBacnetPropertyIdentifier.TIME_DELAY_NORMAL:
          return writeTimeDelayNormal(val);
        default:
          if (logger.isLoggable(Level.FINE))
          {
            logger.fine(getObjectId() + ": unknown property: " + BBacnetPropertyIdentifier.tag(pId));
          }
          // Property not found
          return new NErrorType(BBacnetErrorClass.PROPERTY, BBacnetErrorCode.UNKNOWN_PROPERTY);
      }
    }
    catch (OutOfRangeException e)
    {
      logException(
        Level.INFO,
        new StringBuilder(getObjectId().toString())
          .append(": OutOfRangeException writing property ")
          .append(BBacnetPropertyIdentifier.tag(pId)),
        e);
      return new NErrorType(BBacnetErrorClass.PROPERTY, BBacnetErrorCode.VALUE_OUT_OF_RANGE);
    }
    catch (AsnException e)
    {
      logException(
        Level.INFO,
        new StringBuilder(getObjectId().toString())
          .append(": AsnException writing property ")
          .append(BBacnetPropertyIdentifier.tag(pId)),
        e);
      return new NErrorType(BBacnetErrorClass.PROPERTY, BBacnetErrorCode.INVALID_DATA_TYPE);
    }
    catch (PermissionException e)
    {
      logException(
        Level.INFO,
        new StringBuilder(getObjectId().toString())
          .append(": PermissionException writing property ")
          .append(BBacnetPropertyIdentifier.tag(pId)),
        e);
      return new NErrorType(BBacnetErrorClass.PROPERTY, BBacnetErrorCode.WRITE_ACCESS_DENIED);
    }
  }

  private ErrorType writeNotifyType(byte[] val)
    throws AsnException
  {
    BBacnetNotifyType newValue = BBacnetNotifyType.make(AsnUtil.fromAsnEnumerated(val));
    if (!newValue.equals(BBacnetNotifyType.alarm) && !newValue.equals(BBacnetNotifyType.event))
    {
      if (logger.isLoggable(Level.INFO))
      {
        logger.info(getObjectId() + ": attempt to write invalid notifyType: " + newValue);
      }
      return new NErrorType(property, valueOutOfRange);
    }
    set(notifyType, newValue, BLocalBacnetDevice.getBacnetContext());
    return null;
  }

  private ErrorType writeEventParameters(byte[] val) throws AsnException
  {
    BBacnetEventParameter eventParam = new BBacnetEventParameter();
    AsnUtil.fromAsn(BacnetConst.ASN_ANY, val, eventParam);

    Context bacnetContext = BLocalBacnetDevice.getBacnetContext();
    bacnetContext.getUser().checkWrite(this, eventParameter);
    bacnetContext.getUser().checkWrite(this, typeOfEvent);
    checkEventType(eventParam.getChoice());

    ErrorType error = writeEventParameters(eventParam);
    if (error != null)
    {
      return error;
    }

    set(eventParameter, eventParam, bacnetContext);
    set(typeOfEvent, BBacnetEventType.make(eventParam.getChoice()), bacnetContext);
    return null;
  }

  private void checkEventType(int eventType)
    throws OutOfRangeException
  {
    // These event types are not supported at all. Other types may not be supported based on the
    // referenced object- that is checked in configureExt.
    switch (eventType)
    {
      case BBacnetEventType.CHANGE_OF_STATE:
      case BBacnetEventType.COMMAND_FAILURE:
      case BBacnetEventType.FLOATING_LIMIT:
      case BBacnetEventType.OUT_OF_RANGE:
      case BBacnetEventType.DOUBLE_OUT_OF_RANGE:
      case BBacnetEventType.SIGNED_OUT_OF_RANGE:
      case BBacnetEventType.UNSIGNED_OUT_OF_RANGE:
      case BBacnetEventType.BUFFER_READY:
      case BBacnetEventType.CHANGE_OF_CHARACTERSTRING:
      case BBacnetEventType.NONE:
      case BBacnetEventType.CHANGE_OF_DISCRETE_VALUE:
        return;

      case BBacnetEventType.CHANGE_OF_BITSTRING:
      case BBacnetEventType.CHANGE_OF_VALUE:
      case BBacnetEventType.COMPLEX_EVENT_TYPE:
      case BBacnetEventType.BUFFER_READY_DEPRECATED:
      case BBacnetEventType.CHANGE_OF_LIFE_SAFETY:
      case BBacnetEventType.EXTENDED:
      case BBacnetEventType.UNSIGNED_RANGE:
      case BBacnetEventType.RESERVED:
      case BBacnetEventType.ACCESS_EVENT:
      case BBacnetEventType.CHANGE_OF_STATUS_FLAGS:
      case BBacnetEventType.CHANGE_OF_RELIABILITY:
      default:
        if (logger.isLoggable(Level.FINE))
        {
          logger.fine(getObjectId() + ": event type " + BBacnetEventType.tag(eventType) + " is not supported");
        }
        throw new OutOfRangeException("event type " + BBacnetEventType.tag(eventType) + " is not supported");
    }
  }

  private ErrorType writeEventParameters(BBacnetEventParameter eventParam)
  {
    // getObject will validate that the eventEnrollmentOrd and objectPropertyReference properties
    // are consistent. If not, ext will be null and recreated if there is a target. Keep getObject
    // ahead of resolveTarget.
    BPointExtension pointExt = (BPointExtension) getObject();

    BBacnetDeviceObjectPropertyReference objPropRef = getObjectPropertyReference();
    BComponent target;
    if (pointExt != null)
    {
      // getTarget is called within getObject to verify pointExt's parent matches the target of the
      // objectPropertyReference.
      target = (BComponent) pointExt.getParent();

      // Some properties are outside eventParameters; get their latest values off the pointExt so
      // that they'll be transferred to the updated ext.
      getEventEnable(pointExt);
      getNotificationClass(pointExt);
      updateEventMessageTextsConfig(pointExt);
      updateEventAlgorithmInhibitInfo(pointExt);
      updateTimeDelayToNormal(pointExt);
    }
    else
    {
      // The pointExt has never been configured or was invalid; a new one is being created.
      target = resolveTarget(objPropRef);
      if (target == null)
      {
        if (logger.isLoggable(Level.FINE))
        {
          logger.fine(getObjectId() + ": eventParameters BACnet property (" + eventParam +
            ") was written but a target could not be resolved based on the objectPropertyReference: " +
            objPropRef);
        }
        return null;
      }
    }

    ErrorType error = configureExt(eventParam, objPropRef, pointExt, target);
    if (error != null)
    {
      if (logger.isLoggable(Level.FINE))
      {
        logger.fine(getObjectId() + ": error configuring alarm ext while writing EventParameters property: " + eventParam);
      }
    }
    return error;
  }

  private ErrorType writeObjectPropertyReference(byte[] val) throws AsnException
  {
    BBacnetDeviceObjectPropertyReference objPropRef = new BBacnetDeviceObjectPropertyReference();
    AsnUtil.fromAsn(BacnetConst.ASN_ANY, val, objPropRef);

    BComponent newTarget = resolveTarget(objPropRef);
    if (newTarget == null)
    {
      if (logger.isLoggable(Level.FINE))
      {
        logger.fine(getObjectId() + ": could not find a target when writing BACnet objectPropertyReference property: " + objPropRef);
      }
      return new NErrorType(BBacnetErrorClass.PROPERTY, BBacnetErrorCode.VALUE_OUT_OF_RANGE);
    }

    BPointExtension pointExt = (BPointExtension) getObject();
    if (pointExt != null)
    {
      BComponent oldTarget = (BComponent) pointExt.getParent();
      if (oldTarget == newTarget)
      {
        if (logger.isLoggable(Level.FINE))
        {
          logger.fine(getObjectId() + ": BACnet write of Object_Property_Reference points to existing extension's parent: " + objPropRef);
        }
        return null;
      }

      if (logger.isLoggable(Level.FINE))
      {
        logger.fine(getObjectId() + ": BACnet changing Object_Property_Reference to " + objPropRef);
      }

      updateEventParameters(pointExt);

      // Some properties are outside eventParameters; get their latest values off the pointExt so
      // that they'll be transferred to the new ext.
      getEventEnable(pointExt);
      getNotificationClass(pointExt);
      updateEventMessageTextsConfig(pointExt);
      updateEventAlgorithmInhibitInfo(pointExt);
      updateTimeDelayToNormal(pointExt);
    }
    else
    {
      if (logger.isLoggable(Level.FINE))
      {
        logger.fine(getObjectId() + ": BACnet writing Object_Property_Reference when extension is not yet configured: " + objPropRef);
      }
    }

    // The target point is changing so any current alarm ext will need to be re-created.
    resetDescriptor();

    ErrorType error = configureExt(getEventParameter(), objPropRef, null, newTarget);
    if (error != null)
    {
      if (logger.isLoggable(Level.FINE))
      {
        logger.fine(getObjectId() + ": error configuring alarm ext when writing BACnet" +
          " objectPropertyReference property: " + objPropRef);
      }
    }
    else
    {
      if (isLocalDevice(objPropRef.getDeviceId().getInstanceNumber()))
      {
        // If the reference points at this device, indicate that by using the default device ID.
        objPropRef.setDeviceId(BBacnetObjectIdentifier.DEFAULT_DEVICE);
      }
      setObjectPropertyReference(objPropRef);
    }

    return error;
  }

  private ErrorType writeNotificationClass(byte[] val) throws AsnException
  {
    int instanceNum = AsnUtil.fromAsnUnsignedInt(val);
    Context context = BLocalBacnetDevice.getBacnetContext();
    ErrorType error = configureAlarmClass((BPointExtension) getObject(), instanceNum, context);
    if (error == null)
    {
      setInt(notificationClassId, instanceNum, context);
    }
    else
    {
      if (logger.isLoggable(Level.FINE))
      {
        logger.fine(getObjectId() + ": error in writeNotificationClass");
      }
    }
    return error;
  }

  private static BBacnetNotificationClassDescriptor lookupNotificationClass(int instanceNum)
  {
    BBacnetObjectIdentifier id = BBacnetObjectIdentifier.make(BBacnetObjectType.NOTIFICATION_CLASS, instanceNum);
    return (BBacnetNotificationClassDescriptor) BBacnetNetwork.localDevice().lookupBacnetObject(id);
  }

  private ErrorType writeEventEnable(byte[] val) throws AsnException
  {
    BBacnetBitString eventEnableBits = AsnUtil.fromAsnBitString(val);
    BAlarmTransitionBits alarmEnable = BacnetBitStringUtil.getBAlarmTransitionBits(eventEnableBits);
    if (alarmEnable == null)
    {
      if (logger.isLoggable(Level.FINE))
      {
        logger.fine(getObjectId() + ": could not write the eventEnable property because" +
          " the alarm transition bits could not be retrieved for value " + eventEnableBits);
      }
      return new NErrorType(BBacnetErrorClass.PROPERTY, BBacnetErrorCode.VALUE_OUT_OF_RANGE);
    }

    eventEnable = alarmEnable;

    BPointExtension pointExt = (BPointExtension) getObject();
    if (pointExt instanceof BAlarmSourceExt)
    {
      pointExt.set(BAlarmSourceExt.alarmEnable, alarmEnable, BLocalBacnetDevice.getBacnetContext());
    }
    else if (pointExt instanceof BBacnetTrendLogAlarmSourceExt)
    {
      pointExt.set(BBacnetTrendLogAlarmSourceExt.alarmEnable, alarmEnable, BLocalBacnetDevice.getBacnetContext());
    }
    else
    {
      if (logger.isLoggable(Level.FINE))
      {
        logger.fine(getObjectId() + ": could not write the eventEnable property because the"
          + " associated point ext is not set or not an AlarmSourceExt or BacnetTrendLogAlarmSourceExt");
      }
    }

    return null;
  }

  private ErrorType writeMessageTextsConfig(int ndx, byte[] val)
    throws AsnException
  {
    if (ndx < NOT_USED || ndx > MESSAGE_TEXTS_COUNT)
    {
      return new NErrorType(BBacnetErrorClass.PROPERTY, BBacnetErrorCode.INVALID_ARRAY_INDEX);
    }

    if (ndx == 0)
    {
      // According to BTL Specified Tests-23.1_final, 9.22.2.X2 Resizing a writable fixed size array
      // property: writing a value greater or less than the array size to index zero may return
      // INVALID_ARRAY_INDEX, VALUE_OUT_OF_RANGE, or WRITE_ACCESS_DENIED. Zero is not an invalid
      // array index for reading so receiving this error code when writing might be confusing. The
      // value written at index zero (unless it is three) is technically out-of-range for this
      // fixed-size array but that could imply there is a value that is in-range. The size of the
      // array is not writable, therfore write access denied is returned.
      return new NErrorType(BBacnetErrorClass.PROPERTY, BBacnetErrorCode.WRITE_ACCESS_DENIED);
    }

    switch (ndx)
    {
      case NOT_USED:
        BBacnetArray textsConfig = new BBacnetArray(BString.TYPE, 3);
        AsnUtil.fromAsn(BacnetConst.ASN_ANY, val, textsConfig);
        toOffnormalText = textsConfig.getElement(1).toString(null);
        toFaultText = textsConfig.getElement(2).toString(null);
        toNormalText = textsConfig.getElement(3).toString(null);
        break;

      case 1:
        toOffnormalText = AsnUtil.fromAsnCharacterString(val);
        break;

      case 2:
        toFaultText = AsnUtil.fromAsnCharacterString(val);
        break;

      case 3:
        toNormalText = AsnUtil.fromAsnCharacterString(val);
        break;
    }

    BPointExtension pointExt = (BPointExtension) getObject();
    if (pointExt instanceof BAlarmSourceExt)
    {
      BAlarmSourceExt alarmExt = (BAlarmSourceExt) pointExt;
      Context context = BLocalBacnetDevice.getBacnetContext();
      switch (ndx)
      {
        case NOT_USED:
          alarmExt.set(
            BAlarmSourceExt.toOffnormalText,
            BFormat.make(toOffnormalText),
            context);
          alarmExt.set(
            BAlarmSourceExt.toFaultText,
            BFormat.make(toFaultText),
            context);
          alarmExt.set(
            BAlarmSourceExt.toNormalText,
            BFormat.make(toNormalText),
            context);
          resetOutOfRangeTexts(alarmExt);
          break;

        case 1:
          alarmExt.set(
            BAlarmSourceExt.toOffnormalText,
            BFormat.make(toOffnormalText),
            context);
          resetOutOfRangeTexts(alarmExt);
          break;

        case 2:
          alarmExt.set(
            BAlarmSourceExt.toFaultText,
            BFormat.make(toFaultText),
            context);
          break;

        case 3:
          alarmExt.set(
            BAlarmSourceExt.toNormalText,
            BFormat.make(toNormalText),
            context);
          break;
      }
    }
    else if (pointExt instanceof BBacnetTrendLogAlarmSourceExt)
    {
      BBacnetTrendLogAlarmSourceExt trendAlarmExt = (BBacnetTrendLogAlarmSourceExt) pointExt;
      switch (ndx)
      {
        case NOT_USED:
          if (!toOffnormalText.isEmpty() || !toFaultText.isEmpty())
          {
            return new NErrorType(BBacnetErrorClass.PROPERTY, BBacnetErrorCode.VALUE_OUT_OF_RANGE);
          }

          // BBacnetTrendLogAlarmSourceExt does not have properties for toOffnormal and toFault texts
          trendAlarmExt.set(
            BBacnetTrendLogAlarmSourceExt.toNormalText,
            BFormat.make(toNormalText),
            BLocalBacnetDevice.getBacnetContext());
          break;

        case 1:
          if (!toOffnormalText.isEmpty())
          {
            return new NErrorType(BBacnetErrorClass.PROPERTY, BBacnetErrorCode.VALUE_OUT_OF_RANGE);
          }
          break;

        case 2:
          if (!toFaultText.isEmpty())
          {
            return new NErrorType(BBacnetErrorClass.PROPERTY, BBacnetErrorCode.VALUE_OUT_OF_RANGE);
          }
          break;

        case 3:
          trendAlarmExt.set(
            BBacnetTrendLogAlarmSourceExt.toNormalText,
            BFormat.make(toNormalText),
            BLocalBacnetDevice.getBacnetContext());
          break;
      }
    }
    else
    {
      if (logger.isLoggable(Level.FINE))
      {
        logger.fine(getObjectId() + ": could not write the eventMessageTextsConfig property because"
          + " the associated point ext is not set or not an AlarmSourceExt or BacnetTrendLogAlarmSourceExt");
      }
    }

    return null;
  }

  private ErrorType writeEventAlgorithmInhibit(byte[] val)
    throws AsnException
  {
    boolean newValue = AsnUtil.fromAsnBoolean(val);

    if (!getEventDetectionEnable())
    {
      if (logger.isLoggable(Level.FINE))
      {
        logger.fine(getObjectId() + ": could not write the alarmInhibit property because event detection is disabled");
      }
      return new NErrorType(BBacnetErrorClass.PROPERTY, BBacnetErrorCode.WRITE_ACCESS_DENIED);
    }

    BPointExtension pointExt = (BPointExtension) getObject();
    if (pointExt != null)
    {
      BLink[] alarmInhibitLinks = getAlarmInhibitLinks(pointExt);
      if (alarmInhibitLinks.length > 0)
      {
        if (logger.isLoggable(Level.FINE))
        {
          logger.fine(getObjectId() + ": could not write the eventAlgorithmInhibit property because alarmInhibit is linked");
        }
        return new NErrorType(BBacnetErrorClass.PROPERTY, BBacnetErrorCode.WRITE_ACCESS_DENIED);
      }

      if (pointExt instanceof BAlarmSourceExt)
      {
        pointExt.set(BAlarmSourceExt.alarmInhibit, new BStatusBoolean(newValue), BLocalBacnetDevice.getBacnetContext());
      }
      else if (pointExt instanceof BBacnetTrendLogAlarmSourceExt)
      {
        pointExt.set(BBacnetTrendLogAlarmSourceExt.alarmInhibit, new BStatusBoolean(newValue), BLocalBacnetDevice.getBacnetContext());
      }
    }
    else
    {
      if (eventAlgorithmInhibitRef.getObjectId().isConfigured())
      {
        if (logger.isLoggable(Level.FINE))
        {
          logger.fine(getObjectId() + ": could not write the eventAlgorithmInhibit property because" +
            " eventAlgorithmInhibitRef (" + eventAlgorithmInhibitRef + ") is configured");
        }
        return new NErrorType(BBacnetErrorClass.PROPERTY, BBacnetErrorCode.WRITE_ACCESS_DENIED);
      }
    }

    eventAlgorithmInhibit = newValue;
    return null;
  }

  private static BLink[] getAlarmInhibitLinks(BPointExtension pointExt)
  {
    if (pointExt instanceof BAlarmSourceExt)
    {
      return pointExt.getLinks(BAlarmSourceExt.alarmInhibit);
    }
    else if (pointExt instanceof BBacnetTrendLogAlarmSourceExt)
    {
      return pointExt.getLinks(BBacnetTrendLogAlarmSourceExt.alarmInhibit);
    }
    else
    {
      return EMPTY_LINKS_ARRAY;
    }
  }

  private ErrorType writeEventAlgorithmInhibitRef(byte[] val)
    throws AsnException
  {
    BBacnetObjectPropertyReference newObjPropRef = new BBacnetObjectPropertyReference();
    AsnUtil.fromAsn(val, newObjPropRef);

    BPointExtension pointExt = (BPointExtension) getObject();
    Context context = BLocalBacnetDevice.getBacnetContext();

    if (!newObjPropRef.getObjectId().isConfigured())
    {
      eventAlgorithmInhibitRef = newObjPropRef;
      removeAlarmInhibitLinks(pointExt, context);
      return null;
    }

    BBooleanPoint sourcePoint;
    try
    {
      sourcePoint = findEventAlgorithmInhibitSourcePoint(newObjPropRef);
    }
    catch (ErrorException e)
    {
      return e.getErrorType();
    }

    checkLinkPermissions(sourcePoint, "out", context);

    if (pointExt instanceof BAlarmSourceExt)
    {
      replaceLinks(pointExt, BAlarmSourceExt.alarmInhibit, sourcePoint, context);
    }
    else if (pointExt instanceof BBacnetTrendLogAlarmSourceExt)
    {
      replaceLinks(pointExt, BBacnetTrendLogAlarmSourceExt.alarmInhibit, sourcePoint, context);
    }

    eventAlgorithmInhibitRef = newObjPropRef;
    return null;
  }

  private void removeAlarmInhibitLinks(BPointExtension pointExt, Context context)
  {
    if (pointExt != null)
    {
      if (logger.isLoggable(Level.FINE))
      {
        logger.fine(getObjectId() + ": removing links to alarmInhibit because" +
          " eventAlgorithmInhibitRef is set to the unconfigured instance number 4194303");
      }

      for (BLink link : getAlarmInhibitLinks(pointExt))
      {
        pointExt.remove(link.getPropertyInParent(), context);
      }
    }
  }

  private ErrorType writeTimeDelayNormal(byte[] val)
    throws AsnException
  {
    int newValue = AsnUtil.fromAsnUnsignedInt(val);

    BPointExtension pointExt = (BPointExtension)getObject();
    if (pointExt instanceof BAlarmSourceExt)
    {
      pointExt.set(BAlarmSourceExt.timeDelayToNormal, BRelTime.makeSeconds(newValue), BLocalBacnetDevice.getBacnetContext());
    }

    timeDelayNormal = newValue;
    return null;
  }

  //endregion

  //region ConfigureExt

  private ErrorType configureExt(
    BBacnetEventParameter eventParam,
    BBacnetDeviceObjectPropertyReference objPropRef,
    BPointExtension pointExt,
    BComponent target)
  {
    try
    {
      int eventType = eventParam.getChoice();
      switch (eventType)
      {
        case BBacnetEventType.CHANGE_OF_STATE:
          configureChangeOfStateExt(eventParam, objPropRef, pointExt, target);
          break;
        case BBacnetEventType.COMMAND_FAILURE:
          configureCommandFailureExt(eventParam, pointExt, target);
          break;
        case BBacnetEventType.FLOATING_LIMIT:
          configureFloatingLimitExt(eventParam, pointExt, target);
          break;
        case BBacnetEventType.OUT_OF_RANGE:
        case BBacnetEventType.DOUBLE_OUT_OF_RANGE:
        case BBacnetEventType.SIGNED_OUT_OF_RANGE:
        case BBacnetEventType.UNSIGNED_OUT_OF_RANGE:
          configureOutOfRangeExt(eventParam, objPropRef, pointExt, target);
          break;
        case BBacnetEventType.BUFFER_READY:
          configureTrendAlarmExt(eventParam, pointExt, target);
          break;
        case BBacnetEventType.CHANGE_OF_CHARACTERSTRING:
          configureStringChangeOfStateExt(eventParam, pointExt, target);
          break;
        case BBacnetEventType.NONE:
          configureNoneExt();
          break;
        case BBacnetEventType.CHANGE_OF_DISCRETE_VALUE:
          configureChangeOfDiscreteValueExt(eventParam, objPropRef, pointExt, target);
          break;

        case BBacnetEventType.CHANGE_OF_BITSTRING:
        case BBacnetEventType.CHANGE_OF_VALUE:
        case BBacnetEventType.COMPLEX_EVENT_TYPE:
        case BBacnetEventType.BUFFER_READY_DEPRECATED:
        case BBacnetEventType.CHANGE_OF_LIFE_SAFETY:
        case BBacnetEventType.EXTENDED:
        case BBacnetEventType.UNSIGNED_RANGE:
        case BBacnetEventType.RESERVED:
        case BBacnetEventType.ACCESS_EVENT:
        case BBacnetEventType.CHANGE_OF_STATUS_FLAGS:
        case BBacnetEventType.CHANGE_OF_RELIABILITY:
        default:
          throw new EventEnrollmentException(
            "event type " + BBacnetEventType.tag(eventType) + " is not supported",
            new NErrorType(BBacnetErrorClass.PROPERTY, BBacnetErrorCode.OPTIONAL_FUNCTIONALITY_NOT_SUPPORTED));
      }

      return null;
    }
    catch (EventEnrollmentException e)
    {
      if (logger.isLoggable(Level.FINE))
      {
        logger.fine(getObjectId() + ": error configuring alarm ext; message: " + e.getMessage());
      }
      resetDescriptor();
      return e.errorType;
    }
    catch (PermissionException e)
    {
      logException(
        Level.INFO,
        new StringBuilder(getObjectId().toString()).append(": permission exception configuring alarm ext"),
        e);
      resetDescriptor();
      return new NErrorType(BBacnetErrorClass.PROPERTY, BBacnetErrorCode.WRITE_ACCESS_DENIED);
    }
    catch (Exception e)
    {
      logException(
        Level.INFO,
        new StringBuilder(getObjectId().toString()).append(": unexpected error configuring alarm ext"),
        e);
      resetDescriptor();
      return new NErrorType(BBacnetErrorClass.PROPERTY, BBacnetErrorCode.OTHER);
    }
  }

  private void configureAlarmExt(BBacnetEventParameter eventParam, BAlarmSourceExt alarmExt)
  {
    Context context = BLocalBacnetDevice.getBacnetContext();
    configureTimeDelays(eventParam, alarmExt, context);
    alarmExt.set(BAlarmSourceExt.alarmEnable, eventEnable, context);
    configureAlarmClass(alarmExt, getNotificationClassId(), context);
    alarmExt.set(BAlarmSourceExt.toOffnormalText, BFormat.make(toOffnormalText), context);
    alarmExt.set(BAlarmSourceExt.toFaultText, BFormat.make(toFaultText), context);
    alarmExt.set(BAlarmSourceExt.toNormalText, BFormat.make(toNormalText), context);
    configureAlarmInhibit(alarmExt, context);
  }

  private void configureTimeDelays(BBacnetEventParameter eventParam, BAlarmSourceExt ext, Context context)
  {
    BRelTime timeDelay = BRelTime.makeSeconds(((BBacnetUnsigned) eventParam.get(BBacnetEventParameter.TIME_DELAY_SLOT_NAME)).getInt());
    ext.set(BAlarmSourceExt.timeDelay, timeDelay, context);
    ext.set(BAlarmSourceExt.timeDelayToNormal, BRelTime.makeSeconds(timeDelayNormal), context);
  }

  private ErrorType configureAlarmClass(BPointExtension ext, int instanceNum, Context context)
  {
    if (instanceNum < 0 || instanceNum > BBacnetObjectIdentifier.UNCONFIGURED_INSTANCE_NUMBER)
    {
      if (logger.isLoggable(Level.FINE))
      {
        logger.fine(getObjectId() + ": configureAlarmClass:" +
          " notification class instance number " + instanceNum + " exceeds the maximum allowable instance number value");
      }
      return new NErrorType(BBacnetErrorClass.PROPERTY, BBacnetErrorCode.VALUE_OUT_OF_RANGE);
    }

    String alarmClassName;
    if (instanceNum == BBacnetObjectIdentifier.UNCONFIGURED_INSTANCE_NUMBER)
    {
      alarmClassName = "";
    }
    else
    {
      BBacnetNotificationClassDescriptor descriptor = lookupNotificationClass(instanceNum);
      if (descriptor == null)
      {
        if (logger.isLoggable(Level.FINE))
        {
          logger.fine(getObjectId() + ": configureAlarmClass:" +
            " cannot find descriptor for notification class instance number " + instanceNum);
        }
        return new NErrorType(BBacnetErrorClass.PROPERTY, BBacnetErrorCode.VALUE_OUT_OF_RANGE);
      }

      BAlarmClass alarmClass = descriptor.getAlarmClass();
      if (alarmClass == null)
      {
        if (logger.isLoggable(Level.FINE))
        {
          logger.fine(getObjectId() + ": configureAlarmClass:" +
            " descriptor for notification class instance number " + instanceNum + " could not resolve its alarm class");
        }
        return new NErrorType(BBacnetErrorClass.PROPERTY, BBacnetErrorCode.OTHER);
      }

      alarmClassName = alarmClass.getName();
    }

    if (ext instanceof BAlarmSourceExt)
    {
      ext.setString(BAlarmSourceExt.alarmClass, alarmClassName, context);
    }
    else if (ext instanceof BBacnetTrendLogAlarmSourceExt)
    {
      ext.setString(BBacnetTrendLogAlarmSourceExt.alarmClass, alarmClassName, context);
    }
    else
    {
      if (logger.isLoggable(Level.FINE))
      {
        logger.fine(getObjectId() + ": configureAlarmClass:" +
          " no associated point ext on which to update the alarm class based on" +
          " notification-class instance number " + instanceNum);
      }
    }

    return null;
  }

  private void configureAlarmInhibit(BPointExtension ext, Context context)
  {
    boolean isLinked = addAlarmInhibitLink(ext, context);
    if (!isLinked)
    {
      if (ext instanceof BAlarmSourceExt)
      {
        ((BAlarmSourceExt) ext).setAlarmInhibit(new BStatusBoolean(eventAlgorithmInhibit));
      }
      else if (ext instanceof BBacnetTrendLogAlarmSourceExt)
      {
        ((BBacnetTrendLogAlarmSourceExt) ext).setAlarmInhibit(new BStatusBoolean(eventAlgorithmInhibit));
      }
    }
  }

  private boolean addAlarmInhibitLink(BPointExtension ext, Context context)
  {
    if (!eventAlgorithmInhibitRef.getObjectId().isConfigured())
    {
      removeAlarmInhibitLinks(ext, context);
      return false;
    }

    BBooleanPoint sourcePoint;
    try
    {
      sourcePoint = findEventAlgorithmInhibitSourcePoint(eventAlgorithmInhibitRef);
      checkLinkPermissions(sourcePoint, "out", context);

      if (ext instanceof BAlarmSourceExt)
      {
        replaceLinks(ext, BAlarmSourceExt.alarmInhibit, sourcePoint, context);
        return true;
      }

      if (ext instanceof BBacnetTrendLogAlarmSourceExt)
      {
        replaceLinks(ext, BBacnetTrendLogAlarmSourceExt.alarmInhibit, sourcePoint, context);
        return true;
      }

      if (logger.isLoggable(Level.WARNING))
      {
        logger.log(Level.WARNING, getObjectId() + ": when adding alarm inhibit link, type not supported: " + ext.getType());
      }
      return false;
    }
    catch (Exception e)
    {
      if (logger.isLoggable(Level.FINE))
      {
        logger.log(Level.FINE, getObjectId() + ": error adding eventAlgorithmInhibitRef link", e);
      }
      return false;
    }
  }

  private void configureGeneralFaultAlgorithm(BBacnetEventParameter eventParam, BAlarmSourceExt ext)
  {
    BFaultAlgorithm faultAlgorithm = ext.getFaultAlgorithm();
    if (!faultAlgorithm.getType().equals(BFaultAlgorithm.TYPE))
    {
      if (logger.isLoggable(Level.FINE))
      {
        logger.fine(getObjectId() + ": replacing fault algorithm of type " + faultAlgorithm.getType() +
          " with FaultAlgorithm for event type " + BBacnetEventType.tag(eventParam.getChoice()));
      }
      ext.set(BAlarmSourceExt.faultAlgorithm, new BFaultAlgorithm(), BLocalBacnetDevice.getBacnetContext());
    }
  }

  private BAlarmSourceExt updateToAlarmExt(BPointExtension pointExt)
  {
    if (!(pointExt instanceof BAlarmSourceExt))
    {
      if (logger.isLoggable(Level.FINE))
      {
        logger.fine(getObjectId() + ": replacing point extension of type " +
          (pointExt != null ? pointExt.getType() : null) + " with BAlarmExt");
      }
      // TODO Has the non-AlarmSourceExt already been removed?
      return new BAlarmSourceExt();
    }

    return (BAlarmSourceExt) pointExt;
  }

  private BBacnetTrendLogAlarmSourceExt updateToTrendAlarmExt(BPointExtension pointExt)
  {
    if (!(pointExt instanceof BBacnetTrendLogAlarmSourceExt))
    {
      if (logger.isLoggable(Level.FINE))
      {
        logger.fine(getObjectId() + ": replacing point extension of type " + (pointExt != null ? pointExt.getType() : null) +
          " with BBacnetTrendLogAlarmSourceExt");
      }
      return new BBacnetTrendLogAlarmSourceExt();
    }

    return (BBacnetTrendLogAlarmSourceExt) pointExt;
  }

  private static void addExtIfMissing(BPointExtension pointExt, BComponent target)
  {
    if (pointExt.getParent() == null)
    {
      // Not added to the target yet
      target.add("EventEnrollmentAlarmExt?", pointExt, BLocalBacnetDevice.getBacnetContext());
    }
  }

  private void configureNoneExt()
  {
    if (logger.isLoggable(Level.FINE))
    {
      logger.fine(getObjectId() + ": resetting the descriptor because event type is none");
    }
    resetDescriptor();
  }

  //region ChangeOfState

  private void configureChangeOfStateExt(
    BBacnetEventParameter eventParam,
    BBacnetDeviceObjectPropertyReference objPropRef,
    BPointExtension pointExt,
    BComponent target)
      throws EventEnrollmentException
  {
    PropertyInfo propInfo = getPropertyInfo(objPropRef.getObjectId().getObjectType(), objPropRef.getPropertyId());
    if (propInfo == null)
    {
      throw new EventEnrollmentException(
        "BACnet property information not found for " +
          "object ID: " + objPropRef.getObjectId() +
          ", property ID: " + BBacnetPropertyIdentifier.tag(objPropRef.getPropertyId()),
        new NErrorType(property, other));
    }

    BAlarmSourceExt alarmExt = updateToAlarmExt(pointExt);

    if (target instanceof BBooleanPoint)
    {
      configureBooleanChangeOfStateOffnormal(eventParam, propInfo, alarmExt);
    }
    else if (target instanceof BNumericPoint)
    {
      configureNumericChangeOfStateOffnormal(eventParam, objPropRef, propInfo, alarmExt);
    }
    else if (target instanceof BEnumPoint)
    {
      BObject pointRange = ((BControlPoint)target).getFacets().getFacet(BFacets.RANGE);
      configureEnumChangeOfStateOffnormal(eventParam, propInfo, alarmExt, pointRange);
    }
    else
    {
      throw new EventEnrollmentException(
        "referenced object is of type " + target.getType() +
          " and not instanceof BooleanPoint or EnumPoint or NumericPoint, which is required for change-of-state extensions",
        new NErrorType(BBacnetErrorClass.PROPERTY, BBacnetErrorCode.VALUE_OUT_OF_RANGE));
    }

    configureAlarmExt(eventParam, alarmExt);
    configureGeneralFaultAlgorithm(eventParam, alarmExt);
    addExtIfMissing(alarmExt, target);
    updateDescriptor(alarmExt);
  }

  private void configureBooleanChangeOfStateOffnormal(
    BBacnetEventParameter eventParam,
    PropertyInfo propInfo,
    BAlarmSourceExt alarmExt)
      throws EventEnrollmentException
  {
    boolean alarmValue = getBooleanChangeOfStateAlarmValue(eventParam, propInfo);

    BOffnormalAlgorithm offnormalAlgorithm = alarmExt.getOffnormalAlgorithm();
    if (offnormalAlgorithm instanceof BBooleanChangeOfStateAlgorithm)
    {
      offnormalAlgorithm.setBoolean(BBooleanChangeOfStateAlgorithm.alarmValue, alarmValue, BLocalBacnetDevice.getBacnetContext());
    }
    else
    {
      if (logger.isLoggable(Level.FINE))
      {
        logger.fine(getObjectId() + ": replacing offnormal algorithm of type " + offnormalAlgorithm.getType() +
          " with BooleanChangeOfStateAlgorithm for event type " + BBacnetEventType.tag(eventParam.getChoice()));
      }
      BBooleanChangeOfStateAlgorithm changeOfStateAlgorithm = new BBooleanChangeOfStateAlgorithm();
      changeOfStateAlgorithm.setAlarmValue(alarmValue);
      alarmExt.set(BAlarmSourceExt.offnormalAlgorithm, changeOfStateAlgorithm, BLocalBacnetDevice.getBacnetContext());
    }
  }

  private static boolean getBooleanChangeOfStateAlarmValue(
    BBacnetEventParameter eventParam,
    PropertyInfo propInfo)
      throws EventEnrollmentException
  {
    BBacnetListOf listOfValues = (BBacnetListOf) eventParam.get(BBacnetEventParameter.LIST_OF_VALUES_SLOT_NAME);
    BBacnetPropertyStates[] propStates = listOfValues.getChildren(BBacnetPropertyStates.class);
    if (propStates.length < 1)
    {
      throw new EventEnrollmentException(
        "boolean change-of-state alarm extensions require at least 1 alarm value; event type: " +
          BBacnetEventType.tag(eventParam.getChoice()),
        new NErrorType(BBacnetErrorClass.PROPERTY, BBacnetErrorCode.VALUE_OUT_OF_RANGE));
    }

    if (propInfo.getAsnType() == ASN_BOOLEAN)
    {
      BValue value = propStates[0].getValue();
      if (value instanceof BBoolean)
      {
        return ((BBoolean) value).getBoolean();
      }
      else
      {
        throw new EventEnrollmentException(
          "boolean change-of-state alarm extension requires a BOOLEAN (0) value for property type: " + propInfo,
          new NErrorType(property, valueOutOfRange));
      }
    }
    else if (isBinaryPv(propInfo))
    {
      BValue value = propStates[0].getValue();
      if (value instanceof BBacnetBinaryPv)
      {
        return ((BBacnetBinaryPv) value).isActive();
      }
      else
      {
        throw new EventEnrollmentException(
          "boolean change-of-state alarm extension requires a BACnetBinaryPv (1) value for property type: " + propInfo,
          new NErrorType(property, valueOutOfRange));
      }
    }
    else
    {
      throw new EventEnrollmentException(
        "boolean change-of-state alarm extension is not supported for the property type " + propInfo,
        new NErrorType(property, other));
    }
  }

  private void configureNumericChangeOfStateOffnormal(
    BBacnetEventParameter eventParam,
    BBacnetDeviceObjectPropertyReference objPropRef,
    PropertyInfo propInfo,
    BAlarmSourceExt alarmExt)
      throws EventEnrollmentException
  {
    BEnumRange alarmValues = getNumericChangeOfStateAlarmValues(eventParam, objPropRef, propInfo);

    BOffnormalAlgorithm offnormalAlgorithm = alarmExt.getOffnormalAlgorithm();
    if (offnormalAlgorithm instanceof BNumericChangeOfStateAlgorithm)
    {
      offnormalAlgorithm.set(BNumericChangeOfStateAlgorithm.alarmValues, alarmValues, BLocalBacnetDevice.getBacnetContext());
    }
    else
    {
      if (logger.isLoggable(Level.FINE))
      {
        logger.fine(getObjectId() + ": replacing offnormal algorithm of type " + offnormalAlgorithm.getType() +
          " with NumericChangeOfStateAlgorithm for event type " + BBacnetEventType.tag(eventParam.getChoice()));
      }
      BNumericChangeOfStateAlgorithm changeOfStateAlgorithm = new BNumericChangeOfStateAlgorithm();
      changeOfStateAlgorithm.setAlarmValues(alarmValues);
      alarmExt.set(BAlarmSourceExt.offnormalAlgorithm, changeOfStateAlgorithm, BLocalBacnetDevice.getBacnetContext());
    }
  }

  private static BEnumRange getNumericChangeOfStateAlarmValues(
    BBacnetEventParameter eventParam,
    BBacnetDeviceObjectPropertyReference objPropRef,
    PropertyInfo propInfo)
      throws EventEnrollmentException
  {
    BBacnetListOf listOfValues = (BBacnetListOf) eventParam.get(BBacnetEventParameter.LIST_OF_VALUES_SLOT_NAME);
    BBacnetPropertyStates[] propStates = listOfValues.getChildren(BBacnetPropertyStates.class);

    int asnType = propInfo.getAsnType();

    int[] ordinals = new int[propStates.length];
    String[] tags = new String[propStates.length];
    if (asnType == ASN_UNSIGNED || isArraySize(propInfo, objPropRef))
    {
      for (int i = 0; i < propStates.length; i++)
      {
        BValue value = propStates[i].getValue();
        if (value instanceof BBacnetUnsigned)
        {
          ordinals[i] = ((BBacnetUnsigned)value).getInt();
          tags[i] = SlotPath.escape(String.valueOf(ordinals[i]));
        }
        else
        {
          throw new EventEnrollmentException(
            "numeric change-of-state alarm extension requires an UNSIGNED (11) value for property type: " + propInfo,
            new NErrorType(property, valueOutOfRange));
        }
      }
    }
    else if (asnType == ASN_INTEGER)
    {
      for (int i = 0; i < propStates.length; i++)
      {
        BValue value = propStates[i].getValue();
        if (value instanceof BInteger)
        {
          ordinals[i] = ((BInteger) value).getInt();
          tags[i] = SlotPath.escape(String.valueOf(ordinals[i]));
        }
        else
        {
          throw new EventEnrollmentException(
            "numeric change-of-state alarm extension requires an INTEGER (41) value for the property type: " + propInfo,
            new NErrorType(property, valueOutOfRange));
        }
      }
    }
    else
    {
      throw new EventEnrollmentException(
        "numeric change-of-state alarm extension is not supported for the property type " + propInfo,
        new NErrorType(property, other));
    }

    return BEnumRange.make(ordinals, tags);
  }

  private void configureEnumChangeOfStateOffnormal(
    BBacnetEventParameter eventParam,
    PropertyInfo propInfo,
    BAlarmSourceExt alarmExt,
    BObject pointRange)
      throws EventEnrollmentException
  {
    BEnumRange alarmValues = getEnumChangeOfStateAlarmValues(eventParam, propInfo, pointRange);

    BOffnormalAlgorithm offnormalAlgorithm = alarmExt.getOffnormalAlgorithm();
    if (offnormalAlgorithm instanceof BEnumChangeOfStateAlgorithm)
    {
      offnormalAlgorithm.set(BEnumChangeOfStateAlgorithm.alarmValues, alarmValues, BLocalBacnetDevice.getBacnetContext());
    }
    else
    {
      if (logger.isLoggable(Level.FINE))
      {
        logger.fine(getObjectId() + ": replacing offnormal algorithm of type " + offnormalAlgorithm.getType() +
          " with EnumChangeOfStateAlgorithm for event type " + BBacnetEventType.tag(eventParam.getChoice()));
      }
      BEnumChangeOfStateAlgorithm changeOfStateAlgorithm = new BEnumChangeOfStateAlgorithm();
      changeOfStateAlgorithm.setAlarmValues(alarmValues);
      alarmExt.set(BAlarmSourceExt.offnormalAlgorithm, changeOfStateAlgorithm, BLocalBacnetDevice.getBacnetContext());
    }
  }

  private static BEnumRange getEnumChangeOfStateAlarmValues(
    BBacnetEventParameter eventParam,
    PropertyInfo propInfo,
    BObject pointRange)
      throws EventEnrollmentException
  {
    BBacnetListOf listOfValues = (BBacnetListOf) eventParam.get(BBacnetEventParameter.LIST_OF_VALUES_SLOT_NAME);
    BBacnetPropertyStates[] propStates = listOfValues.getChildren(BBacnetPropertyStates.class);

    int[] ordinals = new int[propStates.length];
    String[] tags = new String[propStates.length];
    if (propInfo.isEnum())
    {
      BTypeSpec propTypeSpec = BTypeSpec.make(propInfo.getType());
      if (pointRange instanceof BEnumRange)
      {
        Type pointRangeType = ((BEnumRange) pointRange).getFrozenType();
        if (pointRangeType != null && !pointRangeType.getTypeSpec().equals(propTypeSpec))
        {
          throw new EventEnrollmentException(
            "enum change-of-state alarm extension enum type \"" + pointRangeType + '"' +
              " does not match property type " + propInfo,
            new NErrorType(property, other));
        }
      }

      for (int i = 0; i < propStates.length; i++)
      {
        BValue value = propStates[i].getValue();
        if (!(value instanceof BEnum) ||
            (!propInfo.isExtensible() && !value.getType().getTypeSpec().equals(propTypeSpec)) ||
             (propInfo.isExtensible() && !((((BEnum) value).getRange().getFrozenType().getTypeSpec().equals(propTypeSpec)))))
        {
          throw new EventEnrollmentException(
            "enum change-of-state alarm value enum type \"" + value.getType() + '"' +
              " does not match property type " + propInfo,
            new NErrorType(property, valueOutOfRange));
        }

        BEnum enumValue = (BEnum) value;
        ordinals[i] = enumValue.getOrdinal();
        tags[i] = enumValue.getTag();
      }
    }
    else if (propInfo.getAsnType() == ASN_UNSIGNED)
    {
      for (int i = 0; i < propStates.length; i++)
      {
        BValue value = propStates[i].getValue();
        if (value instanceof BBacnetUnsigned)
        {
          ordinals[i] = ((BBacnetUnsigned)value).getInt();
          tags[i] = SlotPath.escape(String.valueOf(ordinals[i]));
        }
        else
        {
          throw new EventEnrollmentException(
            "enum change-of-state alarm extension requires an UNSIGNED (11) value for property type: " + propInfo,
            new NErrorType(property, valueOutOfRange));
        }
      }
    }
    else if (propInfo.getAsnType() == ASN_INTEGER)
    {
      for (int i = 0; i < propStates.length; i++)
      {
        BValue value = propStates[i].getValue();
        if (value instanceof BInteger)
        {
          ordinals[i] = ((BInteger) value).getInt();
          tags[i] = SlotPath.escape(String.valueOf(ordinals[i]));
        }
        else
        {
          throw new EventEnrollmentException(
            "enum change-of-state alarm extension requires an INTEGER (41) value for the property type: " + propInfo,
            new NErrorType(property, valueOutOfRange));
        }
      }
    }
    else
    {
      throw new EventEnrollmentException(
        "enum change-of-state alarm extension is not supported for the property type " + propInfo,
        new NErrorType(property, other));
    }

    return BEnumRange.make(ordinals, tags);
  }

  //endregion

  //region CommandFailure

  private void configureCommandFailureExt(
    BBacnetEventParameter eventParam,
    BPointExtension pointExt,
    BComponent target)
      throws EventEnrollmentException
  {
    // TODO check that the property type matches the event type?

    if (target instanceof BBooleanPoint || target instanceof BEnumPoint)
    {
      BAlarmSourceExt alarmExt = updateToAlarmExt(pointExt);
      configureAlarmExt(eventParam, alarmExt);

      if (target instanceof BBooleanPoint)
      {
        configureBooleanCommandFailureOffnormal(eventParam, alarmExt);
        configureGeneralFaultAlgorithm(eventParam, alarmExt);
      }
      else
      {
        configureEnumCommandFailureOffnormal(eventParam, alarmExt);
        configureEnumCommandFailureFault(eventParam, alarmExt);
      }

      addExtIfMissing(alarmExt, target);
      updateDescriptor(alarmExt);
    }
    else
    {
      throw new EventEnrollmentException(
        "referenced object is of type " + target.getType() +
          " and not instanceof BooleanPoint or EnumPoint, which is required for command failure extensions; event type: " +
          BBacnetEventType.tag(eventParam.getChoice()),
        new NErrorType(BBacnetErrorClass.PROPERTY, BBacnetErrorCode.VALUE_OUT_OF_RANGE));
    }
  }

  private void configureBooleanCommandFailureOffnormal(BBacnetEventParameter eventParam, BAlarmSourceExt alarmExt)
    throws EventEnrollmentException
  {
    BOffnormalAlgorithm offnormalAlgorithm = alarmExt.getOffnormalAlgorithm();
    if (offnormalAlgorithm instanceof BBooleanCommandFailureAlgorithm)
    {
      configureBooleanCommandFailureOffnormal(eventParam, (BBooleanCommandFailureAlgorithm) offnormalAlgorithm);
    }
    else
    {
      if (logger.isLoggable(Level.FINE))
      {
        logger.fine(getObjectId() + ": replacing offnormal algorithm of type " + offnormalAlgorithm.getType() +
          " with BooleanCommandFailureAlgorithm for event type " + BBacnetEventType.tag(eventParam.getChoice()));
      }
      BBooleanCommandFailureAlgorithm commandFailureAlgorithm = new BBooleanCommandFailureAlgorithm();
      configureBooleanCommandFailureOffnormal(eventParam, commandFailureAlgorithm);
      alarmExt.set(BAlarmSourceExt.offnormalAlgorithm, commandFailureAlgorithm, BLocalBacnetDevice.getBacnetContext());
    }
  }

  private void configureBooleanCommandFailureOffnormal(
    BBacnetEventParameter eventParam,
    BBooleanCommandFailureAlgorithm algorithm)
      throws EventEnrollmentException
  {
    BControlPoint feedbackPoint = getCommandFailureFeedbackPoint(eventParam);
    if (!(feedbackPoint instanceof BBooleanPoint))
    {
      throw new EventEnrollmentException(
        "feedback point for boolean command failure is type " + feedbackPoint.getType() +
          " but should be instanceof BooleanPoint",
        new NErrorType(BBacnetErrorClass.PROPERTY, BBacnetErrorCode.VALUE_OUT_OF_RANGE));
    }

    Context context = BLocalBacnetDevice.getBacnetContext();
    checkLinkPermissions(feedbackPoint, "out", context);
    replaceLinks(algorithm, BBooleanCommandFailureAlgorithm.feedbackValue, feedbackPoint, context);
  }

  private static void replaceLinks(BComponent target, Property targetSlot, BComponent source, Context context)
  {
    // Set this once a link is found with the correct sourceOrd and sourceSlotName. Any other links
    // with the same sourceOrd or sourceSlotName should be removed so only a single one remains.
    BLink existingLink = null;

    for (BLink link : target.getLinks(targetSlot))
    {
      if (!link.getSourceOrd().equals(source.getHandleOrd()) ||
          !link.getSourceSlotName().equals("out") ||
          existingLink != null)
      {
        if (logger.isLoggable(Level.FINE))
        {
          logger.fine("Before adding a link to feedback point, cleared link to source ord "
            + link.getSourceOrd() + " and source slot " + link.getSourceSlotName());
        }
        target.remove(link.getPropertyInParent(), context);
      }
      else
      {
        existingLink = link;
      }
    }

    if (existingLink == null)
    {
      target.add(
        null,
        new BLink(
          source.getHandleOrd(),
          /* sourceSlot */ "out",
          targetSlot.getName(),
          /* enabled */ true),
        context);
    }
  }

  private BBacnetDeviceObjectPropertyReference getLinkedPropertyReference(
    BComponent component,
    Property targetSlot,
    Type targetDescType)
  {
    BLink[] targetSlotLinks = component.getLinks(targetSlot);
    for (BLink link : targetSlotLinks)
    {
      if (!link.getSourceSlotName().equals("out") || !link.isActive() || !link.isEnabled())
      {
        continue;
      }

      BComponent source = link.getSourceComponent();
      if (source instanceof BControlPoint)
      {
        BAbstractProxyExt proxyExt = ((BControlPoint) source).getProxyExt();
        if (proxyExt instanceof BBacnetProxyExt)
        {
          // Must be a point on a remote device.
          return makeRemoteDeviceObjPropRef((BBacnetProxyExt) proxyExt);
        }
      }

      // TODO support links to remote schedules?

      // Must be a local object
      // Device ID should be blank for local objects
      BIBacnetExportObject descriptor = findDescriptor(source.getHandleOrd());
      if (descriptor != null && descriptor.getType().is(targetDescType))
      {
        return new BBacnetDeviceObjectPropertyReference(
          /* objectId */ descriptor.getObjectId(),
          /* propertyId */ presentValue);
      }
    }

    return makeUnconfiguredDeviceObjPropRef();
  }

  private static BBacnetDeviceObjectPropertyReference makeUnconfiguredDeviceObjPropRef()
  {
    return new BBacnetDeviceObjectPropertyReference(
      /* objectId */ BBacnetObjectIdentifier.make(BBacnetObjectType.ANALOG_INPUT, BBacnetObjectIdentifier.UNCONFIGURED_INSTANCE_NUMBER),
      /* propertyId */ BBacnetPropertyIdentifier.PRESENT_VALUE,
      /* propertyArrayIndex */ NOT_USED,
      /* deviceId */ BBacnetObjectIdentifier.make(BBacnetObjectType.DEVICE, BBacnetObjectIdentifier.UNCONFIGURED_INSTANCE_NUMBER));
  }

  private static BBacnetDeviceObjectPropertyReference makeRemoteDeviceObjPropRef(BBacnetProxyExt proxyExt)
  {
    return new BBacnetDeviceObjectPropertyReference(
      proxyExt.getObjectId(),
      proxyExt.getPropertyId().getOrdinal(),
      proxyExt.getPropertyArrayIndex(),
      proxyExt.device().getObjectId());
  }

  private void configureEnumCommandFailureOffnormal(BBacnetEventParameter eventParam, BAlarmSourceExt alarmExt)
    throws EventEnrollmentException
  {
    BOffnormalAlgorithm offnormalAlgorithm = alarmExt.getOffnormalAlgorithm();
    if (offnormalAlgorithm instanceof BEnumCommandFailureAlgorithm)
    {
      configureEnumCommandFailureOffnormal(eventParam, (BEnumCommandFailureAlgorithm) offnormalAlgorithm);
    }
    else
    {
      if (logger.isLoggable(Level.FINE))
      {
        logger.fine(getObjectId() + ": replacing offnormal algorithm of type " + offnormalAlgorithm.getType() +
          " with EnumCommandFailureAlgorithm for event type " + BBacnetEventType.tag(eventParam.getChoice()));
      }
      BEnumCommandFailureAlgorithm commandFailureAlgorithm = new BEnumCommandFailureAlgorithm();
      configureEnumCommandFailureOffnormal(eventParam, commandFailureAlgorithm);
      alarmExt.set(BAlarmSourceExt.offnormalAlgorithm, commandFailureAlgorithm, BLocalBacnetDevice.getBacnetContext());
    }
  }

  private void configureEnumCommandFailureOffnormal(
    BBacnetEventParameter eventParam,
    BEnumCommandFailureAlgorithm algorithm)
      throws EventEnrollmentException
  {
    BControlPoint feedbackPoint = getCommandFailureFeedbackPoint(eventParam);
    if (!(feedbackPoint instanceof BEnumPoint))
    {
      throw new EventEnrollmentException(
        "feedback point for enum command failure is type " + feedbackPoint.getType() +
          " but should be instanceof EnumPoint",
        new NErrorType(BBacnetErrorClass.PROPERTY, BBacnetErrorCode.VALUE_OUT_OF_RANGE));
    }

    Context context = BLocalBacnetDevice.getBacnetContext();
    checkLinkPermissions(feedbackPoint, "out", context);
    replaceLinks(algorithm, BEnumCommandFailureAlgorithm.feedbackValue, feedbackPoint, context);
  }

  private void configureEnumCommandFailureFault(BBacnetEventParameter eventParam, BAlarmSourceExt alarmExt)
  {
    BFaultAlgorithm faultAlgorithm = alarmExt.getFaultAlgorithm();
    if (!(alarmExt.getFaultAlgorithm() instanceof BEnumFaultAlgorithm))
    {
      if (logger.isLoggable(Level.FINE))
      {
        logger.fine(getObjectId() + ": replacing fault algorithm of type " +
          faultAlgorithm.getType() + " with EnumFaultAlgorithm for event type " +
          BBacnetEventType.tag(eventParam.getChoice()));
      }
      alarmExt.set(BAlarmSourceExt.faultAlgorithm, new BEnumFaultAlgorithm(), BLocalBacnetDevice.getBacnetContext());
    }
  }

  private BControlPoint getCommandFailureFeedbackPoint(BBacnetEventParameter eventParam)
    throws EventEnrollmentException
  {
    BValue feedbackRef = eventParam.get(BBacnetEventParameter.FEEDBACK_PROPERTY_REFERENCE_SLOT_NAME);
    if (!(feedbackRef instanceof BBacnetDeviceObjectPropertyReference))
    {
      throw new EventEnrollmentException(
        "feedback reference for command failure is type " +
          (feedbackRef != null ? feedbackRef.getType() : null) +
          " but should be instanceof BacnetDeviceObjectPropertyReference",
        new NErrorType(BBacnetErrorClass.PROPERTY, BBacnetErrorCode.VALUE_OUT_OF_RANGE));
    }

    try
    {
      return BacnetDescriptorUtil.findOrAddPoint((BBacnetDeviceObjectPropertyReference) feedbackRef);
    }
    catch (Exception e)
    {
      logException(
        Level.FINE,
        new StringBuilder(getObjectId().toString())
          .append(": error finding point for command failure feedback ref"),
        e);
      throw new EventEnrollmentException(
        "error finding point for command failure feedback ref",
        new NErrorType(BBacnetErrorClass.PROPERTY, BBacnetErrorCode.VALUE_OUT_OF_RANGE));
    }
  }

  //endregion

  //region FloatingLimit

  private void configureFloatingLimitExt(
    BBacnetEventParameter eventParam,
    BPointExtension pointExt,
    BComponent target)
      throws EventEnrollmentException
  {
    // TODO should we check that the object property reference points to a property type that matches the event type?
    //  For example, DOUBLE_OUT_OF_RANGE is placed on a NumericPoint tied to a Double parameter?
    //  int objType = getRemoteObjectType();
    if (!(target instanceof BNumericPoint))
    {
      throw new EventEnrollmentException(
        "referenced object is of type " + target.getType() +
          " and not instanceof NumericPoint, which is required for Floating Limit Algorithm extensions; event type: " +
          BBacnetEventType.tag(eventParam.getChoice()),
        new NErrorType(BBacnetErrorClass.PROPERTY, BBacnetErrorCode.VALUE_OUT_OF_RANGE));
    }

    BAlarmSourceExt alarmExt = updateToAlarmExt(pointExt);
    configureAlarmExt(eventParam, alarmExt);
    configureFloatingLimitOffnormal(eventParam, alarmExt);
    configureGeneralFaultAlgorithm(eventParam, alarmExt);
    addExtIfMissing(alarmExt, target);
    updateDescriptor(alarmExt);
  }

  private void configureFloatingLimitOffnormal(BBacnetEventParameter eventParam, BAlarmSourceExt alarmExt)
    throws EventEnrollmentException
  {
    BOffnormalAlgorithm offnormalAlgorithm = alarmExt.getOffnormalAlgorithm();
    if (offnormalAlgorithm instanceof BFloatingLimitAlgorithm)
    {
      configureFloatingLimitOffnormal(eventParam, (BFloatingLimitAlgorithm) offnormalAlgorithm);
    }
    else
    {
      if (logger.isLoggable(Level.FINE))
      {
        logger.fine(getObjectId() + ": replacing offnormal algorithm of type " + offnormalAlgorithm.getType() +
          " with BFloatingLimitAlgorithm for event type " + BBacnetEventType.tag(eventParam.getChoice()));
      }
      BFloatingLimitAlgorithm floatingLimitAlgorithm = new BFloatingLimitAlgorithm();
      configureFloatingLimitOffnormal(eventParam, floatingLimitAlgorithm);
      alarmExt.set(BAlarmSourceExt.offnormalAlgorithm, floatingLimitAlgorithm, BLocalBacnetDevice.getBacnetContext());
    }
  }

  private void configureFloatingLimitOffnormal(BBacnetEventParameter eventParam, BFloatingLimitAlgorithm algorithm)
    throws EventEnrollmentException
  {
    BControlPoint setpoint = getFloatingLimitSetpoint(eventParam);
    if (!(setpoint instanceof BNumericPoint))
    {
      throw new EventEnrollmentException(
        "feedback point for floating limit is type " + setpoint.getType() +
          " but should be instanceof BNumericPoint",
        new NErrorType(BBacnetErrorClass.PROPERTY, BBacnetErrorCode.VALUE_OUT_OF_RANGE));
    }

    Context context = BLocalBacnetDevice.getBacnetContext();
    checkLinkPermissions(setpoint, "out", context);
    replaceLinks(algorithm, BFloatingLimitAlgorithm.setpoint, setpoint, context);

    algorithm.setDouble(BFloatingLimitAlgorithm.lowDiffLimit, ((BNumber) eventParam.get(BBacnetEventParameter.LOW_DIFF_LIMIT_SLOT_NAME)).getDouble(), context);
    algorithm.setDouble(BFloatingLimitAlgorithm.highDiffLimit, ((BNumber) eventParam.get(BBacnetEventParameter.HIGH_DIFF_LIMIT_SLOT_NAME)).getDouble(), context);
    algorithm.setDouble(BFloatingLimitAlgorithm.deadband, ((BNumber) eventParam.get(BBacnetEventParameter.DEADBAND_SLOT_NAME)).getDouble(), context);

    BLimitEnable limitEnable = algorithm.getLimitEnable();
    limitEnable.setBoolean(BLimitEnable.highLimitEnable, true, context);
    limitEnable.setBoolean(BLimitEnable.lowLimitEnable, true, context);
  }

  private BControlPoint getFloatingLimitSetpoint(BBacnetEventParameter eventParam)
    throws EventEnrollmentException
  {
    BValue setpointRef = eventParam.get(BBacnetEventParameter.SETPOINT_REFERENCE_SLOT_NAME);
    if (!(setpointRef instanceof BBacnetDeviceObjectPropertyReference))
    {
      throw new EventEnrollmentException(
        "setpoint reference for Floating Limit is type " +
          (setpointRef != null ? setpointRef.getType() : null) +
          " but should be instanceof BacnetDeviceObjectPropertyReference",
        new NErrorType(BBacnetErrorClass.PROPERTY, BBacnetErrorCode.VALUE_OUT_OF_RANGE));
    }

    try
    {
      return BacnetDescriptorUtil.findOrAddPoint((BBacnetDeviceObjectPropertyReference) setpointRef);
    }
    catch (Exception e)
    {
      logException(
        Level.WARNING,
        new StringBuilder(getObjectId().toString()).append(": error finding point for floating limit setpoint ref"),
        e);
      throw new EventEnrollmentException(
        "error finding point for floating limit setpoint ref",
        new NErrorType(BBacnetErrorClass.PROPERTY, BBacnetErrorCode.VALUE_OUT_OF_RANGE));
    }
  }

  //endregion

  //region OutOfRange

  private void configureOutOfRangeExt(
    BBacnetEventParameter eventParam,
    BBacnetDeviceObjectPropertyReference objPropRef,
    BPointExtension pointExt,
    BComponent target)
      throws EventEnrollmentException
  {
    checkOutOfRangeTarget(eventParam, objPropRef, target);

    BAlarmSourceExt alarmExt = updateToAlarmExt(pointExt);
    configureAlarmExt(eventParam, alarmExt);
    configureOutOfRangeOffnormal(eventParam, alarmExt);
    configureOutOfRangeFault(eventParam, alarmExt);
    addExtIfMissing(alarmExt, target);
    updateDescriptor(alarmExt);
  }

  private static void checkOutOfRangeTarget(
    BBacnetEventParameter eventParam,
    BBacnetDeviceObjectPropertyReference objPropRef,
    BComponent target)
      throws EventEnrollmentException
  {
    if (!(target instanceof BNumericPoint))
    {
      throw new EventEnrollmentException(
        "referenced object is of type " + target.getType() +
          " and not instanceof NumericPoint, which is required for out-of-range extensions" +
          "; event type: " + BBacnetEventType.tag(eventParam.getChoice()),
        new NErrorType(property, valueOutOfRange));
    }

    int objType = objPropRef.getObjectId().getObjectType();
    int propId = objPropRef.getPropertyId();
    PropertyInfo propInfo = getPropertyInfo(objType, propId);
    if (propInfo == null)
    {
      throw new EventEnrollmentException(
        "BACnet property information not found for" +
          " object ID: " + objPropRef.getObjectId() +
          ", property ID: " + BBacnetPropertyIdentifier.tag(propId),
        new NErrorType(property, valueOutOfRange));
    }

    int eventType = eventParam.getChoice();
    int asnType = propInfo.getAsnType();
    switch (eventType)
    {
      case BBacnetEventType.OUT_OF_RANGE:
        if (asnType != ASN_REAL)
        {
          throw makeInvalidDataTypeException(eventType, asnType, objType, propId);
        }
        break;
      case BBacnetEventType.DOUBLE_OUT_OF_RANGE:
        if (asnType != ASN_DOUBLE)
        {
          throw makeInvalidDataTypeException(eventType, asnType, objType, propId);
        }
        break;
      case BBacnetEventType.SIGNED_OUT_OF_RANGE:
        if (asnType != ASN_INTEGER)
        {
          throw makeInvalidDataTypeException(eventType, asnType, objType, propId);
        }
        break;
      case BBacnetEventType.UNSIGNED_OUT_OF_RANGE:
        if (asnType != ASN_UNSIGNED && !isArraySize(propInfo, objPropRef))
        {
          throw makeInvalidDataTypeException(eventType, asnType, objType, propId);
        }
        break;
    }
  }

  private static EventEnrollmentException makeInvalidDataTypeException(int eventType, int asnType, int objType, int propId)
  {
    return new EventEnrollmentException(
      "event type " + BBacnetEventType.tag(eventType) +
        " is not supported for the property data type: " + AsnUtil.getAsnTypeName(asnType) +
        "; object type: " + BBacnetObjectType.tag(objType) +
        ", property ID: " + BBacnetPropertyIdentifier.tag(propId),
      new NErrorType(property, valueOutOfRange));
  }

  private void configureOutOfRangeOffnormal(BBacnetEventParameter eventParam, BAlarmSourceExt ext)
  {
    BOffnormalAlgorithm offnormalAlgorithm = ext.getOffnormalAlgorithm();
    if (offnormalAlgorithm instanceof BOutOfRangeAlgorithm)
    {
      configureOutOfRangeOffnormal(eventParam, (BOutOfRangeAlgorithm) offnormalAlgorithm);
    }
    else
    {
      if (logger.isLoggable(Level.FINE))
      {
        logger.fine(getObjectId() + ": replacing offnormal algorithm of type " + offnormalAlgorithm.getType() +
          " with OutOfRangeAlgorithm for event type " + BBacnetEventType.tag(eventParam.getChoice()));
      }
      BOutOfRangeAlgorithm outOfRangeAlgorithm = new BOutOfRangeAlgorithm();
      configureOutOfRangeOffnormal(eventParam, outOfRangeAlgorithm);
      ext.set(BAlarmSourceExt.offnormalAlgorithm, outOfRangeAlgorithm, BLocalBacnetDevice.getBacnetContext());
    }
  }

  private static void configureOutOfRangeOffnormal(BBacnetEventParameter eventParam, BOutOfRangeAlgorithm algorithm)
  {
    Context context = BLocalBacnetDevice.getBacnetContext();
    algorithm.setDouble(BOutOfRangeAlgorithm.lowLimit, ((BNumber) eventParam.get(BBacnetEventParameter.LOW_LIMIT_SLOT_NAME)).getDouble(), context);
    algorithm.setDouble(BOutOfRangeAlgorithm.highLimit, ((BNumber) eventParam.get(BBacnetEventParameter.HIGH_LIMIT_SLOT_NAME)).getDouble(), context);
    algorithm.setDouble(BOutOfRangeAlgorithm.deadband, ((BNumber) eventParam.get(BBacnetEventParameter.DEADBAND_SLOT_NAME)).getDouble(), context);

    BLimitEnable limitEnable = algorithm.getLimitEnable();
    limitEnable.setBoolean(BLimitEnable.highLimitEnable, true, context);
    limitEnable.setBoolean(BLimitEnable.lowLimitEnable, true, context);
  }

  private void configureOutOfRangeFault(BBacnetEventParameter eventParam, BAlarmSourceExt ext)
  {
    BFaultAlgorithm faultAlgorithm = ext.getFaultAlgorithm();
    if (!(ext.getFaultAlgorithm() instanceof BOutOfRangeFaultAlgorithm))
    {
      if (logger.isLoggable(Level.FINE))
      {
        logger.fine(getObjectId() + ": replacing fault algorithm of type " + faultAlgorithm.getType() +
          " with OutOfRangeFaultAlgorithm for event type " + BBacnetEventType.tag(eventParam.getChoice()));
      }
      ext.set(BAlarmSourceExt.faultAlgorithm, new BOutOfRangeFaultAlgorithm(), BLocalBacnetDevice.getBacnetContext());
    }
  }

  //endregion

  //region BufferReady

  private void configureTrendAlarmExt(BBacnetEventParameter eventParam, BPointExtension pointExt, BComponent target)
    throws EventEnrollmentException
  {
    // TODO check that the property type matches the event type? TREND_LOG
    // int objType = getRemoteObjectType();
    if (!(target instanceof BIBacnetTrendLogExt))
    {
      throw new EventEnrollmentException(
        "target is of type " + target.getType() +
          " and not instanceof BIBacnetTrendLogExt, which is required for trend log alarm source extensions; event type: " +
          BBacnetEventType.tag(eventParam.getChoice()),
        new NErrorType(BBacnetErrorClass.PROPERTY, BBacnetErrorCode.VALUE_OUT_OF_RANGE));
    }

    BBacnetTrendLogAlarmSourceExt trendAlarmExt = updateToTrendAlarmExt(pointExt);
    addExtIfMissing(trendAlarmExt, target);

    Context context = BLocalBacnetDevice.getBacnetContext();
    trendAlarmExt.set(BAlarmSourceExt.alarmEnable, eventEnable, context);
    configureAlarmClass(trendAlarmExt, getNotificationClassId(), context);
    trendAlarmExt.set(BBacnetTrendLogAlarmSourceExt.toNormalText, BFormat.make(toNormalText), context);
    trendAlarmExt.updateParameters(
      getLongParameter(eventParam, BBacnetEventParameter.NOTIFICATION_THRESHOLD_SLOT_NAME),
      getLongParameter(eventParam, BBacnetEventParameter.PREVIOUS_NOTIFICATION_COUNT_SLOT_NAME),
      context);
    configureAlarmInhibit(trendAlarmExt, context);
    updateDescriptor(trendAlarmExt);
  }

  private static long getLongParameter(BBacnetEventParameter eventParam, String slotName)
  {
    return ((BBacnetUnsigned) eventParam.get(slotName)).getLong();
  }

  //endregion

  //region StringChangeOfState

  private void configureStringChangeOfStateExt(
    BBacnetEventParameter eventParam,
    BPointExtension pointExt,
    BComponent target)
      throws EventEnrollmentException
  {
    // TODO check that the property type matches the event type?
    // int objType = getRemoteObjectType();
    if (!(target instanceof BStringPoint))
    {
      throw new EventEnrollmentException(
        "referenced object is of type " + target.getType() +
          " and not instanceof StringPoint, which is required for String change-of-state extensions; event type: " +
          BBacnetEventType.tag(eventParam.getChoice()),
        new NErrorType(BBacnetErrorClass.PROPERTY, BBacnetErrorCode.VALUE_OUT_OF_RANGE));
    }

    BAlarmSourceExt alarmExt = updateToAlarmExt(pointExt);
    configureAlarmExt(eventParam, alarmExt);
    configureStringChangeOfStateOffnormal(eventParam, alarmExt);
    configureStringChangeOfStateFault(eventParam, alarmExt);
    addExtIfMissing(alarmExt, target);
    updateDescriptor(alarmExt);
  }

  private void configureStringChangeOfStateOffnormal(BBacnetEventParameter eventParam, BAlarmSourceExt alarmExt)
    throws EventEnrollmentException
  {
    BOffnormalAlgorithm offnormalAlgorithm = alarmExt.getOffnormalAlgorithm();
    if (offnormalAlgorithm instanceof BStringChangeOfStateAlgorithm)
    {
      configureStringChangeOfStateOffnormal(eventParam, (BStringChangeOfStateAlgorithm) offnormalAlgorithm);
    }
    else
    {
      if (logger.isLoggable(Level.FINE))
      {
        logger.fine(getObjectId() + ": replacing offnormal algorithm of type " + offnormalAlgorithm.getType() +
          " with StringChangeOfStateAlgorithm for event type " + BBacnetEventType.tag(eventParam.getChoice()));
      }
      BStringChangeOfStateAlgorithm stringChangeOfStateAlgorithm = new BStringChangeOfStateAlgorithm();
      configureStringChangeOfStateOffnormal(eventParam, stringChangeOfStateAlgorithm);
      alarmExt.set(BAlarmSourceExt.offnormalAlgorithm, stringChangeOfStateAlgorithm, BLocalBacnetDevice.getBacnetContext());
    }
  }

  private static void configureStringChangeOfStateOffnormal(
    BBacnetEventParameter eventParam,
    BStringChangeOfStateAlgorithm algorithm)
      throws EventEnrollmentException
  {
    BBacnetListOf listOfValues = (BBacnetListOf) eventParam.get(BBacnetEventParameter.LIST_OF_ALARM_VALUES_SLOT_NAME);
    BString[] alarmValues = listOfValues.getChildren(BString.class);

    // TODO Handle multiple alarm values

    if (alarmValues.length < 1)
    {
      throw new EventEnrollmentException(
        "String change-of-state alarm extensions require at least 1 alarm value; event type: " +
          BBacnetEventType.tag(eventParam.getChoice()),
        new NErrorType(BBacnetErrorClass.PROPERTY, BBacnetErrorCode.VALUE_OUT_OF_RANGE));
    }

    String alarmValue = alarmValues[0].getString();
    Context context = BLocalBacnetDevice.getBacnetContext();
    algorithm.setString(BStringChangeOfStateAlgorithm.expression, alarmValue, context);
    algorithm.setBoolean(BStringChangeOfStateAlgorithm.normalOnMatch, false, context);
  }

  private void configureStringChangeOfStateFault(BBacnetEventParameter eventParam, BAlarmSourceExt alarmExt)
  {
    BFaultAlgorithm faultAlgorithm = alarmExt.getFaultAlgorithm();
    if (!(alarmExt.getFaultAlgorithm() instanceof BStringChangeOfStateFaultAlgorithm))
    {
      if (logger.isLoggable(Level.FINE))
      {
        logger.fine(getObjectId() + ": replacing fault algorithm of type " + faultAlgorithm.getType() +
          " with StringChangeOfStateFaultAlgorithm for event type " + BBacnetEventType.tag(eventParam.getChoice()));
      }
      alarmExt.set(BAlarmSourceExt.faultAlgorithm, new BStringChangeOfStateFaultAlgorithm(), BLocalBacnetDevice.getBacnetContext());
    }
  }

  //endregion

  //region ChangeOfStatusFlags

  // Not used. Implementation was in the previous version and has been retained but not enabled
  // until it can be thoroughly tested.
  @SuppressWarnings("unused")
  private void configureChangeOfStatusFlagsExt(
    BBacnetEventParameter eventParam,
    BAlarmSourceExt alarmExt,
    BComponent target)
  {
    // TODO check that the property type matches the event type?
    // int objType = getRemoteObjectType();

    alarmExt = alarmExt == null ? new BAlarmSourceExt() : alarmExt;
    configureAlarmExt(eventParam, alarmExt);
    configureChangeOfStatusFlagsOffnormal(eventParam, alarmExt);
    configureGeneralFaultAlgorithm(eventParam, alarmExt);
    addExtIfMissing(alarmExt, target);
    updateDescriptor(alarmExt);
  }

  private void configureChangeOfStatusFlagsOffnormal(BBacnetEventParameter eventParam, BAlarmSourceExt alarmExt)
  {
    BOffnormalAlgorithm offnormalAlgorithm = alarmExt.getOffnormalAlgorithm();
    if (offnormalAlgorithm instanceof BBacnetStatusAlgorithm)
    {
      configureChangeOfStatusFlagsOffnormal(eventParam, (BBacnetStatusAlgorithm) offnormalAlgorithm);
    }
    else
    {
      if (logger.isLoggable(Level.FINE))
      {
        logger.fine(getObjectId() + ": replacing offnormal algorithm of type " + offnormalAlgorithm.getType() +
          " with BBacnetStatusAlgorithm for event type " + BBacnetEventType.tag(eventParam.getChoice()));
      }
      BBacnetStatusAlgorithm changeOfStatusFlagsAlgorithm = new BBacnetStatusAlgorithm();
      configureChangeOfStatusFlagsOffnormal(eventParam, changeOfStatusFlagsAlgorithm);
      alarmExt.set(
        BAlarmSourceExt.offnormalAlgorithm,
        changeOfStatusFlagsAlgorithm,
        BLocalBacnetDevice.getBacnetContext());
    }
  }

  private static void configureChangeOfStatusFlagsOffnormal(BBacnetEventParameter eventParam, BBacnetStatusAlgorithm algorithm)
  {
    algorithm.set(
      BBacnetStatusAlgorithm.alarmValues,
      eventParam.get(BBacnetEventParameter.STATUS_FLAGS_SLOT_NAME),
      BLocalBacnetDevice.getBacnetContext());
  }

  //endregion

  //region ChangeOfDiscreteValue

  private void configureChangeOfDiscreteValueExt(
    BBacnetEventParameter eventParam,
    BBacnetDeviceObjectPropertyReference objPropRef,
    BPointExtension pointExt,
    BComponent target)
      throws EventEnrollmentException
  {
    checkChangeOfDiscreteValueTarget(objPropRef);

    BAlarmSourceExt alarmExt = updateToAlarmExt(pointExt);
    configureAlarmExt(eventParam, alarmExt);
    configureChangeOfDiscreteValue(eventParam, alarmExt);
    addExtIfMissing(alarmExt, target);
    updateDescriptor(alarmExt);
  }

  private static void checkChangeOfDiscreteValueTarget(BBacnetDeviceObjectPropertyReference objPropRef)
    throws EventEnrollmentException
  {
    PropertyInfo propInfo = getPropertyInfo(objPropRef.getObjectId().getObjectType(), objPropRef.getPropertyId());
    if (propInfo != null)
    {
      switch (propInfo.getAsnType())
      {
        case ASN_BOOLEAN:
        case ASN_UNSIGNED:
        case ASN_INTEGER:
        case ASN_ENUMERATED:
        case ASN_CHARACTER_STRING:
        case ASN_OCTET_STRING:
        case ASN_DATE:
        case ASN_TIME:
        case ASN_OBJECT_IDENTIFIER:
          return;
        case ASN_CONSTRUCTED_DATA:
          if (propInfo.getType().equals(BBacnetDateTime.TYPE.getTypeSpec().toString()))
          {
            return;
          }
      }
    }

    throw new EventEnrollmentException(
      "Change_of_Discrete_Value extension only supports the following data types: " +
        "Boolean, Unsigned, Integer, Enumerated, CharacterString, OctetString, Date, Time, BACnetObjectIdentifier, BACnetDateTime;" +
        "; objectPropertyReference: " + objPropRef + ", property info: " + propInfo,
      new NErrorType(BBacnetErrorClass.PROPERTY, BBacnetErrorCode.VALUE_OUT_OF_RANGE));
  }

  private void configureChangeOfDiscreteValue(BBacnetEventParameter eventParam, BAlarmSourceExt alarmExt)
  {
    BOffnormalAlgorithm offnormalAlgorithm = alarmExt.getOffnormalAlgorithm();
    if (!(offnormalAlgorithm instanceof BBacnetChangeOfDiscreteValueAlgorithm))
    {
      if (logger.isLoggable(Level.FINE))
      {
        logger.fine(getObjectId() + ": replacing offnormal algorithm of type " + offnormalAlgorithm.getType() +
          " with ChangeOfDiscreteValueAlgorithm for event type " + BBacnetEventType.tag(eventParam.getChoice()));
      }
      offnormalAlgorithm = new BBacnetChangeOfDiscreteValueAlgorithm();
      alarmExt.set(BAlarmSourceExt.offnormalAlgorithm, offnormalAlgorithm, BLocalBacnetDevice.getBacnetContext());
    }
  }

  //endregion

  //endregion

  //region Spy

  @Override
  public void spy(SpyWriter out)
    throws Exception
  {
    super.spy(out);

    out.startProps();
    out.trTitle("BacnetEventEnrollmentDescriptor", 2);
    out.prop("pointExt", pointExt);
    out.prop("oldId", oldId);
    out.prop("oldName", oldName);
    out.prop("configOk", configOk);
    out.prop("duplicate", duplicate);
    out.prop("typeOfEvent", getTypeOfEvent());
    out.prop("notificationClass", getNotificationClass());
    out.endProps();
  }

  //endregion

  //region Utility

  /**
   * Is the property referenced by this propertyId an array property?
   */
  private static boolean isArray(int propId)
  {
    for (int arrayPropId : ARRAY_PROPS)
    {
      if (propId == arrayPropId)
      {
        return true;
      }
    }

    return false;
  }

  private static final int[] ARRAY_PROPS = {
    BBacnetPropertyIdentifier.EVENT_TIME_STAMPS,
    BBacnetPropertyIdentifier.EVENT_MESSAGE_TEXTS,
    BBacnetPropertyIdentifier.EVENT_MESSAGE_TEXTS_CONFIG,
    BBacnetPropertyIdentifier.PROPERTY_LIST,
  };

  private BComponent resolveTarget(BBacnetDeviceObjectPropertyReference objPropRef)
  {
    if (!BacnetDescriptorUtil.isValid(objPropRef))
    {
      if (logger.isLoggable(Level.FINE))
      {
        logger.fine(this + ": objectPropertyReference (" + objPropRef + ") is not valid");
      }
      return null;
    }

    try
    {
      int objectType = objPropRef.getObjectId().getObjectType();
      int propId = objPropRef.getPropertyId();
      BComponent target;
      if (isLocalDevice(objPropRef.getDeviceId().getInstanceNumber()))
      {
        switch (objectType)
        {
          case BBacnetObjectType.TREND_LOG:
            if (propId != BBacnetPropertyIdentifier.LOG_BUFFER)
            {
              if (logger.isLoggable(Level.FINE))
              {
                logger.fine(this + ": references to Trend Log objects on the local device" +
                  " may only reference the Log Buffer property" +
                  "; objectPropertyReference: " + objPropRef);
              }
              return null;
            }

            target = BacnetDescriptorUtil.findLocalObject(objPropRef.getObjectId());
            break;

          case BBacnetObjectType.ANALOG_INPUT:
          case BBacnetObjectType.ANALOG_OUTPUT:
          case BBacnetObjectType.ANALOG_VALUE:
          case BBacnetObjectType.MULTI_STATE_INPUT:
          case BBacnetObjectType.MULTI_STATE_OUTPUT:
          case BBacnetObjectType.MULTI_STATE_VALUE:
          case BBacnetObjectType.LOOP:
          case BBacnetObjectType.CHARACTER_STRING_VALUE:
          case BBacnetObjectType.LARGE_ANALOG_VALUE:
          case BBacnetObjectType.INTEGER_VALUE:
          case BBacnetObjectType.POSITIVE_INTEGER_VALUE:
            if (propId != BBacnetPropertyIdentifier.PRESENT_VALUE)
            {
              if (logger.isLoggable(Level.FINE))
              {
                logger.fine(this + ": references to object type " + BBacnetObjectType.tag(objectType) +
                  " on the local device may only reference the Present Value property" +
                  "; objectPropertyReference: " + objPropRef);
              }
              return null;
            }

            target = findOrAddLocalPoint(objPropRef.getObjectId(), objPropRef.getPropertyId(), objPropRef.getPropertyArrayIndex());
            break;

          case BBacnetObjectType.BINARY_INPUT:
          case BBacnetObjectType.BINARY_OUTPUT:
          case BBacnetObjectType.BINARY_VALUE:
            if (propId != BBacnetPropertyIdentifier.PRESENT_VALUE &&
                propId != BBacnetPropertyIdentifier.ELAPSED_ACTIVE_TIME)
            {
              if (logger.isLoggable(Level.FINE))
              {
                logger.fine(this + ": references to object type " + BBacnetObjectType.tag(objectType) +
                  " on the local device may only reference the Present Value and Elapsed Active Time properties" +
                  "; objectPropertyReference: " + objPropRef);
              }
              return null;
            }

            target = findOrAddLocalPoint(objPropRef.getObjectId(), objPropRef.getPropertyId(), objPropRef.getPropertyArrayIndex());
            break;
            
          default:
            if (logger.isLoggable(Level.FINE))
            {
              logger.fine(this + ": references to object type " + BBacnetObjectType.tag(objectType) +
                " are not supported on the local device; objectPropertyReference: " + objPropRef);
            }
            return null;
        }
      }
      else
      {
        // Remote device
        if (objectType == BBacnetObjectType.TREND_LOG)
        {
          if (logger.isLoggable(Level.FINE))
          {
            logger.fine(this + ": references to Trend Log objects are not supported on remote devices" +
              "; objectPropertyReference: " + objPropRef);
          }
          return null;
        }

        target = findOrAddRemotePoint(objPropRef);
      }

      if (target == null)
      {
        if (logger.isLoggable(Level.FINE))
        {
          logger.fine(getObjectId() + ": did not resolve objectPropertyReference " + objPropRef);
        }
      }
      return target;
    }
    catch (Exception e)
    {
      logException(
        Level.SEVERE,
        new StringBuilder(getObjectId().toString())
          .append(": could not resolve target for objectPropertyReference ")
          .append(objPropRef),
        e);
      return null;
    }
  }

  private void updateDescriptor(BPointExtension pointExt)
  {
    this.pointExt = pointExt;

    Context context = BLocalBacnetDevice.getBacnetContext();
    set(eventEnrollmentOrd, pointExt.getHandleOrd(), context);

    // TODO Add support for internal reliability evaluation
    //if (getNotificationClass(pointExt) == null)
    //{
    //  // Notification class ID could not be used to set the alarm class on the point ext
    //  set(reliability, BBacnetReliability.configurationError, context);
    //}
    //else
    //{
    //  set(reliability, BBacnetReliability.noFaultDetected, context);
    //}
  }

  private void resetDescriptor()
  {
    BPointExtension pointExt = this.pointExt;
    if (pointExt != null)
    {
      BComplex parent = pointExt.getParent();
      if (parent instanceof BComponent)
      {
        ((BComponent) parent).remove(pointExt);
      }
    }

    this.pointExt = null;

    Context context = BLocalBacnetDevice.getBacnetContext();
    set(eventEnrollmentOrd, BOrd.NULL, context);
    // TODO Add support for internal reliability evaluation
    //set(reliability, BBacnetReliability.configurationError, context);
  }

  private static void logException(Level level, StringBuilder message, Exception e)
  {
    if (logger.isLoggable(Level.FINE))
    {
      logger.log(Level.FINE, message.append("; exception: ").append(e.getLocalizedMessage()).toString(), e);
    }
    else if (logger.isLoggable(level))
    {
      logger.log(level, message.append("; exception: ").append(e.getLocalizedMessage()).toString());
    }
  }

  private static PropertyInfo getPropertyInfo(int objectType, int propertyId)
  {
    return ObjectTypeList.getInstance().getPropertyInfo(objectType, propertyId);
  }

  private static boolean isBinaryPv(PropertyInfo info)
  {
    return "bacnet:BacnetBinaryPv".equals(info.getType());
  }

  private static boolean isArraySize(PropertyInfo propInfo, BBacnetDeviceObjectPropertyReference objPropRef)
  {
    return propInfo.isArray() && objPropRef.getPropertyArrayIndex() == 0;
  }

  //endregion

  //region Fields

  private static final Logger logger = Logger.getLogger("bacnet.export.object.eventEnrollment");

  private static final int[] REQUIRED_PROPS = {
    BBacnetPropertyIdentifier.OBJECT_IDENTIFIER,
    BBacnetPropertyIdentifier.OBJECT_NAME,
    BBacnetPropertyIdentifier.OBJECT_TYPE,
    BBacnetPropertyIdentifier.EVENT_TYPE,
    BBacnetPropertyIdentifier.NOTIFY_TYPE,
    BBacnetPropertyIdentifier.EVENT_PARAMETERS,
    BBacnetPropertyIdentifier.OBJECT_PROPERTY_REFERENCE,
    BBacnetPropertyIdentifier.EVENT_STATE,
    BBacnetPropertyIdentifier.EVENT_ENABLE,
    BBacnetPropertyIdentifier.ACKED_TRANSITIONS,
    BBacnetPropertyIdentifier.NOTIFICATION_CLASS,
    BBacnetPropertyIdentifier.EVENT_TIME_STAMPS,
    BBacnetPropertyIdentifier.EVENT_DETECTION_ENABLE,
    BBacnetPropertyIdentifier.STATUS_FLAGS,
    BBacnetPropertyIdentifier.RELIABILITY
  };

  private static final int[] OPTIONAL_PROPS = {
    BBacnetPropertyIdentifier.DESCRIPTION,
    BBacnetPropertyIdentifier.EVENT_MESSAGE_TEXTS,
    BBacnetPropertyIdentifier.EVENT_MESSAGE_TEXTS_CONFIG,
    BBacnetPropertyIdentifier.EVENT_ALGORITHM_INHIBIT_REF,
    BBacnetPropertyIdentifier.EVENT_ALGORITHM_INHIBIT,
    BBacnetPropertyIdentifier.TIME_DELAY_NORMAL
  };

  private static final PropertyValue[] EMPTY_PROP_VALUE_ARRAY = new PropertyValue[0];

  private static final BBacnetBitString STATUS_FLAGS_DEFAULT = BBacnetBitString.make(new boolean[] {
    /* inAlarm */ false,
    // TODO Add support for reliability in the alarm lifecycle of a BACnet object
    ///* inFault */ true, // reliability is CONFIGURATION_ERROR and not NO_FAULT_DETECTED
    /* inFault */ false,
    /* isOverridden */ false,
    /* isOutOfService */ false });

  // The pointExt is usually instance BAlarmSourceExt except for BufferReady event types, which are
  // BBacnetTrendLogAlarmSourceExt.
  private BPointExtension pointExt;

  // Holds any Event_Enable BACnet property writes until the point ext can be configured.
  private BAlarmTransitionBits eventEnable = BAlarmTransitionBits.DEFAULT;

  // Holds the text values until the point ext can be configured.
  private String toOffnormalText = "";
  private String toFaultText = "";
  private String toNormalText = "";

  // Holds the alarm inhibit info until the point ext can be configured.
  private boolean eventAlgorithmInhibit;
  private BBacnetObjectPropertyReference eventAlgorithmInhibitRef = OBJECT_PROP_REF_UNCONFIGURED;

  // Holds the timeDelayToNormal until the point ext can be configured.
  private int timeDelayNormal;

  private BBacnetNotifyType oldNotifyType;
  private BBacnetObjectIdentifier oldId;
  private String oldName;
  private boolean duplicate;
  private boolean configOk;

  //endregion

  private static class EventEnrollmentException extends Exception
  {
    public EventEnrollmentException(String message, ErrorType errorType)
    {
      super(message);
      this.errorType = errorType;
    }

    final ErrorType errorType;
  }
}
