/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/*jshint devel: true */
/**
 * API Status: **Private**
 * @module nmodule/wiresheet/rc/wb/layout/SnakeGlyphLayoutStrategy
 */
define(['underscore', 'nmodule/wiresheet/rc/wb/layout/AntRaceRoutingStrategy', 'nmodule/wiresheet/rc/wb/layout/SimpleRoutingStrategy', 'nmodule/wiresheet/rc/wb/util/wsUtils'], function (_, AntRaceRoutingStrategy, SimpleRoutingStrategy, wsUtils) {
  'use strict';

  var isEqual = _.isEqual;
  var boundingBox = wsUtils.boundingBox,
    getLinkStartAndEnd = wsUtils.getLinkStartAndEnd;

  ////////////////////////////////////////////////////////////////
  // SnakeGlyphLayoutStrategy
  ////////////////////////////////////////////////////////////////

  /**
   * Strategy for laying out and routing `SnakeGlyph`s. Replicates the logic
   * in the `LinkSnakeGlyph` class in `wiresheet-wb`.
   *
   * @class
   * @alias module:nmodule/wiresheet/rc/wb/layout/SnakeGlyphLayoutStrategy
   * @implements module:nmodule/wiresheet/rc/typedefs~EdgeLayoutStrategy
   */
  var SnakeGlyphLayoutStrategy = function SnakeGlyphLayoutStrategy() {
    this.$simple = new SimpleRoutingStrategy();
    this.$antRace = new AntRaceRoutingStrategy();
  };

  /**
   * Update one `SnakeGlyph` entity.
   *
   * @param {module:nmodule/wiresheet/rc/typedefs~WiresheetTriple} triple the
   * triple containing the `SnakeGlyph`
   * @param {module:nmodule/wiresheet/rc/wb/layout/Mask} mask
   * @returns {boolean} true if layout was updated
   */
  SnakeGlyphLayoutStrategy.prototype.update = function (triple, mask) {
    var predicate = triple.predicate;
    var result = this.$simple.route(triple, mask) || this.$antRace.route(triple, mask);
    var updated;
    if (result) {
      var segments = result.segments,
        layout = result.layout;
      updated = !isEqual(predicate.glyph.segments, segments) || !isEqual(predicate.layout, layout);
      delete predicate.glyph.stubs;
      predicate.glyph.segments = segments;
      predicate.layout = layout;
    } else {
      //could not route; place stubs
      var _getLinkStartAndEnd = getLinkStartAndEnd(triple),
        start = _getLinkStartAndEnd.start,
        end = _getLinkStartAndEnd.end;
      var newStubs;
      if (start && end) {
        var _layout = boundingBox([start, end]);
        var x = _layout.x,
          y = _layout.y;
        delete predicate.glyph.segments;
        newStubs = [{
          x: start.x - x,
          y: start.y - y
        }, {
          x: end.x - x,
          y: end.y - y
        }];
        predicate.layout = _layout;
      }
      updated = !isEqual(newStubs, predicate.glyph.stubs);
      predicate.glyph.stubs = newStubs;
    }
    return updated;
  };
  return SnakeGlyphLayoutStrategy;
});
