/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Danesh Kamal
 */

/**
 * API Status: **Private**
 * @module nmodule/alarm/rc/db/commands/NotesCommand
 */
define(['baja!', 'lex!alarm', 'dialogs', 'bajaux/commands/Command', 'nmodule/webEditors/rc/fe/feDialogs', 'nmodule/alarm/rc/console/commands/NotesCommand', 'css!nmodule/alarm/rc/db/templates/view'], function (baja, lexs, dialogs, Command, feDialogs, ConsoleNotesCommand) {
  'use strict';

  var LEX = lexs[0];

  /**
   * Adds notes to the selected alarm.
   *
   * @class
   * @alias module:nmodule/alarm/rc/db/commands/NotesCommand
   * @param {module:nmodule/alarm/rc/db/DatabaseMaintenance} view The alarm database
   * @param  {Array} [selectedRecords] An optional array of alarm records.
   * maintenance view.
   * @extends {module:bajaux/commands/Command}
   */
  var _NotesCommand = function NotesCommand(view, selectedRecords) {
    Command.call(this, {
      module: 'alarm',
      lex: 'commands.notes',
      func: function func() {
        var records;
        if (selectedRecords !== undefined) {
          records = selectedRecords;
        } else {
          records = view.$getCurrentRecords();
        }
        if (!records.length) {
          return dialogs.showOk(LEX.get('alarm.noRecordSelected')).promise();
        }
        var currentNotes = _NotesCommand.$getCurrentNotes(records),
          value = view.$value,
          isAlarmArchiveSpace;
        if (baja.hasType(value, 'alarm:ArchiveAlarmProvider') || baja.hasType(value, 'alarm:AlarmArchive')) {
          isAlarmArchiveSpace = true;
        }
        return ConsoleNotesCommand.showNotesEditor(records, currentNotes).then(function (notes) {
          // Bail if the user hit cancel.
          if (notes === null) {
            return;
          }
          return getAlarmSpace(isAlarmArchiveSpace).then(function (space) {
            // Add the notes to the server.
            return space.addNoteToAlarms({
              ids: records.map(function (record) {
                return record.getUuid().encodeToString();
              }),
              notes: notes
            });
          }).then(function () {
            // Refresh the page after the notes have been added.
            if (view.pager) {
              view.pager.update();
            }
          });
        });
      }
    });
  };
  function getAlarmSpace(isAlarmArchiveSpace) {
    if (isAlarmArchiveSpace) {
      return baja.Ord.make('alarm:archive').get();
    }
    return baja.Ord.make('alarm:').get();
  }

  /**
   * Return the current notes to use for the alarm records.
   *
   * @inner
   * @private
   *
   * @param  {Array} records An array of alarm records.
   * @returns {String} The current notes to display in any any UI.
   */
  _NotesCommand.$getCurrentNotes = function (records) {
    var notes;
    switch (records.length) {
      case 0:
        notes = '';
        break;
      case 1:
        notes = records[0].getAlarmData().get('notes', '');
        break;
      default:
        notes = LEX.get('alarm.multipleAlarmNotes');
    }
    return notes;
  };
  _NotesCommand.prototype = Object.create(Command.prototype);
  _NotesCommand.prototype.constructor = _NotesCommand;
  return _NotesCommand;
});
