/**
 * @copyright 2017 Tridium, Inc. All Rights Reserved.
 * @author Jeremy Narron
 */

/**
 * API Status: **Private**
 * @module nmodule/alarm/rc/console/fe/DetailsEditor
 */
define(['baja!', 'lex!alarm', 'Promise', 'underscore', 'nmodule/webEditors/rc/fe/CompositeEditor', 'nmodule/webEditors/rc/fe/config/CompositeBuilder', 'nmodule/webEditors/rc/wb/table/model/Column', 'nmodule/webEditors/rc/wb/table/tree/TreeTable', 'nmodule/webEditors/rc/wb/table/tree/TreeTableModel', 'nmodule/webEditors/rc/wb/tree/BajaComplexTreeNode', 'nmodule/webEditors/rc/util/htmlUtils', 'nmodule/webEditors/rc/util/textUtils'], function (baja, lexs, Promise, _, CompositeEditor, CompositeBuilder, Column, TreeTable, TreeTableModel, BajaComplexTreeNode, htmlUtils, textUtils) {
  'use strict';

  var lex = lexs[0],
    nameColumn = new Column('name', {
      displayName: lex.get('alarm.console.details.name')
    }),
    valueColumn = new Column('value', {
      displayName: lex.get('alarm.console.details.value')
    });
  nameColumn.getValueFor = function (row) {
    return row.getTreeNode().$display;
  };
  valueColumn.getValueFor = function (row) {
    var treeNode = row.getTreeNode(),
      slotName = treeNode.getName();
    return treeNode.getParent().value().get(slotName);
  };
  valueColumn.buildCell = function (row, dom) {
    var value = this.getValueFor(row);
    if (typeof value !== 'string') {
      return;
    }
    dom.html(htmlUtils.escapeHtml(value).replace(/\n/g, '<br>'));
  };

  /**
   * This editor requires an instance of
   * {@link nmodule/alarm/rc/console/support/AlarmConsoleSupport} to work
   * properly. Pass this dependency to the constructor as a part of its
   * {@link bajaux/Properties} with the key `support`.
   *
   * API Status: **Private**
   * @class
   * @extends module:nmodule/webEditors/rc/fe/CompositeEditor
   * @alias module:nmodule/alarm/rc/console/fe/DetailsEditor
   */
  var DetailsEditor = function DetailsEditor() {
    CompositeEditor.apply(this, arguments);
    this.$support = this.properties().getValue('support');
  };
  DetailsEditor.prototype = Object.create(CompositeEditor.prototype);
  DetailsEditor.prototype.constructor = DetailsEditor;

  /**
   * @private
   * @param {Object} subject
   * @returns {Promise.<module:nmodule/webEditors/rc/wb/table/tree/TreeTableModel>}
   */
  DetailsEditor.prototype.$makeDetails = function (subject) {
    var support = this.$support,
      instructions = subject.alarmData.instructions || '',
      msgText = subject.alarmData.msgText || '',
      sourceName = subject.alarmData.sourceName || '',
      ignoredKeys = ['timezone', 'value', 'icons', 'ackedCount', 'unackedCount', 'ackPendingCount', 'alarmData'],
      alarmInfo = _.pick(subject, function (value, key) {
        return !_.contains(ignoredKeys, key);
      }),
      alarmData = _.pick(subject._alarmData, function (value, key) {
        switch (key) {
          case 'instructions':
            return instructions !== '';
          case 'msgText':
            return msgText !== '';
          default:
            // If present, also filter out the 'unconvertedHyperlinkOrd' alarm data facet since it
            // is only present when the 'hyperlinkOrd' alarm data facet has been converted to
            // nspace ORD form (see NCCB-54949). It's only needed for framework use (to stash
            // away the original hyperlink ORD) and therefore not needed to display to the user.
            return !_.contains(['notes', 'unconvertedHyperlinkOrd'], key) && value !== '';
        }
      }),
      detailsComponent = baja.$('baja:Component'),
      alarmDataComponent = baja.$('baja:Component'),
      alarmDataKeys = _.keys(alarmData),
      val;
    alarmDataKeys.sort(function (str1, str2) {
      return str1.toLowerCase().localeCompare(str2.toLowerCase());
    });
    if (instructions) {
      alarmData.instructions = instructions;
    }
    if (msgText) {
      alarmData.msgText = msgText;
    }
    alarmData.sourceName = sourceName;
    _.each(_.keys(alarmInfo), function (key) {
      switch (key) {
        case 'timestamp':
        case 'normalTime':
        case 'ackTime':
        case 'lastUpdate':
          val = support.createDateString(subject[key], subject.alarmData.TimeZone);
          break;
        case 'source':
          var sourceName = alarmData.sourceName,
            sourceOrd = baja.SlotPath.unescape(subject[key]);
          val = sourceName ? sourceName + '\n' + sourceOrd : sourceOrd;
          break;
        case 'priority':
          val = baja.Integer.make(subject[key]).toString();
          break;
        default:
          val = String(subject[key]);
      }

      // Skip anything that starts with an underscore.
      if (key.indexOf("_") !== 0) {
        detailsComponent.add({
          slot: key,
          value: val,
          cx: {
            displayName: lex.get({
              key: 'alarm.console.columns.' + key,
              def: textUtils.toFriendly(key)
            })
          }
        });
      }
    });
    _.each(alarmDataKeys, function (key) {
      if (key === 'Count') {
        val = baja.Integer.make(alarmData[key]).toString();
      } else {
        val = String(alarmData[key]);
      }
      alarmDataComponent.add({
        slot: key,
        value: val,
        cx: {
          displayName: lex.get({
            key: 'alarm.console.columns.' + key,
            def: textUtils.toFriendly(key)
          })
        }
      });
    });
    detailsComponent.add({
      slot: 'alarmData',
      value: alarmDataComponent,
      cx: {
        displayName: lex.get({
          key: 'alarm.console.columns.alarmData',
          def: textUtils.toFriendly('alarmData')
        })
      }
    });
    return TreeTableModel.make({
      node: new BajaComplexTreeNode(detailsComponent),
      columns: [nameColumn, valueColumn]
    });
  };

  /**
   * @returns {module:nmodule/webEditors/rc/fe/config/CompositeBuilder}
   */
  DetailsEditor.prototype.makeBuilder = function () {
    var that = this,
      builder = new CompositeBuilder();
    builder.getKeys = function () {
      return ['details'];
    };
    builder.getValueFor = function (key) {
      if (key === 'details') {
        return that.$makeDetails(this.getDataSource());
      }
    };
    builder.getConfigFor = function (key) {
      if (key === 'details') {
        return {
          formFactor: 'max',
          type: TreeTable
        };
      }
    };
    builder.getDomFor = function (key) {
      return that.jq().children('.js-' + key);
    };
    return builder;
  };

  /**
   * @param {JQuery} dom
   */
  DetailsEditor.prototype.doInitialize = function (dom) {
    dom.addClass('DetailsEditor');
    dom.html('<table class="js-details ux-table"/>');
    return CompositeEditor.prototype.doInitialize.apply(this, arguments);
  };
  return DetailsEditor;
});
