/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/* eslint-env browser */

/**
 * @module nmodule/js/rc/switchboard/switchboard
 */
define(['Promise', 'underscore'], function (Promise, _) {
  'use strict';

  var NO_KEY = '__NO_KEY__';

  // eslint-disable-next-line promise/avoid-new
  var neverSettled = new Promise(_.noop);

  ////////////////////////////////////////////////////////////////
  // switchboard()
  ////////////////////////////////////////////////////////////////

  /**
   * `switchboard` is a way of orchestrating multiple calls to asynchronous,
   * Promise-based instance methods on an object. Use it when you want to
   * define constraints on which async methods such as:
   *
   * - can these methods be called concurrently?
   * - should these methods queue up and wait for the other to finish?
   * - are there inter-dependencies between these methods; i.e., should these
   *   methods never be called out of order?
   *
   * The advantage of using `switchboard` is that these constraints can be
   * defined in a declarative way, rather than having to keep "instance
   * promises" and manually checking their status.
   *
   * Please note: if using `switchboard` to manage a function that sometimes
   * returns a promise, sometimes doesn't:
   *
   * 1. Don't do that (either always return a promise, or never), and
   * 2. it will change the behavior to always return a promise. Don't rely on
   *    the sometimes-sync behavior of a function you manage with `switchboard`.
   *
   * @alias module:nmodule/js/rc/switchboard/switchboard
   * @param {Object|Function} obj The object whose promise-based instance
   * methods we want to regulate; or a single function to regulate
   * @param {Object} rules This can be an array of rule objects,
   * each with a `method` property, or an object literal mapping method names
   * to rule objects.
   * @returns {{release: Function}|Function} an object that can be used to
   * reverse the switchboard on an object; if the argument was a function, this
   * returns an irreversibly switchboarded version of that function
   *
   * @example
   *   <caption>load() must never be called before initialize().</caption>
   *
   *   switchboard(widget, {
   *     load: { onlyAfter: 'initialize' }
   *   });
   *
   * @example
   *   <caption>initialize() must never be called after destroy().</caption>
   *
   *   switchboard(widget, {
   *     initialize: { notAfter: 'destroy' }
   *   });
   *
   * @example
   *   <caption>I should never have multiple calls to save() executing at once.
   *   If save() is called a second time before the first promise resolves,
   *   consider them one operation: the second call will "fold into" the first,
   *   the first promise will be returned, and so the work will only be done
   *   once. It can be started again after the first call resolves. This would
   *   protect against a user double-clicking the save button when they meant to
   *   single-click.</caption>
   *
   *   switchboard(widget, {
   *     save: { allow: 'oneAtATime', onRepeat: 'returnLast' }
   *   });
   *
   * @example
   *   <caption>Similar to the above, but calling save() while another call
   *   is already executing will cause the second call to reject.</caption>
   *
   *   switchboard(widget, {
   *     save: { allow: 'oneAtATime', onRepeat: 'reject' }
   *   });
   *
   * @example
   *   <caption>destroy() is a one-time operation - period. Any subsequent calls
   *   after the first will just return the same promise as the first call.
   *   </caption>
   *
   *   switchboard(widget, {
   *     destroy: { allow: 'once', onRepeat: 'returnLast' }
   *   });
   *
   * @example
   *   <caption>advancePage() is legit to call multiple times, but should not
   *   have more than one call concurrently executing. If I click the button
   *   five times, I want to see five page turns, one after the other. This
   *   ensures that calls queue up nicely, one after the other.</caption>
   *
   *   switchboard(widget, {
   *     advancePage: { allow: 'oneAtATime', onRepeat: 'queue' }
   *   });
   *
   * @example
   *   <caption>When the user selects a new row in the table, I call up to the
   *   server for some information for that row and display it. Say the user
   *   clicks row 1, then clicks rows 2, 3, and 4 before the response for 1
   *   comes back. There's no reason to make the network calls for 2 and 3.
   *   "preempt" causes every concurrent call except the first and last to be
   *   aborted.</caption>
   *
   *   switchboard(widget, {
   *     getRowDetails: { allow: 'oneAtATime', onRepeat: 'preempt' }
   *   });
   *
   * @example
   *   <caption>My widget has an expanded state and a collapsed state, but the
   *   question of "what do I do if the user tries to collapse it when it hasn't
   *   finished expanding" isn't a problem I really want to deal with. This
   *   groups calls to setExpanded together based on the arguments, ensuring
   *   that the widget is either expanding or collapsing, never both at the
   *   same time.</caption>
   *
   *   MyWidget.prototype.setExpanded = function (expanded) {};
   *
   *   switchboard(widget, {
   *     setExpanded: {
   *       allow: 'oneAtATime',
   *       onRepeat: 'returnLast',
   *       keyedOn: function (expanded) { return expanded; }
   *   });
   *
   * @example
   *   <caption>My object has different read/write functions. Read operations
   *   are fine to operate in parallel, but to ensure data integrity, I want to
   *   ensure that a read and a write, or two writes, do not happen at the same
   *   time.</caption>
   *
   *   switchboard(database, {
   *     read: { notWhile: 'put,delete' },
   *     search: { notWhile: 'put,delete' },
   *     put: { notWhile: 'read,search,delete' },
   *     delete: { notWhile: 'read,search,put' }
   *   });
   *
   * @example
   *   <caption>Switchboard just one function to ensure only one call to it
   *   can be concurrent at one time.</caption>
   *
   *   var saveDatabase = switchboard(function () {
   *     //prevent two concurrent calls to db.save()
   *     return db.save();
   *   }, { allow: 'oneAtATime', onRepeat: 'queue' });
   */
  var switchboard = function switchboard(obj, rules) {
    var isFunction = typeof obj === 'function';
    if (isFunction) {
      obj = {
        exec: obj
      };
      rules = {
        exec: rules
      };
    }
    if (!_.isArray(rules)) {
      rules = _.map(rules, function (rule, method) {
        return _.extend({}, rule, {
          method: method
        });
      }); //object to array
    }
    var trackers = {};
    _.each(rules, function (ruleObj) {
      addTrackedRules(trackers, obj, ruleObj);
    });
    if (isFunction) {
      return obj.exec;
    } else {
      return {
        release: function release() {
          _.invoke(trackers, 'release');
        }
      };
    }
  };

  ////////////////////////////////////////////////////////////////
  // Support classes
  ////////////////////////////////////////////////////////////////

  /**
   * Represents one call to a tracked function.
   *
   * @inner
   * @class
   * @param {Object} ctx the object on which the function is called
   * @param {Function} method the (original, unwrapped) function to be called
   * @param {Array} args arguments to the function
   */
  var Invocation = function Invocation(ctx, method, args) {
    var that = this,
      promise;
    that.args = args;
    that.$prereqs = [];

    // eslint-disable-next-line promise/avoid-new
    that.promise = new Promise(function (resolve, reject) {
      /**
       * Invoke the tracked function.
       * @returns {Promise} promise to be resolved when the function call is
       */
      that.invoke = function () {
        if (!promise) {
          promise = that.satisfyPrereqs().then(function () {
            return method.apply(ctx, args);
          }).then(resolve, reject);
        }
        return that.promise;
      };
    });
  };

  /**
   * Satisfy all prerequisites. If more were added during the process of
   * satisfying the existing ones, this function will call itself again to
   * ensure all are satisfied.
   *
   * @returns {Promise}
   */
  Invocation.prototype.satisfyPrereqs = function () {
    var that = this,
      prereqs = that.$prereqs;
    return Promise.all(prereqs).then(function (results) {
      if (results.length < prereqs.length) {
        return that.satisfyPrereqs();
      }
    });
  };

  /**
   * Add a prerequisite to invoking this function. When `invoke()` is called,
   * all of these prerequisite promises must resolve before the actual function
   * call happens.
   *
   * @param {Promise} prereq promise that must resolve before the original
   * function is invoked
   */
  Invocation.prototype.addPrereq = function (prereq) {
    this.$prereqs.push(prereq);
  };

  /**
   * A rule for invoking a tracked function. Rules apply to a function that is
   * invoked, and have their behavior defined by a function that is watched.
   * e.g., function A (the invoked function) will always reject if function B
   * (the watched function) has already been called.
   *
   * @inner
   * @class
   * @param {Tracker} watchedTracker
   * @param {Object} config
   * @param {String} config.method the name of the tracked function
   * @param {String} [config.onRepeat='queue'] the way this rule should alter
   * behavior when a repeat condition is detected. See `#prepare()`.
   * @param {Function} [config.keyedOn] a function that receives the arguments
   * to the original function call and returns a `String` that indicates how
   * this call should be grouped. This allows you to define behavior for
   * multiple calls to a single function. Only applies to `once` and
   * `oneAtATime`.
   */
  var Rule = function Rule(watchedTracker, config) {
    this.$watchedTracker = watchedTracker;
    this.$name = config.method;
    this.$onRepeat = config.onRepeat || 'queue';
    this.$keyedOn = config.keyedOn;
  };

  /**
   * Get a key for the given invocation, used for grouping together calls when
   * using `once` and `oneAtATime`.
   *
   * @param {Tracker} tracker
   * @param {Invocation} inv
   * @returns {*}
   */
  Rule.prototype.getKey = function (tracker, inv) {
    var keyedOn = this.$keyedOn;
    return keyedOn ? keyedOn.apply(this, inv.args) : NO_KEY;
  };

  /**
   * Perform preparations for doing the given function call based on this rule's
   * configured behavior.
   *
   * @param {Tracker} invokedTracker
   * @param {Invocation} inv
   * @returns {boolean} return `false` if this invocation is 'squelched'
   * (we rejected it, swapped in a different promise, etc.) meaning the original
   * function will not be called.
   */
  Rule.prototype.prepare = function (invokedTracker, inv) {
    return this[this.$onRepeat](invokedTracker, inv);
  };

  /**
   * For `onRepeat: queue`, will ensure that all outstanding function calls
   * settle before invoking the function. Prevents multiple concurrent calls
   * to the same function, instead they queue up 'single file.'
   *
   * @param {Tracker} invokedTracker
   * @param {Invocation} inv
   */
  Rule.prototype.queue = function (invokedTracker, inv) {
    var that = this,
      tracker = that.$watchedTracker,
      key = that.getKey(tracker, inv),
      matching = _.filter(tracker.$queue, function (otherInv) {
        return that.getKey(tracker, otherInv) === key;
      });
    inv.addPrereq(Promise.all(_.map(matching, function (obj) {
      return obj.promise["catch"](function (ignore) {});
    })));
  };
  function withSameKey(rule, inv) {
    var t = rule.$watchedTracker;
    return function (o) {
      return rule.getKey(t, o) === rule.getKey(t, inv);
    };
  }

  /**
   * A rule for preventing a function from executing while some function (which
   * might be the same function) is also executing. A sort of 'synchronization'
   * which prevents different async functions, which might mess with the same
   * state, from stepping on each other's toes.
   *
   * @inner
   * @class
   * @extends Rule
   */
  var NotWhileRule = function NotWhileRule() {
    Rule.apply(this, arguments);
  };
  NotWhileRule.prototype = Object.create(Rule.prototype);

  /**
   * For `onRepeat: abort`, this causes the invoked function to never execute,
   * and the returned promise to never settle. It vanishes into the ether.
   *
   * @param {Tracker} invokedTracker
   * @param {Invocation} inv
   * @returns {boolean} `false` if we cancelled the execution because the other
   * function is currently executing.
   */
  NotWhileRule.prototype.abort = function (invokedTracker, inv) {
    if (_.find(this.$watchedTracker.$queue, withSameKey(this, inv))) {
      inv.addPrereq(neverSettled);
      return false;
    }
  };

  /**
   * For `onRepeat: reject`, this causes the invoked function to reject its
   * promise.
   *
   * @param {Tracker} invokedTracker
   * @param {Invocation} inv
   * @returns {boolean} `false` if we rejected the promise because the other
   * function is currently executing.
   */
  NotWhileRule.prototype.reject = function (invokedTracker, inv) {
    var watchedTracker = this.$watchedTracker;
    if (_.find(watchedTracker.$queue, withSameKey(this, inv))) {
      inv.addPrereq(Promise.reject(new Error('NotWhileRule: rejecting since ' + 'a call to method ' + watchedTracker.$name + ' is ongoing')));
      return false;
    }
  };

  /**
   * For `onRepeat: returnLast`, this cancels execution of the tracked function,
   * instead just returning the same promise for the previous call. Good for
   * 'throttling' behavior.
   *
   * This is valid only for `allow: oneAtATime`, not `notWhile`.
   *
   * @param {Tracker} invokedTracker
   * @param {Invocation} inv
   * @returns {boolean} `false` if we cancelled the execution because the
   * previous call to the other function is still executing.
   */
  NotWhileRule.prototype.returnLast = function (invokedTracker, inv) {
    var that = this,
      tracker = this.$watchedTracker,
      q = tracker.$queue,
      last = q[q.length - 1];
    if (last) {
      var key = that.getKey(tracker, inv);
      if (that.getKey(tracker, last) === key) {
        inv.promise = last.promise;
        inv.invoke = _.constant(last.promise);
        return false;
      } else {
        inv.addPrereq(last.promise);
      }
    }
  };

  /**
   * For `onRepeat: preempt`, this cancels execution of all concurrent calls
   * to the tracked function except for the first and last one. Dropped calls
   * will never settle a la `onRepeat: abort`. Use this for throttling a
   * function such as a window resize or button click.
   *
   * This is valid only for `allow: oneAtATime`, not `notWhile`.
   *
   * @param {Tracker} invokedTracker
   * @param {Invocation} inv
   */
  NotWhileRule.prototype.preempt = function (invokedTracker, inv) {
    var q = invokedTracker.$queue;
    if (q.length) {
      inv.addPrereq(q[0].promise);
      for (var i = 1; i < q.length; i++) {
        q[i].addPrereq(neverSettled);
      }
      q.splice(1, q.length);
    }
  };

  /**
   * A rule for preventing a function from executing if some other function
   * (which might be the same function) was already called.
   *
   * @inner
   * @class
   * @extends Rule
   */
  var NotAfterRule = function NotAfterRule(watchedTracker, config) {
    Rule.call(this, watchedTracker, _.extend(config, {
      onRepeat: config.onRepeat || 'reject'
    }));
  };
  NotAfterRule.prototype = Object.create(Rule.prototype);

  /**
   * For `onRepeat: abort`, this causes the invoked function to never execute,
   * and the returned promise to never settle. It vanishes into the ether.
   *
   * @param {Tracker} invokedTracker
   * @param {Invocation} inv
   * @returns {boolean} `false` if we cancelled the execution because the other
   * function was already called.
   */
  NotAfterRule.prototype.abort = function (invokedTracker, inv) {
    var last = this.$watchedTracker.lastInvForRule(this, inv);
    if (last) {
      inv.invoke = _.constant(neverSettled);
      return false;
    }
  };

  /**
   * For `onRepeat: returnLast`, this cancels execution of the tracked function,
   * instead just returning the same promise for the previous call.
   *
   * This is only valid for `allow: once`, not `notAfter`.
   *
   * @param {Tracker} invokedTracker
   * @param {Invocation} inv
   * @returns {boolean} `false` if we cancelled the execution because the
   * other function was already called.
   */
  NotAfterRule.prototype.returnLast = function (invokedTracker, inv) {
    var last = invokedTracker.lastInvForRule(this, inv);
    if (last) {
      inv.invoke = _.constant(last.promise);
      return false;
    }
  };

  /**
   * For `onRepeat: reject`, this causes the invoked function to reject its
   * promise.
   *
   * @param {Tracker} invokedTracker
   * @param {Invocation} inv
   * @returns {boolean} `false` if we rejected the promise because the other
   * function was already called.
   */
  NotAfterRule.prototype.reject = function (invokedTracker, inv) {
    var tracker = this.$watchedTracker,
      last = tracker.lastInvForRule(this, inv);
    if (last) {
      inv.addPrereq(Promise.reject(new Error('NotAfterRule: rejecting since ' + 'method ' + tracker.$name + ' was already called')));
      return false;
    }
  };

  /**
   * A rule for preventing execution if some other function has not yet been
   * called. e.g., `load onlyAfter initialize`.
   *
   * @inner
   * @class
   * @extends Rule
   */
  var OnlyAfterRule = function OnlyAfterRule(name, watchedTracker, onRepeat) {
    Rule.apply(this, arguments);
  };
  OnlyAfterRule.prototype = Object.create(Rule.prototype);

  /**
   * `onRepeat` does not apply to `onlyAfter`: it only rejects. It will ensure
   * that the call to the other function fully resolves, if it has not yet,
   * before allowing the function to execute.
   *
   * @param {Tracker} invokedTracker
   * @param {Invocation} inv
   * @returns {boolean} `false` if we rejected the promise because the other
   * function has not yet been called.
   */
  OnlyAfterRule.prototype.prepare = function (invokedTracker, inv) {
    var watchedTracker = this.$watchedTracker,
      last = watchedTracker.lastInvForRule(this, inv),
      name = watchedTracker.$name;
    if (last) {
      inv.addPrereq(last.promise);
    } else {
      inv.addPrereq(Promise.reject(new Error('OnlyAfterRule: rejecting since ' + 'method ' + name + ' was not called yet')));
      return false;
    }
  };

  /**
   * Manages calls to a tracked function, ensuring that all calls to it
   * satisfy the various rules configured on the switchboard.
   *
   * @inner
   * @class
   */
  var Tracker = function Tracker(name) {
    this.$name = name;
    this.$queue = [];
    this.$rules = [];
    //TODO: delete key when invocation has resolved
    this.$lastInvByKey = {};
  };

  /**
   * Add a new rule to the set of rules.
   * @param {Rule} rule
   */
  Tracker.prototype.addRule = function (rule) {
    this.$rules.push(rule);
  };

  /**
   * Put a call to the tracked function on the queue. Give all rules configured
   * on this function a chance to verify that it can be called and resolved,
   * and attempt to invoke it.
   *
   * @param {Invocation} inv
   * @returns {Promise} promise to be resolved when the invocation completes.
   * Note that this promise may not come from the original function, but may be
   * swapped out for a rejection or a different value by the configured rules.
   */
  Tracker.prototype.enqueue = function (inv) {
    var that = this,
      queue = that.$queue,
      rules = that.$rules,
      rejectedRules;
    rejectedRules = _.filter(rules, function (rule) {
      if (rule.prepare(that, inv) !== false) {
        that.$lastInvByKey[rule.getKey(that, inv)] = inv;
      } else {
        return true;
      }
    });
    if (!rejectedRules.length) {
      that.$lastInvByKey[NO_KEY] = inv;
    }
    queue.push(inv);
    inv.promise = inv.promise["finally"](function () {
      queue.splice(_.indexOf(queue, inv), 1);
    });
    return inv.invoke();
  };

  /**
   * Get the last invocation that matches the rule (even if it has resolved
   * and dropped off the active queue). Needed for `once` and `notAfter`.
   *
   * @param {Rule} rule
   * @param {Invocation} inv
   * @returns {Invocation}
   */
  Tracker.prototype.lastInvForRule = function (rule, inv) {
    return this.$lastInvByKey[rule.getKey(this, inv)];
  };

  /**
   * Get or attach a tracker to the specified instance method.
   *
   * @inner
   * @param {Object} trackers map of trackers for this object
   * @param {Object} obj the instance we're attaching a switchboard to
   * @param {String} method the method name
   * @returns {Tracker} the (newly created, or already existing) tracker for
   * the instance method
   */
  function track(trackers, obj, method) {
    var existing = trackers[method];
    if (existing) {
      return existing;
    }
    var tracker = new Tracker(method),
      _method = obj[method];
    tracker.$name = method;
    obj[method] = function () {
      var args = Array.prototype.slice.call(arguments);
      return tracker.enqueue(new Invocation(obj, _method, args));
    };
    tracker.release = function () {
      obj[method] = _method;
    };
    return trackers[method] = tracker;
  }

  /**
   * Set up trackers and rules on the object, as specified by the original call
   * to `switchboard()`.
   *
   * @inner
   * @param {Object} trackers map of trackers for this object
   * @param {Object} obj the instance we're attaching a switchboard to
   * @param {Object} ruleObj the rule config for this method that was passed to
   * `switchboard()`
   */
  function addTrackedRules(trackers, obj, ruleObj) {
    var method = ruleObj.method;
    if (!method || !obj[method]) {
      throw new Error('valid method required');
    }
    var tracker = track(trackers, obj, method),
      allow = ruleObj.allow,
      keyedOn = ruleObj.keyedOn,
      notAfter = ruleObj.notAfter,
      notWhile = ruleObj.notWhile,
      onlyAfter = ruleObj.onlyAfter,
      onRepeat = ruleObj.onRepeat;
    if (onRepeat === 'returnLast' && notAfter) {
      throw new Error('onRepeat: returnLast not allowed with notAfter');
    }
    if (onRepeat === 'preempt' && allow !== 'oneAtATime') {
      throw new Error('onRepeat: preempt only allowed with allow: oneAtATime');
    }
    if (allow === 'once') {
      tracker.addRule(new NotAfterRule(track(trackers, obj, method), ruleObj));
    } else if (allow === 'oneAtATime') {
      tracker.addRule(new NotWhileRule(track(trackers, obj, method), ruleObj));
    } else if (keyedOn && !notWhile) {
      throw new Error('keyedOn only allowed with once or oneAtATime');
    }
    if (onlyAfter) {
      tracker.addRule(new OnlyAfterRule(track(trackers, obj, onlyAfter), ruleObj));
    }
    if (notAfter) {
      tracker.addRule(new NotAfterRule(track(trackers, obj, notAfter), ruleObj));
    }
    if (notWhile) {
      if (onRepeat === 'returnLast') {
        console.error('onRepeat: returnLast with notWhile may cause ' + 'unexpected results. Did you mean onRepeat: \'queue\'?');
      }
      notWhile.split(',').forEach(function (method) {
        tracker.addRule(new NotWhileRule(track(trackers, obj, method), ruleObj));
      });
    }
  }
  return switchboard;
});
