/*jshint browser: true */
/*global chrome: false */

/**
 * Relevant Dropdowns plugin from css-tricks.com:
 * http://css-tricks.com/relevant-dropdowns-polyfill-for-datalist/
 * 
 * With browser detection functions from the ruanmer fork:
 * https://github.com/ruanmer/Relevant-Dropdowns
 * 
 * Modified for use with bajaux/Workbench by Tridium
 *
 * @module nmodule/js/rc/jquery/Relevant-Dropdowns/js/jquery.relevant-dropdown
 */
define(['jquery',
        'css!nmodule/js/rc/jquery/Relevant-Dropdowns/css/style'], function (
         $) {
  
  'use strict';

  
  // Check <datalist> and input[list] support
  var supportDatalist = (function () {
    var userAgent = navigator.userAgent,
        hasChrome = typeof chrome !== 'undefined';

    // restore native datalists in chrome after this issue is fixed:
    // https://code.google.com/p/chromium/issues/detail?id=375637
    if (hasChrome) {
      return false;
    }

    //JavaFX pretends to support datalists but really doesn't.
    if (userAgent.match('JavaFX')) {
      return false;
    }

    //JxBrowser reports its userAgent as Chrome/Chromium, but doesn't have
    //an actual chrome object. It also pretends to support datalists.
    if (userAgent.match(/Chrom[ie]/i) && !hasChrome) {
      return false;
    }

    //IE10 has the same issue as chrome
    if (userAgent.match('MSIE (8|9|10)')) {
      return false;
    }
    
    var options     = 'options' in document.createElement('datalist');
    var listoptions = ('list' in document.createElement('input') && 'options' in document.createElement('datalist'));
    var globalobj   = !!window.HTMLDataListElement;
    var shivglobobj = !!(document.createElement('datalist') && window.HTMLDataListElement);
    
    if (options && listoptions && globalobj && shivglobobj) {
      return true;
    }
  })();

  // Make jQuery's :contains case insensitive (like HTML5 datalist)
  // Changed the name to prevent overriding original functionality
  $.expr[":"].RDContains = $.expr.createPseudo(function (arg) {
      return function (elem) {
          return $(elem).text().toUpperCase().indexOf(arg.toUpperCase()) >= 0;
      };
  });

  $.fn.relevantDropdown = function (options) {

    options = $.extend({
      fadeOutSpeed: 'normal', // speed to fade out the dataList Popup
      change: null
    }, options);

    if (!options.force && supportDatalist) {
      return;
    }

    return this.each(function () {

      var that = this,
          $input = $(this),
          listId = $input.attr('list'),
          $datalist = $("#" + listId),
          container = options.container || $("body"),
          showAll = options.showAll,
          datalistItems,
          ul,

          scrollValue = 0;


      function hideDatalist() {
        $datalist.fadeOut(options.fadeOutSpeed);
        datalistItems.removeClass("active");
        scrollValue = 0;
        //$(window).off('resize', updateDatalistPosition);
      }

      function updateDatalistPosition() {
        var containerOffset = container.offset(),
            offset = $input.offset(),
            inputHeight = $input.outerHeight(),
            datalistHeight = $datalist.outerHeight(),
            top = Math.max(offset.top - containerOffset.top + inputHeight, 0),
            bottom,
            win = $(window),
            windowHeight = win.innerHeight(),
            windowWidth = win.innerWidth(),
            lostBottom = top + datalistHeight - windowHeight,
            lostTop = -(top - inputHeight - datalistHeight),
            inputWidth = Math.min($input.outerWidth(), windowWidth),
            maxHeight;

        if (top + datalistHeight > windowHeight && lostBottom > lostTop) {
          $datalist.addClass('inverted');
          bottom = top - inputHeight;
          maxHeight = Math.min(datalistHeight - (bottom - windowHeight), bottom);
          bottom = Math.max(windowHeight - (bottom), 0);
          top = 'auto';
        } else {
          $datalist.removeClass('inverted');
          maxHeight = windowHeight - top;
          bottom = 'auto';
        }

        // Build datalist
        $datalist
          .scrollTop(scrollValue)
          .css({
            top: top,
            bottom: bottom,
            left: Math.max(0, Math.min(offset.left - containerOffset.left, windowWidth - inputWidth)),
            maxHeight: maxHeight,
            width: inputWidth
          });
      }

      function getSelectedItem() {
        return $('li.active', $datalist);
      }

      // Insert home for new fake datalist
      ul = $("<ul />", {
        "class": "datalist",
        "id": listId
      }).appendTo(container);

      if (options.datalist) {
        $datalist = options.datalist;
        $datalist.remove();
      } else {
        $datalist.remove();
        $datalist = $("#" + listId);
      }

      datalistItems = $datalist.find("option");

      // Fill new fake datalist
      ul.html($.map(datalistItems, function (el) {
        // .val is required here, not .text or .html
        // HTML *needs* to be <option value="xxx"> not <option>xxx</option>  (IE)
        return $('<li/>').text($(el).val());
      }));

      // Update pointer
      $datalist = ul;
      
      //IE blurs the input if you use the mouse button to scroll.
      $datalist.on('mousedown', function (e) {
        if (e.target === $datalist[0]) {
          setTimeout(function () {
            $input.focus();
            scrollValue = $datalist.scrollTop();
          }, 50);
        }
      });
      
      datalistItems = $datalist.find("li");

      // Typey type type
      $input
        .on("blur", function () {
          // If this fires immediately, it prevents click-to-select from working
          setTimeout(function () {
            if (!$input.is(':focus')) {
              hideDatalist();
            }
          }, 100);
        })
        .on("keyup focus click show-dropdown", function (e) {
          if (e.keyCode === 13) {
            //close datalist on enter keyup instead of keydown so it plays nicer with RelevantStringEditor
            hideDatalist();
          }
          if ($input.prop('readonly') || $input.prop('disabled') || e.keyCode === 13 || e.keyCode === 27 || e.keyCode === 9) { return; }

          $input.addClass('js-calculate');
          updateDatalistPosition();
          $input.removeClass('js-calculate');

          if (e.type === 'show-dropdown') {
            $input.focus();
          }

          var wasDatalistShowing = $datalist.is(':visible');
          $datalist.show();

          datalistItems.hide();
          var text = $input.val(), item;
          if (showAll) {
            $datalist.find("li").show();
          } else {
            $datalist.find("li:RDContains('" + text + "')").show();
          }

          if (text && e.keyCode !== 38 && e.keyCode !== 40 && e.keyCode !== 9 && e.keyCode !== 16) {
            //set active item on keyup - expect up arrow, down arrow, tab & shift
            datalistItems.removeClass("active");
            if (!showAll) {
              $datalist.find("li:RDContains('" + text + "'):first").addClass("active");
            }
          }

          if (!wasDatalistShowing) {
            datalistItems.removeClass("active");
            for (var i = 0, len = datalistItems.length; i < len; i++) {
              item = $(datalistItems[i]);
              if (item.text() === text) {
                item.addClass("active");
                var selectedTop = item.position().top,
                  datalistHeight = $datalist.outerHeight();

                if (selectedTop < 0 || selectedTop >= datalistHeight) {
                  scrollValue = selectedTop + $datalist.scrollTop();
                  $datalist.scrollTop(scrollValue);
                }
                break;
              }
            }
          }
        });

      // Don't want to use :hover in CSS so doing this instead
      // really helps with arrow key navigation
      $datalist
        .on('mouseenter', 'li', function () {
          $(this).addClass("active").siblings().removeClass("active");
        })
        .on('mouseleave', 'li', function () {
          $(this).removeClass("active");
        });

      // Window resize
      //$(window).on('resize', updateDatalistPosition);

      // Watch arrow keys for up and down
      $input.on("keydown", function (e) {

        var active = getSelectedItem(),
            datalistHeight = $datalist.outerHeight(),
            datalistItemsHeight = datalistItems.outerHeight();

        // up arrow
        if (e.keyCode === 38) {
          if (active.length) {
            var prev = active.prevAll("li:visible").eq(0),
                selectedTop;

            if (prev.length) {
              active.removeClass("active");
              prev.addClass("active");
              selectedTop = prev.position().top;

              if (selectedTop < 0 || selectedTop >= datalistHeight) {
                scrollValue = selectedTop + $datalist.scrollTop();
                $datalist.scrollTop(scrollValue);
              }
            }
          }
        }

        // down arrow
        if (e.keyCode === 40) {
          if (active.length) {
            var next = active.nextAll("li:visible").eq(0),
                selectedBottom;
            
            if (next.length) {
              active.removeClass("active");
              next.addClass("active");
              selectedBottom = next.position().top + datalistItemsHeight;
              
              if (selectedBottom < 0 || selectedBottom >= datalistHeight) {
                scrollValue = selectedBottom - datalistHeight + $datalist.scrollTop();
                $datalist.scrollTop(scrollValue);
              }
            }
          } else {
            datalistItems.removeClass("active");
            $datalist.find("li:visible:first").addClass("active");
          }
        }

        // return or tab key
        if (e.keyCode === 13 || e.keyCode === 9) {
          if (active.length) {
            $input.val(active.text());
            itemSelected(active.text());
          }
          if (e.keyCode === 9) {
            hideDatalist();
          }
        }

        // keys
        if (e.keyCode !== 13 && e.keyCode !== 38 && e.keyCode !== 40) {
          // Reset scroll
          $datalist.scrollTop(0);
          scrollValue = 0;
        }
        
        // esc
        if (e.keyCode === 27) {
          hideDatalist();
        }
      });

      // When choosing from dropdown
      $datalist.on('click', 'li', function () {
        $input.val($(this).text()).trigger('input').focus();
        hideDatalist();
        itemSelected($(this).text());
      });

      function itemSelected(newText) {
        if (typeof options.change === 'function') {
          options.change.call(that, newText);
        }
      }

    });
  };
});
