/**
 * @copyright 2019 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/videoDriver/rc/live/VideoFeedMixin
 */
define(['nmodule/js/rc/tinyevents/tinyevents', 'nmodule/webEditors/rc/wb/mixin/mixinUtils'], function (tinyevents, mixinUtils) {
  'use strict';

  var applyMixin = mixinUtils.applyMixin;
  var MIXIN_NAME = 'videoDriver:VideoFeedMixin';

  /**
   * This mixin should be applied to a `VideoStream` to mark it as supporting
   * streaming video.
   *
   * A `VideoStream` implementing this mixin *may* implement the following
   * functions:
   *
   * - `getVideoConnectionStatus`: return a string describing the current
   *   video connection status ("fetching," "connecting," "logging in," etc.).
   *   `emit()` a `CONNECTION_STATUS_CHANGED_EVENT` to indicate that a new
   *   status is available. The framework will use this to show the status to
   *   the user as it gains a connection to the video stream (or if connection
   *   is lost).
   * - `getVideoTimestamp`: return a timestamp (long instant) describing the
   *   timestamp the video is streaming. `emit()` a `TIMESTAMP_CHANGED_EVENT` to
   *   indicate that a new timestamp is available. The framework will use this
   *   to display the current timestamp to the user.
   * - `isVideoStreaming`: return a boolean to indicate that video data is
   *   actually being received. `emit()` a `STREAM_DATA_RECEIVED_EVENT` to
   *   indicate that stream data was received. The framework will use this to
   *   perform operations that only make sense when video is actually being
   *   displayed (such as enabling pan/tilt/zoom controls).
   *
   * @mixin
   * @mixes tinyevents
   * @alias module:nmodule/videoDriver/rc/fe/live/VideoFeedMixin
   */
  var VideoFeedMixin = function VideoFeedMixin(videoStream) {
    if (!applyMixin(videoStream, MIXIN_NAME, VideoFeedMixin.prototype)) {
      return;
    }
    tinyevents(videoStream);
  };

  /**
   * String identifying this mixin.
   * @type {string}
   */
  VideoFeedMixin.MIXIN_NAME = MIXIN_NAME;

  /**
   * Notifies the framework that stream data was received. `isVideoStreaming()`
   * may be called after this event.
   * @type {string}
   */
  VideoFeedMixin.STREAM_DATA_RECEIVED_EVENT = 'videoDriver:streamDataReceived';

  /**
   * Notifies the framework that the connection status has changed.
   * `getVideoConnectionStatus()` may be called after this event.
   * @type {string}
   */
  VideoFeedMixin.CONNECTION_STATUS_CHANGED_EVENT = 'videoDriver:connectionStatusChanged';

  /**
   * Notifies the framework that the video timestamp has changed.
   * `getVideoTimestamp()` may be called after this event.
   * @type {string}
   */
  VideoFeedMixin.TIMESTAMP_CHANGED_EVENT = 'videoDriver:timestampChanged';
  return VideoFeedMixin;
});
