/**
 * @copyright 2017 Tridium, Inc. All Rights Reserved.
 */

/**
 * API Status: **Private**
 * @module nmodule/schedule/rc/baja/fe/ScheduleEnumSetEditor
 */
define(['baja!', 'bajaux/mixin/batchSaveMixin', 'jquery', 'Promise', 'nmodule/webEditors/rc/fe/baja/EnumSetEditor'], function (baja, batchSaveMixin, $, Promise, EnumSetEditor) {
  'use strict';

  /**
   * @class
   * @alias module:nmodule/schedule/rc/baja/fe/ScheduleEnumSetEditor
   * @extends module:nmodule/webEditors/rc/fe/baja/EnumSetEditor
   */
  var ScheduleEnumSetEditor = function ScheduleEnumSetEditor() {
    EnumSetEditor.apply(this, arguments);
  };
  ScheduleEnumSetEditor.prototype = Object.create(EnumSetEditor.prototype);
  ScheduleEnumSetEditor.prototype.constructor = ScheduleEnumSetEditor;

  /**
   * When an enum item is checked, determine if that element is exclusive.
   *
   * The framework will deselect all other elements when an exclusive element is checked.
   *
   * The default behavior is that the first element is exclusive, which is normal for
   * most ScheduleEnumSetEditor implementations.
   *
   * Override this function if you want to change the default behavior.
   *
   * @param element - DOM element that was checked
   * @returns {boolean} true if the element is an exclusive element.
   *
   */
  ScheduleEnumSetEditor.prototype.isExclusiveElement = function (element) {
    return $(element).is(':first-child > input');
  };

  /**
   * Handle a non-exclusive element being checked.
   *
   * The default behavior is to deselect the first element, which is an exclusive element.
   * Override this if you want to change the default behavior.
   *
   */
  ScheduleEnumSetEditor.prototype.nonExclusiveElementChecked = function () {
    this.jq().find('input')[0].checked = false;
  };
  ScheduleEnumSetEditor.prototype.doInitialize = function (dom) {
    var that = this;
    return Promise.resolve(EnumSetEditor.prototype.doInitialize.apply(this, arguments)).then(function () {
      that.$listSelection.setExclusiveFilter(function (idx) {
        return that.isExclusiveSelection(idx);
      });
    });
  };

  /**
   * Filter function to determine if an index is exclusive.
   *
   * The default implementation sets the first item (index === 0) as
   * exclusive.  Override this function to change the behavior in derived editors.
   *
   * @param {Number} idx - index into num set list.
   * @returns {boolean} return true if the element is exclusive.
   */
  ScheduleEnumSetEditor.prototype.isExclusiveSelection = function (idx) {
    return idx === 0;
  };
  ScheduleEnumSetEditor.prototype.doLoad = function (enumSet) {
    var that = this,
      ordinals = enumSet.getOrdinals(),
      complex = that.getComplex();
    if (complex) {
      // If effectiveWhenEmpty and the EnumSet is empty, or alwaysEffective then
      // indicate this by selecting the wildcard.
      if (complex.get('effectiveWhenEmpty') && !ordinals.length || complex.get('alwaysEffective')) {
        ordinals = that.getWildcard();
      }
    } else {
      // If empty EnumSet without parent complex, indicate this by selecting the wildcard.
      if (!ordinals.length) {
        ordinals = that.getWildcard();
      }
    }
    return that.$makeEnumSet(ordinals).then(function (enumSet) {
      return EnumSetEditor.prototype.doLoad.call(that, enumSet);
    });
  };

  /**
   * Get the ordinals that represent a wildcard (always effective schedule)
   *
   * Override if you want to change the default behavior.
   *
   * @returns {Array.<Number>}
   */
  ScheduleEnumSetEditor.prototype.getWildcard = function () {
    return [-1];
  };

  /**
   * Check to see if the enumSet is a wildcard (typically getOrdinals === [-1])
   *
   * Override if you want to change the default behavior.
   *
   * @param {baja.EnumSet} enumSet
   * @returns {boolean} true if the EnumSet indicates a wildcard
   */
  ScheduleEnumSetEditor.prototype.isWildcard = function (enumSet) {
    var ordinals = enumSet.getOrdinals();
    return ordinals.length === 1 && ordinals[0] === -1;
  };
  ScheduleEnumSetEditor.prototype.saveToComplex = function (readValue, params) {
    var that = this,
      complex = that.getComplex(),
      slot = that.getSlot(),
      enumSet = readValue,
      batch = params && params.batch,
      progressCallback = params && params.progressCallback,
      proms = [];

    // Save readValue to this.getSlot() on this.getComplex()
    // Assuming -1 is a wildcard, if the wildcard is selected, the enumSet
    // should be empty and either effectiveWhenEmpty or alwaysEffective should be
    // set.
    if (that.isWildcard(enumSet)) {
      enumSet = baja.EnumSet.make({
        ordinals: []
      });
      if (!complex.get('effectiveWhenEmpty')) {
        proms.push(complex.set({
          slot: 'alwaysEffective',
          value: true,
          batch: batch
        }));
      }
    } else {
      proms.push(complex.set({
        slot: 'alwaysEffective',
        value: false,
        batch: batch
      }));
    }
    proms.push(complex.set({
      slot: slot,
      value: enumSet,
      batch: batch
    }));
    if (progressCallback) {
      progressCallback(batchSaveMixin.COMMIT_READY);
    }
    return Promise.all(proms);
  };
  ScheduleEnumSetEditor.prototype.doRead = function () {
    return Promise.resolve(EnumSetEditor.prototype.doRead.apply(this, arguments)).then(function (enumSet) {
      return baja.EnumSet.make({
        ordinals: enumSet.getOrdinals()
      });
    });
  };

  /**
   * Make an EnumSet for this editor
   *
   * @param {Array.<Number>} ordinalsToLoad
   * @returns {Promise.<baja.EnumSet>} resolves to an EnumSet
   */
  ScheduleEnumSetEditor.prototype.$makeEnumSet = function (ordinalsToLoad) {
    var that = this;
    return that.getEnumRange().then(function (range) {
      return baja.EnumSet.make({
        ordinals: ordinalsToLoad,
        range: range
      });
    });
  };
  return ScheduleEnumSetEditor;
});
