/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/wb/commands/ReorderSlotsDialog
 */
define(['baja!', 'bajaux/commands/Command', 'jquery', 'Promise', 'underscore', 'bajaux/util/CommandButtonGroup', 'nmodule/webEditors/rc/fe/fe', 'nmodule/webEditors/rc/fe/baja/BaseEditor', 'nmodule/webEditors/rc/wb/tree/BajaSlotTreeNode', 'nmodule/webEditors/rc/wb/tree/NavTree', 'nmodule/webEditors/rc/wb/tree/TreeNode'], function (baja, Command, $, Promise, _, CommandButtonGroup, fe, BaseEditor, BajaSlotTreeNode, NavTree, TreeNode) {
  'use strict';

  ////////////////////////////////////////////////////////////////
  // Support functions
  ////////////////////////////////////////////////////////////////

  /**
   * Swap two elements in an array.
   *
   * @inner
   * @param {Array} arr
   * @param {Number} i1 index 1
   * @param {Number} i2 index 2
   */
  function swap(arr, i1, i2) {
    if (i1 < 0 || i2 < 0 || i1 > arr.length - 1 || i2 > arr.length - 1) {
      return;
    }
    var t = arr[i1];
    arr[i1] = arr[i2];
    arr[i2] = t;
  }

  /**
   * Tree node for showing dynamic slots on a component. Child nodes
   * are for slots only and will never expand.
   *
   * @inner
   * @class
   * @extends module:nmodule/webEditors/rc/wb/tree/TreeNode
   * @param {baja.Component} comp
   */
  var DynamicSlotsTreeNode = function DynamicSlotsTreeNode(comp) {
    var name = comp.getType().getTypeName();
    TreeNode.call(this, name);
    this.$comp = comp;
  };
  DynamicSlotsTreeNode.prototype = Object.create(TreeNode.prototype);
  DynamicSlotsTreeNode.prototype.constructor = DynamicSlotsTreeNode;

  /**
   * Get the backing component behind this node.
   *
   * @returns {baja.Component}
   */
  DynamicSlotsTreeNode.prototype.value = function () {
    return this.$comp;
  };

  /**
   * Resolve an array of kid nodes for all dynamic slots on the component.
   * @returns {Promise}
   */
  DynamicSlotsTreeNode.prototype.$loadKids = function (params) {
    //no batch because this will only contain kids for one component

    var that = this,
      complex = that.value(),
      progressCallback = params && params.progressCallback || _.noop,
      prom = complex.loadSlots().then(function () {
        var slots = complex.getSlots().dynamic().toArray();
        return _.map(slots, function (slot) {
          return new BajaSlotTreeNode(complex, slot);
        });
      });
    progressCallback('commitReady');
    return prom;
  };

  ////////////////////////////////////////////////////////////////
  // ReorderSlotsDialog
  ////////////////////////////////////////////////////////////////

  /**
   * A dialog for performing a reorder operation on a component. Based on
   * `workbench:ReorderPad`.
   *
   * Note that this dialog will not actually perform the reorder operation,
   * but only resolve the an array of slot names in the desired order. It is
   * up to the consumer of this dialog to pass the slot names into
   * `baja.Component#reorder` or not.
   *
   * @class
   * @extends module:nmodule/webEditors/rc/fe/baja/BaseEditor
   * @alias module:nmodule/webEditors/rc/wb/commands/ReorderSlotsDialog
   */
  var ReorderSlotsDialog = function ReorderSlotsDialog() {
    var that = this;
    BaseEditor.apply(that, arguments);
    that.getCommandGroup().add(new Command({
      module: 'webEditors',
      lex: 'commands.reorder.up',
      func: function func() {
        return that.$doMoveUp();
      }
    }), new Command({
      module: 'webEditors',
      lex: 'commands.reorder.down',
      func: function func() {
        return that.$doMoveDown();
      }
    }), new Command({
      module: 'webEditors',
      lex: 'commands.reorder.byName',
      func: function func() {
        var reverse = this.$reverse;
        this.$reverse = !reverse;
        return that.$doSortByName(reverse);
      }
    }), new Command({
      module: 'webEditors',
      lex: 'commands.reorder.byType',
      func: function func() {
        var reverse = this.$reverse;
        this.$reverse = !reverse;
        return that.$doSortByType(reverse);
      }
    }),
    //TODO: Sort By Position, has to do with wiresheet position
    new Command({
      module: 'webEditors',
      lex: 'commands.reorder.reset',
      func: function func() {
        return that.$doReset();
      }
    }));
  };
  ReorderSlotsDialog.prototype = Object.create(BaseEditor.prototype);
  ReorderSlotsDialog.prototype.constructor = ReorderSlotsDialog;

  /**
   * Get the nav tree showing the dynamic slots on the component to reorder.
   *
   * @private
   * @returns {module:nmodule/webEditors/rc/wb/tree/NavTree}
   */
  ReorderSlotsDialog.prototype.$getNavTree = function () {
    return this.jq().children('.NavTree').data('widget');
  };

  /**
   * Get the button group showing a button for each available reorder command.
   *
   * @private
   * @returns {module:bajaux/util/CommandButtonGroup}
   */
  ReorderSlotsDialog.prototype.$getCommandButtonGroup = function () {
    return this.jq().children('.CommandButtonGroup').data('widget');
  };

  /**
   * Get the "move up" command.
   *
   * @private
   * @returns {bajaux/commands/Command}
   */
  ReorderSlotsDialog.prototype.$getMoveUpCommand = function () {
    return this.getCommandGroup().get(0);
  };

  /**
   * Get the "move down" command.
   *
   * @private
   * @returns {bajaux/commands/Command}
   */
  ReorderSlotsDialog.prototype.$getMoveDownCommand = function () {
    return this.getCommandGroup().get(1);
  };

  /**
   * Get the "sort by name" command.
   *
   * @private
   * @returns {bajaux/commands/Command}
   */
  ReorderSlotsDialog.prototype.$getSortByNameCommand = function () {
    return this.getCommandGroup().get(2);
  };

  /**
   * Get the "sort by type" command.
   *
   * @private
   * @returns {bajaux/commands/Command}
   */
  ReorderSlotsDialog.prototype.$getSortByTypeCommand = function () {
    return this.getCommandGroup().get(3);
  };

  /**
   * Get the "reset" command.
   *
   * @private
   * @returns {bajaux/commands/Command}
   */
  ReorderSlotsDialog.prototype.$getResetCommand = function () {
    return this.getCommandGroup().get(4);
  };

  /**
   * Get all dynamic slot names that are currently selected for moving
   * up/down by the user.
   *
   * @private
   * @returns {Array.<String>}
   */
  ReorderSlotsDialog.prototype.$getSelectedNames = function () {
    return _.map(this.$getNavTree().getSelectedNodes(), function (node) {
      return node.getName();
    });
  };

  /**
   * Resolve all dynamic slot names on the component to reorder, in order
   * currently configured by the user.
   *
   * @private
   * @returns {Promise}
   */
  ReorderSlotsDialog.prototype.$getAllNames = function () {
    return this.$getNavTree().value().getKids().then(function (kids) {
      return _.map(kids, function (kid) {
        return kid.getName();
      });
    });
  };

  /**
   * Get the dynamic slot names on the component in their original, un-reordered
   * order.
   *
   * @private
   * @returns {Array.<String>}
   */
  ReorderSlotsDialog.prototype.$getOriginalSlotNames = function () {
    return _.map(this.value().getSlots().dynamic().toArray(), String);
  };

  /**
   * Moves all selected slots up one space in the list.
   *
   * @private
   * @returns {Promise}
   */
  ReorderSlotsDialog.prototype.$doMoveUp = function () {
    var that = this;
    return that.$getAllNames().then(function (names) {
      var selectedSlots = that.$getSelectedNames(),
        idx;
      for (var i = 0; i < selectedSlots.length; i++) {
        idx = names.indexOf(selectedSlots[i]);
        if (selectedSlots.indexOf(names[idx - 1]) === -1) {
          swap(names, idx, idx - 1);
        }
      }
      return that.$getNavTree().value().reorder(names);
    });
  };

  /**
   * Moves all selected slots down one space in the list.
   *
   * @private
   * @returns {Promise}
   */
  ReorderSlotsDialog.prototype.$doMoveDown = function () {
    var that = this;
    return that.$getAllNames().then(function (names) {
      var selectedSlots = that.$getSelectedNames(),
        idx;
      for (var i = selectedSlots.length - 1; i >= 0; i--) {
        idx = names.indexOf(selectedSlots[i]);
        if (selectedSlots.indexOf(names[idx + 1]) === -1) {
          swap(names, idx, idx + 1);
        }
      }
      return that.$getNavTree().value().reorder(names);
    });
  };

  /**
   * Sort slots by display name.
   *
   * @private
   * @param {Boolean} reverse
   * @returns {Promise}
   */
  ReorderSlotsDialog.prototype.$doSortByName = function (reverse) {
    var that = this,
      comp = that.value();
    return that.$getAllNames().then(function (names) {
      names.sort(function (name1, name2) {
        var disp1 = comp.getDisplayName(name1),
          disp2 = comp.getDisplayName(name2);
        return disp1 < disp2 ? -1 : disp1 === disp2 ? 0 : 1;
      });
      if (reverse) {
        names.reverse();
      }
      return that.$getNavTree().value().reorder(names);
    });
  };

  /**
   * Sort slots by type name.
   *
   * @private
   * @param {Boolean} reverse
   * @returns {Promise}
   */
  ReorderSlotsDialog.prototype.$doSortByType = function (reverse) {
    var that = this,
      comp = that.value();
    return that.$getAllNames().then(function (names) {
      names.sort(function (name1, name2) {
        var type1 = comp.getSlot(name1).getType().getTypeName(),
          type2 = comp.getSlot(name2).getType().getTypeName();
        return type1 < type2 ? -1 : type1 === type2 ? 0 : 1;
      });
      if (reverse) {
        names.reverse();
      }
      return that.$getNavTree().value().reorder(names);
    });
  };

  /**
   * Reset slots to their original, unaltered order.
   *
   * @private
   */
  ReorderSlotsDialog.prototype.$doReset = function () {
    return this.$getNavTree().value().reorder(this.$getOriginalSlotNames());
  };

  /**
   * Initialize a single-layer NavTree to show the configured slot order,
   * and a button group to invoke the different reordering commands.
   *
   * Adds a `ReorderSlotsDialog` CSS class.
   *
   * @param {jQuery} dom
   * @returns {Promise}
   */
  ReorderSlotsDialog.prototype.doInitialize = function (dom) {
    dom.addClass('ReorderSlotsDialog');
    return Promise.all([new NavTree({
      loadKids: true,
      properties: {
        hideRoot: true
      }
    }).initialize($('<div/>').appendTo(dom)), fe.buildFor({
      dom: $('<div class="editor ButtonList"/>').appendTo(dom),
      value: this.getCommandGroup(),
      type: CommandButtonGroup
    })]);
  };

  /**
   * Load the dynamic slots of the given component into the nav tree for
   * selection and reordering.
   *
   * @param {baja.Component} value
   * @returns {Promise}
   */
  ReorderSlotsDialog.prototype.doLoad = function (value) {
    var enabled = !!value.getSlots().dynamic().next();
    _.each(this.getCommandGroup().getChildren(), function (kid) {
      kid.setEnabled(enabled);
    });
    return this.$getNavTree().load(new DynamicSlotsTreeNode(value));
  };

  /**
   * Resolve an array of slot names in the order configured by the user.
   *
   * @returns {Promise}
   */
  ReorderSlotsDialog.prototype.doRead = function () {
    return this.$getAllNames();
  };

  /**
   * Remove the `ReorderSlotsDialog` class added in `doInitialize`.
   */
  ReorderSlotsDialog.prototype.doDestroy = function () {
    this.jq().removeClass('ReorderSlotsDialog');
  };
  return ReorderSlotsDialog;
});
