/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

define(['baja!', 'log!nmodule.webEditors.rc.servlets.userData', 'underscore', 'Promise', 'nmodule/webEditors/rc/fe/baja/util/compUtils'], function (baja, log, _, Promise, compUtils) {
  'use strict';

  var logSevere = log.severe.bind(log);

  ////////////////////////////////////////////////////////////////
  // Exports
  ////////////////////////////////////////////////////////////////

  /**
   * API Status: **Private**
   *
   * RPC calls for working with server-side user data. All data will be
   * automatically keyed to the current user.
   *
   * @exports nmodule/webEditors/rc/servlets/userData
   */
  var exports = {};

  /**
   * Retrieve existing data for this key from the station.
   * Returns null if there is no data.
   *
   * @param {string} key
   * @returns {Promise.<string|null>}
   */
  exports.get = function (key) {
    return isInvalidKey(key) || doRpc('get', [key]);
  };

  /**
   * Retrieve existing data for this key from the station and parse the string results into a JSON Object.
   * Note that this function does not support decoding bson encoded values.
   * If you would like support for this, see `userData.fromPersistenceObject()`.
   *
   * If the userData cannot be parsed or if the data is missing the `def` parameter will be used for the result.
   *
   * @param {string} key
   * @param {Object|null} [def=null]
   * @returns {Promise.<Object|null>}
   * @since Niagara 4.14
   */
  exports.getJson = function (key) {
    var def = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : null;
    return isInvalidKey(key) || doRpc('get', [key]).then(function (result) {
      if (result === null) {
        return def;
      }
      try {
        return JSON.parse(result);
      } catch (err) {
        logSevere("Could not parse JSON:\"" + result + "\"", err);
        return def;
      }
    });
  };

  /**
   * Store a value on the station for this key.
   *
   * @param {string} key
   * @param {string} data
   * @returns {Promise}
   */
  exports.put = function (key, data) {
    return isInvalidKey(key) || isInvalidData(data) || doRpc('put', [key, data]);
  };

  /**
   * Store a JSON object on the station for this key.
   * Note that this does not support encoding non-primitive baja.Values that require bson encoding to be persisted properly.
   * If you would like support for this, see `userData.toPersistenceObject()`.
   *
   * @param {string} key
   * @param {Object} data the JSON Object to persist.
   * @returns {Promise}
   */
  exports.putJson = function (key, data) {
    var dataToStore = JSON.stringify(data);
    return isInvalidKey(key) || isInvalidData(dataToStore) || hasType(data) || doRpc('put', [key, dataToStore]);
  };

  /**
   * Delete the value on the station for this key.
   *
   * @param {string} key
   * @returns {Promise}
   */
  exports.del = function (key) {
    return isInvalidKey(key) || doRpc('delete', [key]);
  };

  ////////////////////////////////////////////////////////////////
  // Support functions
  ////////////////////////////////////////////////////////////////

  /**
   * @param {String} method
   * @param {Array} args
   * @returns {Promise.<String|null>}
   */
  function doRpc(method, args) {
    return baja.rpc({
      ord: 'service:web:UserDataConfig',
      method: method,
      args: args
    })["catch"](function (ignore) {
      return null;
    });
  }
  function isInvalidKey(key) {
    if (!key || typeof key !== 'string') {
      return Promise.reject(new Error('key required'));
    }
  }
  function isInvalidData(data) {
    if (typeof data !== 'string') {
      return Promise.reject(new Error('data required'));
    }
  }
  function hasType(data) {
    if (baja.hasType(data)) {
      return Promise.reject(new Error('No baja types allowed'));
    }
  }

  /**
   * Encodes all child values of an object to BSON objects.
   * @param {Object.<String, baja.Value>} valueMap
   * @return {Object.<String, object>}
   */
  exports.toPersistenceObject = function (valueMap) {
    if (!valueMap) {
      return {};
    } else {
      return _.mapObject(valueMap, function (value) {
        if (baja.hasType(value)) {
          return baja.bson.encodeValue(value);
        } else {
          return exports.toPersistenceObject(value);
        }
      });
    }
  };

  /**
   * Takes a component and a PersistenceObject and applies all pairs in the value map to
   * a new copy of the component. Only components with a predefined structure are supported; adding
   * Complex slots to a component without matching structure will not work.
   * @param {baja.Component} config
   * @param {Object.<string, *>} [userPreferences]
   * @returns {Promise.<baja.Component>}
   */
  exports.fromPersistenceObject = function (config, userPreferences) {
    if (!userPreferences) {
      userPreferences = {};
    }
    var allSlotPromises = [];
    var newComponent = compUtils.newCopyComplete(config);
    _.each(userPreferences, function (value, slot) {
      var slotObject = {
        slot: slot,
        value: value
      };
      if (config.get(slot) instanceof baja.Complex) {
        allSlotPromises.push(exports.fromPersistenceObject(newComponent.get(slot), value).then(function (comp) {
          newComponent.set({
            slot: slot,
            value: comp
          });
        }));
      } else if (config.has(slot)) {
        allSlotPromises.push(newComponent.set(slotObject));
      } else {
        allSlotPromises.push(newComponent.add(slotObject));
      }
    });
    return Promise.all(allSlotPromises).then(function () {
      return newComponent;
    });
  };
  return exports;
});
