/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/*global niagara: false */

define(['baja!', 'jquery', 'Promise', 'underscore', 'nmodule/webEditors/rc/wb/tree/TreeNode'], function (baja, $, Promise, _, TreeNode) {
  'use strict';

  var PALETTE_ICON_URIS = baja.Icon.make('module://icons/x16/palette.png').getImageUris(),
    PALETTE_SERVLET_URI = '/webEditors/palette/';
  var BLACKLISTED_MODULES = ['workbench'];
  function PaletteNode(obj) {
    var kids = [];
    TreeNode.call(this, obj.n, obj.d, _.map(obj.k || [], function (kidObj) {
      var node = new PaletteNode(kidObj);
      kids.push(node.$json);
      return node;
    }));
    this.$json = {
      ord: obj.o,
      name: obj.n,
      displayName: obj.d,
      icon: obj.i,
      description: obj.e,
      typeSpec: obj.e,
      kids: kids
    };
    this.$icon = baja.Icon.make(obj.i).getImageUris();
  }
  PaletteNode.prototype = Object.create(TreeNode.prototype);
  PaletteNode.prototype.constructor = PaletteNode;
  PaletteNode.prototype.getIcon = function () {
    return this.$icon;
  };
  PaletteNode.prototype.value = function () {
    return this.$json;
  };
  PaletteNode.prototype.isDraggable = function () {
    return true;
  };
  function notBlacklisted(moduleName) {
    return BLACKLISTED_MODULES.indexOf(moduleName) === -1;
  }

  //TODO: add java.awt.headless=true to unit tests

  function cacheBuster() {
    if (typeof niagara === 'undefined') {
      return '';
    }
    return '?regLastBuildTime=' + niagara.env.regLastBuildTime;
  }
  function getPaletteIcon() {
    return PALETTE_ICON_URIS;
  }
  function moduleNamesToUri(moduleNames) {
    return PALETTE_SERVLET_URI + 'get/' + moduleNames.join() + cacheBuster();
  }
  function doGetPalettes(moduleNames) {
    moduleNames = _.filter(moduleNames, notBlacklisted);
    return Promise.resolve($.get(moduleNamesToUri(moduleNames)));
  }
  function doGetList() {
    return Promise.resolve($.get(PALETTE_SERVLET_URI + 'list' + cacheBuster())).then(function (data) {
      return _.filter(data, function (obj) {
        return notBlacklisted(obj.n);
      });
    });
  }
  function toPaletteObjectNode(obj) {
    return new PaletteNode(obj);
  }
  function toPaletteModuleNode(obj) {
    var moduleName = obj.n,
      encodedNavNodes = obj.v,
      kidNodes = _.map(encodedNavNodes, toPaletteObjectNode),
      node = new TreeNode(moduleName, moduleName, kidNodes);
    node.getIcon = getPaletteIcon;
    return node;
  }

  /**
   * API Status: **Private**
   *
   * Functions for retrieving palette data from the station.
   *
   * This module communicates with the following servlets:
   *
   * - `com.tridium.webeditors.ux.servlets.PaletteServlet`
   *
   * @exports nmodule/webEditors/rc/servlets/palette
   */
  var exports = {};

  //TODO: make this data-only and move TreeNode generation to PaletteNavTree
  /**
   * Retrieve palette data from the station for the requested modules.
   *
   * @param {Array.<String>} moduleNames names of modules to retrieve palette
   * data for
   * @returns {Promise} promise to be resolved with a single
   * {@link module:nmodule/webEditors/rc/wb/tree/TreeNode}, with one child per
   * requested module
   */
  exports.getPalettes = function (moduleNames) {
    if (!Array.isArray(moduleNames)) {
      return Promise.reject(new Error('array required'));
    }
    return doGetPalettes(moduleNames).then(function (data) {
      var nodesForEachPalette = _.map(data, toPaletteModuleNode);
      return new TreeNode('Palette', 'Palette', nodesForEachPalette);
    });
  };

  /**
   * List palette data for all modules on the station that have them. Resolves
   * an array of objects, each with the following properties:
   *
   * - `d`: the module description
   * - `n`: the module name
   *
   * @returns {Promise} promise to be resolved with an array of palette data
   * objects
   */
  exports.list = function () {
    return doGetList();
  };
  return exports;
});
