/**
 * @copyright 2020 Tridium, Inc. All Rights Reserved.
 * @author Cody Short
 */

/*eslint-env browser */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/fe/ToolTip
 */

define(['jquery', 'css!nmodule/webEditors/rc/fe/webEditors-structure'], function ($) {
  'use strict';

  /**
   * @typedef {Object} module:nmodule/webEditors/rc/fe/ToolTip~hooks
   *
   * @property {module:nmodule/webEditors/rc/fe/ToolTip~setDisplay} setDisplay sets
   * the value currently displayed on the ToolTip.
   * @property {module:nmodule/webEditors/rc/fe/ToolTip~cancel} cancel this function
   * will cancel the tooltip. This will give rid of the created ToolTip and
   * remove it from the DOM.
   */

  /**
   * A callback that will update the value displayed on the ToolTip
   *
   * @callback module:nmodule/webEditors/rc/fe/ToolTip~setDisplay
   * @param {*} value the value to display. This value will be converted to a
   * string. This value should not contain any HTML characters as it would be
   * escaped.
   */

  /**
   * A callback to be invoked when the given ToolTip is cancelled whether done
   * by the ToolTip itself or the cancel hook has been invoked.
   *
   * @callback module:nmodule/webEditors/rc/fe/ToolTip~onCancel
   * @param {String} [reason] the reason the ToolTip was cancelled. This can
   * either be specified via the call to cancel or will be decided by ToolTip
   * if available. Values generated via ToolTip itself include `multitouch` when
   * the user touches the screen with more than one finger. `end` when the user
   * has performed a standard end gesture such as touchup/mouseup.
   */

  /**
   * A callback that will cancel the ToolTip.
   *
   * @callback module:nmodule/webEditors/rc/fe/ToolTip~cancel
   * @param {String} [reason] provides a way for the caller to specify the reason
   * for the ToolTip to be cancelled. This will be passed along to the onCancel
   * callback.
   */
  var MOVE_EVENTS = 'mousemove touchmove',
    TOUCH_Y_OFFSET = -50,
    MOUSE_Y_OFFSET = 15;
  function isMultiTouch(event) {
    return !!(event && event.touches && event.touches.length > 1);
  }

  /**
   * Creates a ToolTip that will by default display in a fixed position of the
   * screen. Optionally, `fixedPosition` can be turned off, and the ToolTip
   * will follow the mouse around the screen.
   *
   * This ToolTip will exit early / not be created if more than one touch
   * is detected.
   *
   * @param {Object} params
   * @param {MouseEvent|TouchEvent} [params.initiatingEvent] the event that
   * initiated the ToolTip. This is required if fixedPosition is false.
   * @param {*} params.value the value to display in the ToolTip. This value will be
   * converted to a string. This value should not contain any HTML characters as it
   * would be escaped.
   * @param {module:nmodule/webEditors/rc/fe/ToolTip~onCancel} [params.onCancel]
   * called when the given ToolTip is cancelled. Defaults to a noop.
   * @param {boolean} [params.fixedPosition=true] if set to true, the ToolTip
   * will be displayed in a fixed position on the screen and will not move. If
   * false, the ToolTip will follow the mouse / finger on the screen and will
   * also cancel automatically on mouseup.
   * @param {number} [params.delay=0] if specified, the ToolTip will wait the
   * provided number of milliseconds before displaying. If canceled before the
   * time has passed, the ToolTip will not display.
   * @returns {module:nmodule/webEditors/rc/fe/ToolTip~hooks|undefined} returns hooks
   * if the ToolTip was created.
   */
  function create(_ref) {
    var initiatingEvent = _ref.initiatingEvent,
      value = _ref.value,
      _ref$onCancel = _ref.onCancel,
      onCancel = _ref$onCancel === void 0 ? function () {} : _ref$onCancel,
      _ref$fixedPosition = _ref.fixedPosition,
      fixedPosition = _ref$fixedPosition === void 0 ? true : _ref$fixedPosition,
      _ref$delay = _ref.delay,
      delay = _ref$delay === void 0 ? 0 : _ref$delay;
    if (isMultiTouch(initiatingEvent)) {
      return;
    }
    var CANCEL_EVENTS = "touchend touchcancel ".concat(fixedPosition ? '' : 'mouseup'),
      body = $('body');
    var delayTimeoutHandle;
    function updatePosition(e) {
      if (isMultiTouch(e)) {
        cancel('multitouch');
        return;
      }
      if (fixedPosition) {
        dom.css('bottom', '0');
        dom.css('left', '0');
        return;
      }
      var isTouch = e && e.touches && e.touches.length;
      if (isTouch) {
        var touch = e.touches[0];
        dom.css('left', "".concat(touch.pageX, "px"));
        dom.css('top', "".concat(touch.pageY + TOUCH_Y_OFFSET, "px"));
      } else {
        var toolTipBottomPositionIfBelow = $(window).height() - (MOUSE_Y_OFFSET + dom.height()),
          isToolTipAboveMouse = e.clientY > toolTipBottomPositionIfBelow,
          appliedYOffset = isToolTipAboveMouse ? -MOUSE_Y_OFFSET - dom.height() : MOUSE_Y_OFFSET;
        dom.css('left', "".concat(e.clientX, "px"));
        dom.css('top', "".concat(e.clientY + appliedYOffset, "px"));
      }
    }
    var cancelEventHandler = function cancelEventHandler() {
      return cancel('end');
    };
    function cancel(reason) {
      if (delayTimeoutHandle) {
        clearTimeout(delayTimeoutHandle);
      }
      onCancel(reason);
      dom.remove();
      body.off(CANCEL_EVENTS, cancelEventHandler);
      body.off(MOVE_EVENTS, updatePosition);
    }
    function createToolTip() {
      var dom = $("<div class=\"ux-ToolTip\"></div>");
      body.on(MOVE_EVENTS, updatePosition);
      body.on(CANCEL_EVENTS, cancelEventHandler);
      return dom;
    }
    function updateDisplay(val) {
      dom.text(val);
    }
    var dom = createToolTip();
    if (delay) {
      delayTimeoutHandle = setTimeout(function () {
        return body.append(dom);
      }, delay);
    } else {
      body.append(dom);
    }
    updatePosition(initiatingEvent);
    updateDisplay(value);
    return {
      setDisplay: function setDisplay(val) {
        return updateDisplay(val);
      },
      cancel: cancel
    };
  }
  return {
    create: create
  };
});
