/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/fe/CompositeEditor
 */
define(['baja!', 'log!nmodule.webEditors.rc.fe.CompositeEditor', 'Promise', 'underscore', 'bajaux/events', 'bajaux/mixin/batchSaveMixin', 'nmodule/webEditors/rc/fe/baja/BaseEditor'], function (baja, log, Promise, _, events, batchSaveMixin, BaseEditor) {
  'use strict';

  var MODIFY_EVENT = events.MODIFY_EVENT,
    VALIDATE_ON_READ = BaseEditor.VALIDATE_ON_READ,
    saveWidgets = batchSaveMixin.saveWidgets,
    COMMIT_READY = batchSaveMixin.COMMIT_READY,
    logError = log.severe.bind(log);

  ////////////////////////////////////////////////////////////////
  // Support functions
  ////////////////////////////////////////////////////////////////

  /**
   * Validate all child editors of this composite editor that need to be
   * validated.
   *
   * @inner
   * @param {module:nmodule/webEditors/rc/fe/CompositeEditor} ed the
   * composite editor whose child editors we need to validate
   * @returns {Promise} promise to be resolved if all kids validate,
   * or rejected if any fail to validate
   */
  function allChildrenValid(ed) {
    var builder = ed.getBuilder();
    if (!builder.getDataSource()) {
      //if no value loaded, no kids to validate
      return Promise.resolve();
    }
    return Promise.resolve(builder.getKeys()).then(function (keys) {
      var toValidate = _.filter(keys, function (key) {
        var ed = builder.getEditorFor(key);

        //validate vanilla bajaux/Widgets
        if (typeof ed.shouldValidate !== 'function') {
          return ed.isModified();
        }

        //editor should not validate on read, so postpone it until doSave
        if (!ed.shouldValidate(VALIDATE_ON_READ)) {
          return false;
        }
        return ed.shouldValidate();
      });
      return builder.$validateForKeys(toValidate);
    });
  }

  ////////////////////////////////////////////////////////////////
  // Exports
  ////////////////////////////////////////////////////////////////

  /**
   * An abstract editor intended to manage the instantiation, initialization,
   * and loading of multiple sub-editors.
   *
   * @class
   * @extends module:nmodule/webEditors/rc/fe/baja/BaseEditor
   * @alias module:nmodule/webEditors/rc/fe/CompositeEditor
   * @mixes module:bajaux/mixin/batchSaveMixin
   */
  var CompositeEditor = function CompositeEditor() {
    var that = this;
    BaseEditor.apply(that, arguments);

    /**
     * A CompositeEditor will validate all of its child editors before
     * validating itself. If any child editor fails validation, `validate()`
     * will reject with a CompositeError with the rejected keys and their
     * rejection reasons.
     *
     * @method module:nmodule/webEditors/rc/fe/CompositeEditor#validate
     */
    that.validators().add(function (val) {
      return allChildrenValid(that);
    });
    batchSaveMixin(that);
  };
  CompositeEditor.prototype = Object.create(BaseEditor.prototype);
  CompositeEditor.prototype.constructor = CompositeEditor;
  CompositeEditor.prototype.$getAllEditors = function () {
    var builder = this.getBuilder();
    if (this.value() === null || builder.getDataSource() === null) {
      return Promise.resolve([]); //not loaded yet, no kid editors
    }
    return Promise.resolve(builder.getKeys()).then(function (keys) {
      return _.compact(_.map(keys, function (key) {
        return builder.getEditorFor(key);
      }));
    });
  };

  /**
   * Get the `CompositeBuilder` for this editor. If it does not yet exist, it
   * will be created via `this.makeBuilder()` and that instance will be
   * returned hereafter.
   *
   * @returns {module:nmodule/webEditors/rc/fe/config/CompositeBuilder}
   */
  CompositeEditor.prototype.getBuilder = function () {
    return this.$builder || (this.$builder = this.makeBuilder());
  };

  /**
   * Create the `CompositeBuilder` that will be used to manage the sub-editors
   * in this composite editor.
   *
   * @abstract
   * @returns {CompositeBuilder}
   * @throws {Error} if not implemented
   */
  CompositeEditor.prototype.makeBuilder = function () {
    throw new Error('makeBuilder not implemented');
  };

  /**
   * Swallows modified events from subeditors, instead setting its
   * own modified status.
   *
   * @param {jQuery} dom
   */
  CompositeEditor.prototype.doInitialize = function (dom) {
    var that = this;
    dom.on(MODIFY_EVENT, '.bajaux-initialized', function (e, ed) {
      that.setModified(true, ed);
      return false;
    });
  };

  /**
   * Simply delegates to `getBuilder().buildAll()`. Override as appropriate.
   *
   * @returns {Promise}
   */
  CompositeEditor.prototype.doLoad = function (value) {
    var that = this,
      builder = that.getBuilder();
    return builder.setDataSource(value).then(function () {
      return builder.buildAll();
    }).then(function () {
      //update readonly/enabled status of built kids
      return Promise.all([that.isReadonly() && that.doReadonly(true), !that.isEnabled() && that.doEnabled(false)]);
    });
  };

  //TODO: a child layout rejecting should also reject this; revisit when more confident
  /**
   * When laying out a `CompositeEditor`, lay out all of its child widgets.
   * This will be a noop until `load()` is complete (because the child widgets
   * do not exist until then). A child widget failing to lay out will _not_
   * cause this to reject; the error will merely be logged.
   */
  CompositeEditor.prototype.doLayout = function () {
    if (this.value() === null || !this.isInitialized() || this.isLoading()) {
      return;
    }
    var builder = this.getBuilder();
    return Promise.resolve(builder.getKeys()).then(function (keys) {
      return Promise.all(keys.map(function (key) {
        var ed = builder.getEditorFor(key);
        return ed && ed.layout();
      }));
    })["catch"](logError);
  };

  /**
   * Saves all subeditors. A single BajaScript batch will be passed into each
   * editor's `save` method parameters, so any editors that support batching
   * can all save with a single network call.
   *
   * @param {baja.Complex} value validated value
   * @param {Object} [params]
   * @param {baja.comm.Batch} [params.batch] if passed to
   * `CompositeEditor#save`, the same batch will be used to save all child
   * editors. If no batch given, a new batch will be created.
   * @returns {Promise} promise to be resolved with all editors have
   * saved successfully
   */
  CompositeEditor.prototype.doSave = function (value, params) {
    var builder = this.getBuilder(),
      batchParam = params && params.batch,
      batch = batchParam || new baja.comm.Batch(),
      _progressCallback = params && params.progressCallback;
    return Promise.resolve(builder.getKeys()).then(function (keys) {
      var kids = _.compact(_.map(keys, function (key) {
        return builder.getEditorFor(key);
      }));

      //explicitly validate all subeditors that need it. reject
      //if any fail to validate.
      return Promise.all(kids.map(function (kid) {
        var shouldValidate = typeof kid.shouldValidate === 'function' ? kid.shouldValidate() : kid.isModified();
        return shouldValidate && kid.validate();
      })).then(function () {
        var modifiedKids = _.filter(kids, function (kid) {
          return kid.isModified();
        });
        return saveWidgets(modifiedKids, {
          batch: batch,
          progressCallback: function progressCallback(msg) {
            if (msg === COMMIT_READY && !batchParam) {
              batch.commit();
            }
            return _progressCallback && _progressCallback(msg);
          }
        });
      });
    });
  };

  /**
   * Reads all child editors, assembles the child values together as
   * appropriate, and resolves a reflection of the current state of the loaded
   * value. This function simply delegates to `CompositeBuilder#readAll`.
   *
   * @returns {Promise}
   */
  CompositeEditor.prototype.doRead = function () {
    return this.getBuilder().readAll();
  };

  /**
   * Enables/disables all child editors.
   *
   * @param {Boolean} enabled
   * @returns {Promise} promise to be resolved when all child editors
   * are enabled/disabled
   */
  CompositeEditor.prototype.doEnabled = function (enabled) {
    enabled = !!enabled;
    return this.$getAllEditors().then(function (kids) {
      return Promise.all(_.map(kids, function (kid) {
        return enabled !== kid.isEnabled() && kid.setEnabled(enabled);
      }));
    });
  };

  /**
   * Sets all child editors readonly / not readonly.
   *
   * @returns {Promise} promise to be resolved when all child editors
   * are readonly / not readonly
   */
  CompositeEditor.prototype.doReadonly = function (readonly) {
    readonly = !!readonly;
    return this.$getAllEditors().then(function (kids) {
      return Promise.all(_.map(kids, function (kid) {
        return readonly !== kid.isReadonly() && kid.setReadonly(readonly);
      }));
    });
  };

  /**
   * Destroys all child editors.
   *
   * @returns {Promise} promise to be resolved when all child editors
   * are destroyed
   */
  CompositeEditor.prototype.doDestroy = function () {
    return this.$getAllEditors().then(function (kids) {
      return Promise.all(_.invoke(kids, 'destroy'));
    });
  };
  return CompositeEditor;
});
