/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/baja/PasswordStrength
 */
define(['baja!', 'lex!baja', 'Promise'], function (baja, lexs, Promise) {
  'use strict';

  var bajaLex = lexs[0],
    subclass = baja.subclass,
    Struct = baja.Struct,
    LOWERCASE = /[a-z]/g,
    UPPERCASE = /[A-Z]/g,
    DIGITS = /[0-9]/g,
    SPECIAL = /[^a-zA-Z0-9]/g;
  function reject(arr) {
    return Promise.reject(new Error(arr.join('\n')));
  }
  function countMatches(str, regex) {
    var match = str.match(regex);
    return match ? match.length : 0;
  }

  /**
   * BajaScript representation of `baja:PasswordStrength`.
   *
   * @class
   * @alias module:nmodule/webEditors/rc/baja/PasswordStrength
   * @extends baja.Struct
   */
  var PasswordStrength = function PasswordStrength() {
    Struct.apply(this, arguments);
  };
  subclass(PasswordStrength, Struct);
  PasswordStrength.validatePassword = function (strength, password) {
    if (typeof password !== 'string') {
      return reject(['string required']);
    }
    if (password.length < strength.getMinimumLength() || password.length > strength.getMaximumLength() || countMatches(password, LOWERCASE) < strength.getMinimumLowerCase() || countMatches(password, UPPERCASE) < strength.getMinimumUpperCase() || countMatches(password, DIGITS) < strength.getMinimumDigits() || countMatches(password, SPECIAL) < strength.getMinimumSpecial()) {
      return reject(strength.$getErrorDisplay());
    }
    return Promise.resolve();
  };

  /**
   * Verify the password's validity given this `PasswordStrength`'s configured
   * minimums. Note that this is for client-side validation purposes only -
   * any password that this validates will be/must be re-validated station-side
   * upon save.
   *
   * @param {String} password
   * @returns {Promise} promise to be resolved if password is valid.
   * If rejected, it will be rejected with an array of error strings describing
   * the `PasswordStrength` configuration. Join the array, map to divs,
   * whatever your app needs.
   */
  PasswordStrength.prototype.validatePassword = function (password) {
    return PasswordStrength.validatePassword(this, password);
  };
  PasswordStrength.prototype.$getErrorDisplay = function () {
    var arr = [bajaLex.get('user.password.notStrong')],
      that = this;
    function add(slot, key) {
      var minimum = that.get(slot);
      if (minimum > 0) {
        arr.push(bajaLex.get({
          key: key,
          args: [minimum]
        }));
      }
    }
    add('minimumLength', 'user.password.notLongEnough');
    add('minimumDigits', 'user.password.notEnoughDigits');
    add('minimumLowerCase', 'user.password.notEnoughLowerCase');
    add('minimumUpperCase', 'user.password.notEnoughUpperCase');
    add('minimumSpecial', 'user.password.notEnoughSpecial');
    add('maximumLength', 'user.password.notShortEnough');
    return arr;
  };
  baja.registerType('baja:PasswordStrength', function () {
    return PasswordStrength;
  });
  return PasswordStrength;
});
