/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 */

/**
 * API Status: **Private**
 * @module nmodule/bacnet/rc/baja/job/DiscoveryPoint
 */
define(['baja!', 'nmodule/bacnet/rc/baja/enums/BacnetPropertyIdentifier'], function (baja, BacnetPropertyIdentifier) {
  'use strict';

  var OBJECT = 0,
    //PROPERTY = 1,
    ARRAY = 2;
  //ELEMENT  = 3;

  /**
   * Bajascript representation of the `bacnet:DiscoveryPoint` type.
   *
   * @class
   * @extends {baja.Component}
   * @alias module:nmodule/bacnet/rc/baja/job/DiscoveryPoint
   */
  var DiscoveryPoint = function DiscoveryPoint() {
    baja.Component.apply(this, arguments);
  };
  DiscoveryPoint.prototype = Object.create(baja.Component.prototype);
  DiscoveryPoint.prototype.constructor = DiscoveryPoint;

  /**
   * Return the property id as a numeric value, obtained from the string
   * value of the 'propertyId' slot. If the string has a known prefix to
   * indicate it is an ASHRAE or proprietary property, this will be parsed
   * out by this function.
   *
   * @returns {number} the numeric value of the propertyId slot, or -1 if it could not be
   *                   obtained.
   */
  DiscoveryPoint.prototype.getPropertyId = function () {
    var id = this.getPropertyIdentifier();
    if (!id) {
      return -1;
    }
    try {
      return BacnetPropertyIdentifier.idForTag(id);
    } catch (e) {
      return -1;
    }
  };

  /**
   * Return the index number of this item in a property array, or -1 if the item
   * is not indexed.
   *
   * @returns {number}
   */
  DiscoveryPoint.prototype.getPropertyArrayIndex = function () {
    var val,
      index = this.getIndex();
    if (!index) {
      return -1;
    }
    val = parseInt(index);
    return isNaN(val) ? -1 : val;
  };

  /**
   * Return `true` if the type of this discovered point is one that may have
   * children.
   *
   * @returns {boolean}
   */
  DiscoveryPoint.prototype.hasChildren = function () {
    var type = this.getPointType();
    return type === OBJECT || type === ARRAY;
  };

  /**
   * Return any child points as an array of `DiscoveryPoint` instances.
   *
   * @returns {Array.<baja.Value>}
   */
  DiscoveryPoint.prototype.getPoints = function () {
    return this.getSlots().properties().is('bacnet:DiscoveryPoint').toValueArray();
  };

  /**
   * Override of `toString()`, returns an encoded debug string equivalent
   * to the one produced by the Java type.
   *
   * @returns {string}
   */
  DiscoveryPoint.prototype.toString = function () {
    var that = this;
    return 'DiscPt:' + that.getObjectName() + '[' + that.getObjectId() + ']' + ':' + that.getPropertyIdentifier() + '{' + that.getIndex() + '}' + '=' + that.getValue() + '; f=' + that.getFacets() + '; t=' + that.getPointType();
  };
  return DiscoveryPoint;
});
