/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 */

/**
 * API Status: **Private**
 * @module nmodule/bacnet/rc/baja/datatypes/DiscoveryNetworks
 */
define(['baja!', 'underscore'], function (baja, _) {
  'use strict';

  var Simple = baja.Simple;

  /**
   * Bajascript representation of the `bacnet:DiscoveryNetworks` type.
   *
   * @class
   * @extends {baja.Simple}
   * @alias module:nmodule/bacnet/rc/baja/datatypes/DiscoveryNetworks
   */
  var DiscoveryNetworks = function DiscoveryNetworks() {
    Simple.apply(this, arguments);
  };
  DiscoveryNetworks.prototype = Object.create(Simple.prototype);
  DiscoveryNetworks.prototype.constructor = DiscoveryNetworks;

  /**
   * Create a new `DiscoveryNetworks` instance from the given arguments.
   *
   * @static
   *
   * @param {Boolean} all - `true` to indicate all network numbers are chosen.
   * @param {Array.<Number>} nets - an array of network numbers.
   * @returns {module:nmodule/bacnet/rc/baja/datatypes/DiscoveryNetworks}
   */
  DiscoveryNetworks.make = function (all, nets) {
    var dn = new DiscoveryNetworks();
    dn.$nets = nets ? nets.slice() : [];
    dn.$all = !!all;
    return dn;
  };
  DiscoveryNetworks.prototype.make = function () {
    return DiscoveryNetworks.make.apply(DiscoveryNetworks, arguments);
  };

  /**
   * Encode the 'all networks' flag, and the individual network numbers to
   * a string.
   *
   * @returns {String}
   */
  DiscoveryNetworks.prototype.encodeToString = function () {
    return (this.$all ? 'all;' : '-;') + this.$nets.join(',');
  };

  /**
   * Decode the `DiscoveryNetworks` from a string.
   *
   * @param {String} str - the DiscoveryNetworks instance encoded as a string.
   * @returns {module:nmodule/bacnet/rc/baja/datatypes/DiscoveryNetworks}
   */
  DiscoveryNetworks.prototype.decodeFromString = function (str) {
    var all, nets, match;
    if (str) {
      all = /^all;/.test(str);
      match = str.match(/^(?:all|-);(.+)/);
      if (match && match.length > 1) {
        nets = _.map(match[1].split(','), function (n) {
          return parseInt(n);
        });
      }
      return DiscoveryNetworks.make(all, nets);
    }
    return DiscoveryNetworks.DEFAULT;
  };

  /**
   * Returns true if a discovery or who-has job should target all networks
   * globally.
   *
   * @returns {Boolean}
   */
  DiscoveryNetworks.prototype.isAllNetworks = function () {
    return this.$all;
  };

  /**
   * Return an array of the network numbers to be targeted for
   * a discovery or who-has job.
   *
   * @returns {Array.<Number>}
   */
  DiscoveryNetworks.prototype.getNetworks = function () {
    return this.$nets.slice();
  };

  /**
   * Test whether the array of network numbers contains the given number.
   *
   * @param {Number} networkNumber - the number to find
   * @returns {boolean} true if the network number was found in the list.
   */
  DiscoveryNetworks.prototype.contains = function (networkNumber) {
    return _.indexOf(this.$nets, networkNumber) > -1;
  };
  DiscoveryNetworks.DEFAULT = DiscoveryNetworks.make(false, []);
  return DiscoveryNetworks;
});
