/*jshint browser:true*/

/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam and Gareth Johnson
 */
define(['jquery', 'baja!', 'bajaux/Widget', 'Promise', 'underscore'], function ($, baja, Widget, Promise, _) {
  'use strict';

  /**
   * Useful methods for loading bajaux Widgets in a Niagara Mobile view.
   * 
   * @private
   * @exports mobile/util/mobile/views/widgetViewUtil
   */

  var exports = {};

  /**
   * Initializes and loads a bajaux Widget into a mobile view.
   *
   * This method bootstraps and handles the entire process of loading
   * a bajaux Widget into a Niagara Mobile view.
   * 
   * @param {function} makeWidget Invoked to create a new instance of the
   * bajaux Widget.
   */
  exports.loadPageView = function (makeWidget) {
    // Delay calling initialize page until we're ready (see initializePage() call below).
    $(document).one('mobileinit', function () {
      $.mobile.autoInitializePage = false;
    });

    require(['jquerymobile', 'mobile/util/mobile/commands', 'mobile/util/mobile/views/PageView', 'mobile/util/mobile/views/PageViewManager', 'mobile/util/mobile/pages', 'mobile/util/mobile/mobile'], function (jqm, commands, PageView, PageViewManager, pages, mobileUtil) {

      var pageViewManager;

      /**
       * A page view implementation for a bajaux widget.
       *
       * This dynamically creates the jquery mobile page DOM that wraps
       * the widget.
       *
       * @class
       * @private
       * @inner
       * @extends module:mobile/util/mobile/views/PageView
       */
      var WidgetPageView = function WidgetPageView() {
        PageView.call(this, {
          contentViewConstructor: function contentViewConstructor() {
            return makeWidget();
          }
        });
      };

      WidgetPageView.prototype = Object.create(PageView.prototype);
      WidgetPageView.prototype.constructor = WidgetPageView;

      /**
       * Returns a newly created DOM element for the jquery mobile page.
       * 
       * @returns The jquery DOM wrapper around the DOM element.
       */
      WidgetPageView.prototype.createPage = function () {
        var dom = PageView.prototype.createPage.apply(this, arguments);
        // Reduce padding so the content fills all available room.
        dom.children("[data-role=content]").css("padding", 0);

        // Add the commands button.
        dom.children("[data-role=header]").append(commands.getCommandsButton());

        return dom;
      };

      WidgetPageView.prototype.initialize = function () {
        this.$pageViewResizeHandler = _.debounce(function () {
          if (pageViewManager) {
            pageViewManager.layout();
          }
        }, 200);
        $(window).on('resize', this.$pageViewResizeHandler);

        return PageView.prototype.initialize.apply(this, arguments);
      };

      WidgetPageView.prototype.destroy = function () {
        $(window).off('resize', this.$pageViewResizeHandler);
        this.$pageViewResizeHandler = null;

        return PageView.prototype.destroy.apply(this, arguments);
      };

      /**
       * The page view manager for the widget. This manages the underlying
       * page view widget and registers it with jquery mobile.
       *
       * @class
       * @private
       * @inner
       * @extends module:mobile/util/mobile/views/PageViewManager
       */
      var WidgetPageViewManager = function WidgetPageViewManager() {
        PageViewManager.apply(this, arguments);
      };

      WidgetPageViewManager.prototype = Object.create(PageViewManager.prototype);
      WidgetPageViewManager.prototype.constructor = WidgetPageViewManager;

      /**
       * Returns a promise that resolves to the widget page view.
       * 
       * @returns {Promise} A promise that resolves to a new page view instance.
       */
      WidgetPageViewManager.prototype.instantiateView = function () {
        return Promise.resolve(new WidgetPageView());
      };

      pages.register({
        pageshow: function pageshow() {
          // There is no page defined when first loaded. Therefore redirect
          // to an internal ORD page that in turn will trigger the dynamic creation
          // of the widget page view.
          mobileUtil.linkToOrdInternal(window.niagara.view.ord, {
            transition: "none",
            changeHash: false
          });
        }
      });

      pageViewManager = new WidgetPageViewManager();
      pageViewManager.registerPages();

      // Once we're all done, initialize jquery mobile.
      jqm.initializePage();
    });
  }; // loadPageView

  return exports;
});
