/**
 * @file Mobile-related utilities.
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/*global niagara, unescape */

define(['baja!', 'jquery', 'jquerymobile', 'Promise'], function (baja, $, jqm, Promise) {

  "use strict";

  var NON_HTML_ID_CHARS = /[^\w-_]/g,
      entityMap = {
    "&": "&amp;",
    "<": "&lt;",
    ">": "&gt;",
    '"': '&quot;',
    "'": '&#39;',
    "/": '&#x2F;'
  };

  /**
   * Safely escapes non-HTML characters ( &amp; &lt; &gt; &quot; &#39; &#x2F; )
   * in the given string.
   * @memberOf niagara.util.mobile
   * @param {String} string
   * @returns {String}
   */
  function escapeHtml(string) {
    return String(string).replace(/[&<>"'\/]/g, function (s) {
      return entityMap[s];
    });
  }

  function showPageLoadingMsg() {
    jqm.loading('show');
  }

  function hidePageLoadingMsg() {
    jqm.loading('hide');
  }

  /**
   * Determines the actual visible height of the content of the current
   * JQM page - minus header and footer height.
   * 
   * @memberOf niagara.util.mobile
   * 
   * @param {jQuery} page the JQM page whose visible content height you want
   * @returns {Number} the height of the visible portion of the
   * given JQM page, in pixels
   */
  function getVisibleHeight(page) {
    var header = page.find(':jqmData(role="header")'),
        footer = page.find(':jqmData(role="footer")'),

    //http://bugs.jquery.com/ticket/6724
    height = window.innerHeight || $(window).height();

    if (header.length) {
      height -= header.outerHeight();
    }

    if (footer.length) {
      height -= footer.outerHeight();
    }

    return height;
  }

  /**
   * Explicitly sets the height of the current JQM page's content div to
   * maximize visible space.
   * 
   * @memberOf niagara.util.mobile
   * @param {jQuery} page the JQM page whose content div you want to maximize
   * @param {jQuery} [divToSet] you can specify a particular div whose
   * height to set - if omitted, defaults to 
   * `page.children('jqmData(role=content)')`
   */
  function setContentHeight(page, divToSet) {
    var div = divToSet || page.children(':jqmData(role="content")'),
        visibleHeight = getVisibleHeight(page),
        offset = div.outerHeight() - div.height();
    div.height(visibleHeight - offset);
    div.parent().height(Math.max(div.parent().height(), visibleHeight));
    setTimeout(function () {
      div.trigger('updatelayout');
    }, 100);
  }

  /**
   * Adds an event handler to a DOM element, but ensures that the handler
   * winds up first in the handler list. This way, if your handler decides to
   * return false or cancel event propagation, the handlers already registered
   * on that element will not fire.
   * 
   * @memberOf niagara.util.mobile
   * @param {jQuery} dom the element on which to bind
   * @param {String} eventName
   * @param {Function} handler
   */
  function prependEventHandler(dom, eventName, handler) {
    dom.on(eventName, handler);

    var data = $._data(dom[0]),
        events = data.events,
        handlers = events[eventName];

    if (handlers.length > 1) {
      handlers.splice(0, 0, handlers.pop());
    }
  }

  /**
   * @class
   * @memberOf niagara.util.mobile
   * @private
   */
  function PageLoadingTicket(delay, timeout, timeoutFunc) {
    var that = this;

    that.$ticket = setTimeout(function () {
      if (that.$ticket) {
        showPageLoadingMsg();
      }
    }, delay);

    if (timeout) {
      that.$timeoutTicket = setTimeout(function () {
        if (that.$ticket) {
          var msg = 'timeout ' + timeout + 'ms reached';
          if (typeof timeoutFunc === 'function') {
            timeoutFunc(msg);
          } else {
            baja.error(msg);
          }
        }
        that.hide();
      }, timeout);
    }
  }

  PageLoadingTicket.prototype.hide = function () {
    hidePageLoadingMsg();

    var that = this;
    clearTimeout(that.$ticket);
    delete that.$ticket;
    clearTimeout(that.$timeoutTicket);
    delete that.$timeoutTicket;
  };

  /**
   * Returns a ticket object that will show a page loading message after a
   * specified delay. Retrieve this ticket object before performing some
   * asynchronous action, then call the ticket's `hide` method after the action
   * returns - this way, if the async action takes less than the specified time,
   * the loading message won't be shown at all.
   * 
   * You can also specify a maximum amount of time to spin before some
   * error action is taken.
   * 
   * @memberOf niagara.util.mobile
   * @param {Number} delay amount of time (in ms) to wait before showing
   * the loading message
   * @param {Number} [timeout] maximum amount of time the loading message
   * may spin
   * @param {Function} [timeoutFunc] a function to execute if the timeout
   * is reached before `hide()` is called - an error message describing the
   * timeout will be passed as the first parameter
   * 
   * @returns {niagara.util.mobile.PageLoadingTicket} call `hide()` on this to
   * hide the page loading message (or prevent it from showing if it is not
   * already shown)
   * 
   * @example
   * var ticket = spinnerTicket(1000, 5000, function () {
   *   baja.error("spun for 5 seconds - timing out");
   * });
   * performSomeAsynchronousAction({
   *   ok: function () {
   *     //if the async action takes less than 1 seconds, no loading message
   *     //will show
   *     ticket.hide();
   *   }
   * });
   */
  function spinnerTicket(delay, timeout, timeoutFunc) {
    return new PageLoadingTicket(delay || 1000, timeout, timeoutFunc);
  }

  function setListviewInset(ul, inset) {
    ul.toggleClass("ui-listview-inset ui-corner-all ui-shadow", inset);

    var lis = ul.children('li'),
        first = lis.first(),
        last = lis.last();

    first.toggleClass('ui-corner-top', inset);
    last.toggleClass('ui-corner-bottom', inset);

    //for split button list, the split link on far right needs rounding too
    first.children('a').last().toggleClass('ui-corner-tr', inset);
    last.children('a').last().toggleClass('ui-corner-br', inset);
  }

  /**
   * Converts a JQM listview to inset form (adds margins and rounded 
   * corners).
   * 
   * @memberOf niagara.util.mobile
   * @param {jQuery} ul a JQM `ul:jqmData(role=listview)` listview element
   */
  function applyListviewInset(ul) {
    setListviewInset(ul, true);
  }

  /**
   * Converts a JQM listview to non-inset form (removes margins and rounded 
   * corners).
   * 
   * @memberOf niagara.util.mobile
   * @param {jQuery} ul a JQM `ul:jqmData(role=listview)` listview element
   */
  function removeListviewInset(ul) {
    setListviewInset(ul, false);
  }

  /**
   * Don't allow a navbar button to retain its JQM highlighting / hover status
   * after it is clicked. 
   * 
   * @memberOf niagara.util.mobile
   * @param {jQuery} navbar a navbar div (`:jqmData(role=navbar)`)
   */
  function preventNavbarHighlight(navbar) {
    navbar.on('click', 'a', function () {
      var $this = $(this);
      setTimeout(function () {
        $this.removeClass(function (index, css) {
          var classesToRemove = 'ui-btn-active ' + (css.match(/\bui-btn-hover-\w/g) || []).join(' ');
          return classesToRemove;
        });
      }, 100);
    });
  }

  /**
   * Creates an appropriate HTML ID for a property sheet's containing div.
   * All non-alphanumeric characters (except an actual underscore, or a
   * hyphen) will be escaped by an underscore followed by the character's 
   * ASCII code, e.g. `|` becomes `_7C`. If the input page id begins with `#`
   * (i.e. in jQuery selector form), the beginning `#` will be stripped from
   * the returned page ID (it will be not be encoded as part of it!)
   * 
   * @memberOf niagara.util.mobile
   * 
   * @param {baja.Ord|String} ord the ORD to convert to a page ID
   * @returns {String} a valid HTML5 ID (not in selector form, i.e. does not
   * start with `#`)
   */
  function encodePageId(slotPath) {
    slotPath = String(slotPath);
    if (slotPath.charAt(0) === '#') {
      slotPath = slotPath.substring(1);
    }

    return slotPath.replace(NON_HTML_ID_CHARS, function (s) {
      return '_' + s.charCodeAt(0).toString(16).toUpperCase();
    });
  }

  /**
   * Converts a page ID (the output of `encodePageId` back into a regular
   * string.
   * 
   * @memberOf niagara.util.mobile
   * @param {String|jQuery} str the page id to decode (or a jQuery object,
   * in which case attr('id') will be used)
   * @returns {String} the decoded string
   */
  function decodePageId(str) {
    if (!str) {
      return str;
    }

    if (str instanceof $) {
      str = str.attr('id');
    } else {
      str = String(str);
    }

    var re = /\_([A-Z0-9][A-Z0-9])/g;

    str = str.replace(re, function (s) {
      return String.fromCharCode(parseInt(s.substring(1, s.length), 16));
    });
    return unescape(str);
  }

  /**
   * Converts an `options` input to one of the other `linkToOrd` function into
   * an object with a `viewQuery` property. This enables the input of a number
   * of different types to `linkToOrd` depending on what you have handy.
   * 
   * @memberOf niagara.util.mobile
   * @private
   * @param options
   * @returns {Object} an object literal with a `viewQuery` property to be used
   * in a `linkToOrd` method (`viewQuery` will be undefined if not provided)
   * 
   * @example
   * var ord = 'station:|slot:',
   *     id = 'workbench:PropertySheet',
   *     params = { foo: 'bar' };
   *
   * linkToOrd(ord); //no view query
   * linkToOrd(ord, id); //new ViewQuery using the id
   * linkToOrd(ord, { id: id, params: params } ); //new ViewQuery using id and params
   * linkToOrd(ord, new baja.ViewQuery(id)); //uses the ViewQuery as is
   */
  function viewQueryify(options) {
    var viewQuery = options && options.viewQuery;
    if (!viewQuery) {
      options = baja.objectify(options, 'viewQuery');
      viewQuery = options.viewQuery;
    }

    if (typeof viewQuery === 'string' || viewQuery && viewQuery.id) {
      viewQuery = new baja.ViewQuery(viewQuery);
    } else if (!viewQuery && options.id) {
      viewQuery = new baja.ViewQuery(options);
    }

    options.viewQuery = viewQuery;
    return options;
  }

  /**
   * Performs a link to a new ORD, triggering a page refresh. Used when linking
   * to an ORD to be viewed in a different app. Simply calls
   * `window.location.assign`.
   * 
   * @memberOf niagara.util.mobile
   * @see niagara.util.mobile.viewQueryify
   * @param {String|baja.Ord} ord the ORD to link to
   * @param {Object} [options] additional options used in the page transition
   * @param {baja.ViewQuery} [options.viewQuery] can specify any view parameters to be
   * encoded into the ORD
   */
  function linkToOrdExternal(ord, options) {
    options = viewQueryify(options);

    var o;
    if (options.viewQuery) {
      o = baja.Ord.make({
        base: ord,
        child: options.viewQuery
      });
    } else {
      o = baja.Ord.make(ord);
    }

    window.location.assign(o.toUri());
  }

  /**
   * Performs a link to a new ORD without causing a page refresh to a new app.
   * Used for linking between `PageView`s within the same app by listening to
   * changes to the ORD in the browser's current URL.
   * 
   * @memberOf niagara.util.mobile
   * @see niagara.util.mobile.viewQueryify
   * @param {String|baja.Ord} ord the ORD to link to
   * @param {Object} [options] additional options used in the page transition -
   * will be passed directly into `$.mobile.changePage()`
   * @param {baja.ViewQuery} [options.viewQuery] can specify any view parameters to be
   * encoded into the ORD
   */
  function linkToOrdInternal(ord, options) {
    options = viewQueryify(options);

    var o;
    if (options.viewQuery) {
      o = baja.Ord.make({
        base: ord,
        child: options.viewQuery
      });
    } else {
      o = baja.Ord.make(ord);
    }

    $.mobile.changePage(o.toUri(), options);
  }

  /**
   * Queries the server to ascertain that the view for the requested ORD is the
   * same as the current view (that is, the requested ORD should be viewed using
   * the same app we are currently in). If it is, it triggers an internal link
   * ($.mobile.changePage). Otherwise, it instructs 
   * the browser to do a full page reload to the requested app.
   * 
   * @memberOf niagara.util.mobile
   * @see niagara.util.mobile.viewQueryify
   * @param {String|baja.Ord} ord the ORD to link to
   * @param {Object} [options] additional options used in the page transition -
   * will be passed directly into `$.mobile.changePage()`
   * @param {baja.ViewQuery} [options.viewQuery] can specify any view parameters to be
   * encoded into the ORD
   * @returns {Promise}
   */
  function linkToOrd(ord, options) {
    var ticket = spinnerTicket(1000);

    return Promise.resolve($.ajax({
      url: baja.Ord.make(ord).toUri(),
      type: "POST",
      headers: {
        "niagara-mobile-rpc": "typeSpec"
      }
    })).then(function (data) {
      ticket.hide();
      if (data.typeSpec === niagara.view.typeSpec) {
        return linkToOrdInternal(ord, options);
      } else {
        return linkToOrdExternal(ord, options);
      }
    }, function (err) {
      ticket.hide();
      return linkToOrdExternal(ord, options);
    });
  }

  function changePage(url, params) {
    getPageContainer().pagecontainer('change', url, params);
  }

  /**
   * Get the jQuery Mobile page container (in previous versions,
   * $.mobile.pageContainer).
   * @returns {jQuery}
   */
  function getPageContainer() {
    return $(":mobile-pagecontainer");
  }

  /**
   * Get the jQuery Mobile active page (in previous versions,
   * $.mobile.activePage).
   * @returns {jQuery}
   */
  function getActivePage() {
    return getPageContainer().pagecontainer('getActivePage');
  }

  /**
   * Run the given function as soon the JQM page container has been created.
   * If the page container already exists, the function will run immediately.
   * @memberOf niagara.util.mobile
   */
  function onPageContainerCreate(func) {
    var pageContainer = getPageContainer();
    if (pageContainer.length) {
      func();
    } else {
      $(window).one('pagecontainercreate', func);
    }
  }

  /**
   * @namespace
   * @name niagara.util.mobile
   */
  return {
    applyListviewInset: applyListviewInset,
    changePage: changePage,
    decodePageId: decodePageId,
    encodePageId: encodePageId,
    escapeHtml: escapeHtml,
    getActivePage: getActivePage,
    getPageContainer: getPageContainer,
    getVisibleHeight: getVisibleHeight,
    hidePageLoadingMsg: hidePageLoadingMsg,
    linkToOrd: linkToOrd,
    linkToOrdExternal: linkToOrdExternal,
    linkToOrdInternal: linkToOrdInternal,
    onPageContainerCreate: onPageContainerCreate,
    prependEventHandler: prependEventHandler,
    preventNavbarHighlight: preventNavbarHighlight,
    removeListviewInset: removeListviewInset,
    setContentHeight: setContentHeight,
    showPageLoadingMsg: showPageLoadingMsg,
    spinnerTicket: spinnerTicket
  };
});
