/**
 * @file Niagara Mobile Utils - functions relating to the
 * "available commands" button and associated dialog in our JQM page headers
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson
 */

/*global niagara*/

define(function (require) {

  "use strict";

  require('css!mobile/util/mobile/commands');
  require('jquerymobile');

  var baja = require('baja!'),
      $ = require('jquery'),
      Promise = require('Promise'),
      _ = require('underscore'),
      Command = require('bajaux/commands/Command'),
      dialogs = require('mobile/util/mobile/dialogs'),
      pages = require('mobile/util/mobile/pages'),
      mobileUtil = require('mobile/util/mobile/mobile'),
      mobileLex = require('lex!mobile')[0],
      callbackify = baja.callbackify,
      defaultCommands = [],
      commandsButtonHtml = "<a class='commandsButton profileHeader' title='{title}' data-icon='bars' data-role='button' data-iconpos='notext'></a>",
      homeCmd,
      logoutCmd;

  ////////////////////////////////////////////////////////////////
  // Command Button and HTML
  ////////////////////////////////////////////////////////////////

  /**
   * Returns HTML to be used for the command button (an `<a>` tag).
   * 
   * @memberOf niagara.util.mobile.commands
   * 
   * @returns {jQuery} an `<a>` tag, ready to be appended
   * into a JQM page header
   */
  function getCommandsButton() {
    return $(commandsButtonHtml.patternReplace({
      title: mobileLex.get('menu')
    }));
  }

  /**
   * Shows a listview dialog with commands for the user to select.
   * The entries in the dialog are populated by a command object when the dialog is
   * shown.
   * 
   * 
   * @memberOf niagara.util.mobile.commands
   * @param {Array} commands an array of commands to show
   * @param {String} [title] a title for the command dialog
   */
  function showCommandsDialog(commands, title, callbacks) {
    commands = _.filter(commands, function (command) {
      return command.isEnabled();
    });

    // Fire a jQuery event just before the commands are shown
    $("a.commandsButton").trigger("onShowCommands", commands);

    callbacks = callbackify(callbacks);

    title = title || mobileLex.get('commands.title');

    if (commands.length === 0) {
      dialogs.ok({
        title: title,
        content: mobileLex.get('commands.noneAvailable')
      });
    } else {
      var dialogInvocation = dialogs.cancel({
        title: title,
        content: function content(targetElement) {
          var makeListItems = _.map(commands, function (command) {
            var li = $('<li/>'),
                link = $('<a/>').appendTo(li);

            return command.toDisplayName().then(function (displayName) {
              link.text(displayName);
              link.click(function () {
                dialogInvocation.invoke(function (callbacks) {
                  command.invoke().then(callbacks.ok, callbacks.fail);
                });
              });
              return li;
            });
          });

          return Promise.all(makeListItems).then(function (lis) {
            var ul = $('<ul data-role="listview" data-theme="c" />').html(lis);
            targetElement.html(ul);
            ul.listview();
          });
        },
        callbacks: callbacks
      });
    }
  }

  ////////////////////////////////////////////////////////////////
  // Default Commands
  ////////////////////////////////////////////////////////////////

  /**
   * Add a Command to the end of the default list.
   *
   * @memberOf niagara.util.mobile.commands
   * @param {Command} cmd a command that will appear when the commands button is clicked.
   *                      Please note, to add multiple commands in one go, specify multiple
   *                      command arguments for this function.     
   */
  function addDefaultCommand() {
    var args = Array.prototype.slice.call(arguments);

    baja.iterate(args, function (c) {
      // Add the command (providing it isn't already present)
      if (!defaultCommands.contains(c)) {
        defaultCommands.push(c);
      }
    });
  }

  /**
   * Add a Command to the top of the default list.
   *
   * @memberOf niagara.util.mobile.commands
   * @param {Command} cmd a command that will appear when the commands button is clicked.
   *                      Please note, to add multiple commands in one go, specify multiple
   *                      command arguments for this function.     
   */
  function prependDefaultCommand() {
    var args = Array.prototype.slice.call(arguments);

    baja.iterate(args, function (c) {
      // Add the command (providing it isn't already present)
      if (!defaultCommands.contains(c)) {
        if (defaultCommands.length) {
          defaultCommands.splice(0, 0, c);
        } else {
          defaultCommands.push(c);
        }
      }
    });
  }

  /**
   * Remove a Command from the default list.
   *
   * @memberOf niagara.util.mobile.commands
   * @param {Command|Array} cmd the command Object to remove.
   *                            Please note, to add multiple commands in one go, specify multiple
   *                            command arguments for this function.     
   */
  function removeDefaultCommand(cmd) {
    var args = Array.prototype.slice.call(arguments);

    if (args.length > 1) {
      baja.iterate(args, function (c) {
        removeDefaultCommand(c);
      });
    } else {
      baja.iterate(defaultCommands, function (defC, i) {
        if (defC === cmd) {
          defaultCommands.splice(i, 1);
          return true;
        }
      });
    }
  }

  /**
   * Return an array of the default Commands.
   * 
   * Please note, this returns a defensive copy of the default commands array.
   *
   * @memberOf niagara.util.mobile.commands
   * @returns {Array}
   */
  function getDefaultCommands() {
    return defaultCommands.slice(0);
  }

  /**
   * Set our default commands array to the given array of Command objects.
   *
   * @param cmds - Array of niagara.util.mobile.commands.Command objects.
   *  
   * @memberOf niagara.util.mobile.commands
   */
  function setDefaultCommands(cmds) {
    defaultCommands = cmds;
  }

  ////////////////////////////////////////////////////////////////
  // Event Handling
  ////////////////////////////////////////////////////////////////

  function showCommandsHandler() {
    // Get the default list of commands
    var cmds = getDefaultCommands(),
        handler = pages.getCurrentHandler();

    // If the pages framework is being used then see if the current handler defines a list of commands
    if (handler && typeof handler.getCommands === "function") {

      // Ask for the commands (while passing in the original default list)
      cmds = handler.getCommands({
        commands: cmds,
        page: pages.getCurrent()
      });
    }

    // Show the commands dialog whenever this button is clicked
    showCommandsDialog(cmds);

    // Return false to stop event bubbling
    return false;
  }

  ////////////////////////////////////////////////////////////////
  // Logout Command
  ////////////////////////////////////////////////////////////////

  // Automatically add the logout command
  logoutCmd = new Command("%lexicon(mobile:commands.logout)%", function (callbacks) {
    var url = location.protocol + "//" + location.host + "/logout";
    window.location.assign(url + "?csrfToken=" + encodeURIComponent($("#csrfToken").val()));
    //don't bother calling callbacks.ok, dialog can stay up since we're linking away
  });

  if (typeof niagara !== 'undefined' && niagara.view.profile.showLogoutCmd) {
    prependDefaultCommand(logoutCmd);
  }

  ////////////////////////////////////////////////////////////////
  // Home Command
  ////////////////////////////////////////////////////////////////

  // Automatically add the home command if we can
  homeCmd = new Command("%lexicon(mobile:commands.home)%", function (callbacks) {
    // Link to user home
    mobileUtil.linkToOrd(baja.getUserHome());
    //don't bother calling callbacks.ok, dialog can stay up since we're linking away
  });

  if (typeof niagara !== 'undefined' && niagara.view.profile.showHome) {
    prependDefaultCommand(homeCmd);
  }

  /**
   * Return a Command to return to the current user's home node.
   * 
   * @memberOf niagara.util.mobile.commands
   * @returns {Command}
   */
  function getHomeCmd() {
    return homeCmd;
  }

  /**
   * Return a Command to log out the current user.
   * 
   * @memberOf niagara.util.mobile.commands
   * @returns {Command}
   */
  function getLogoutCmd() {
    return logoutCmd;
  }

  ////////////////////////////////////////////////////////////////
  // Export
  ////////////////////////////////////////////////////////////////

  /**
   * @namespace
   * @name niagara.util.mobile.commands
   */
  return {
    getCommandsButton: getCommandsButton,
    getHomeCmd: getHomeCmd,
    getLogoutCmd: getLogoutCmd,
    showCommandsDialog: showCommandsDialog,
    addDefaultCommand: addDefaultCommand,
    prependDefaultCommand: prependDefaultCommand,
    removeDefaultCommand: removeDefaultCommand,
    getDefaultCommands: getDefaultCommands,
    setDefaultCommands: setDefaultCommands,
    showCommandsHandler: showCommandsHandler
  };
});
