/**
 * @file Gradient utilities.
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

define(['mobile/util/color'], function gradientUtil(colorUtil) {

  "use strict";

  var parseRgba = colorUtil.parseRgba,
      VENDOR_PREFS = ['-webkit-', '-moz-', '-ms-', '-o-'],
      IE_FILTER = "progid:DXImageTransform.Microsoft.gradient(" + "GradientType={gradientType}," + "startColorstr='{startColor}', " + "endColorstr='{endColor}')";

  /**
   * @private
   * @exports mobile/util/color
   */
  var exports = {};

  function createGradientPointData(angle) {
    //project our angle from the origin onto a square from -1 to 1
    var rads = angle * Math.PI / 180,
        cos = Math.cos(rads),
        sin = Math.sin(rads),
        mag = 1 / Math.max(Math.abs(sin), Math.abs(cos)),
        endX = cos * mag,
        endY = -sin * mag,
        //flip the sign of y since higher Y = lower on screen
    startX = -endX,
        startY = -endY;

    //shift our box 1 unit up and right, multiply by 50 so we get 0 to 100
    startX = Math.round((startX + 1) * 50);
    startY = Math.round((startY + 1) * 50);
    endX = Math.round((endX + 1) * 50);
    endY = Math.round((endY + 1) * 50);

    return startX + "% " + startY + "%, " + endX + "% " + endY + "%";
  }

  /**
   * @param {Number} angle
   * @param {Array} stops
   * @return {String} something like
   * `(linear, 0% 0%, 100% 100%, color-stop(0%, red), color-stop(25%, orange), color-stop(100%, yellow))`
   * - note no prefix
   */
  function makeWebkitGradient(angle, stops) {
    var pointData = createGradientPointData(angle),
        stop,
        webkitCss,
        i;

    webkitCss = "(linear, " + pointData;

    for (i = 0; i < stops.length; ++i) {
      stop = stops[i];
      webkitCss += ", color-stop(" + stop[0] + ", " + stop[1].css + ")";
    }

    webkitCss += ")";

    return webkitCss;
  }

  /**
   * @param {Number} angle
   * @param {Array} stops
   * @return {String} something like `(45deg, red 0%, orange 25%, yellow 100%)`
   * - note no prefix
   */
  function makeLinearGradient(angle, stops) {
    var css,
        stop,
        len = stops.length,
        i = 0;

    css = '(' + angle + 'deg, ';

    while (i < len) {
      stop = stops[i];

      css += stop[1].css + ' ' + stop[0];

      if (++i < len) {
        css += ', ';
      }
    }

    css += ')';

    return css;
  }

  function makeIEFilter(angle, startColor, endColor) {
    return IE_FILTER.patternReplace({
      gradientType: angle ? "0" : "1",
      startColor: startColor.hexString,
      endColor: endColor.hexString
    });
  }

  function parseStops(stops) {
    var parsed = [],
        stop,
        i;
    for (i = 0; i < stops.length; i++) {
      stop = stops[i];
      parsed[i] = [stop[0], parseRgba(stop[1])];
    }
    return parsed;
  }

  /**
   * @param {Number} angle in degrees. This is a NIAGARA angle like you would
   * find on BBrush.LinearGradient, NOT a CSS angle. A Niagara angle works
   * like a Cartesian plane, where 0 degrees goes straight right and 90 degrees
   * goes straight up. A CSS angle works like a clock, where 0 degrees goes
   * straight up and 90 degrees goes straight right. This Niagara angle will
   * be converted to a CSS angle.
   * @param {Array} stops a two dimensional array of stop percentages and
   * colors. e.g.: `[["0%", "red"], ["25%", "orange"], ["100%", "yellow"]]`
   *
   * @return {Object} an object with the following properties:
   *
   * - `background` background color for older browsers, equal to first stop
   *   color
   * - `filter` a filter suitable for use in IE8
   * - `webkitGradient` gradient CSS good for Android 2.3 and iOS 4. DOES NOT
   *   include `-webkit-gradient` prefix, you must provide it
   * - `linearGradientNonStandard` gradient CSS good for Android 4, iOS5,
   *   other modern browsers. DOES NOT include `-linear-gradient` prefix, you
   *   must provide it and any vendor-specific prefixes you need
   * - `linearGradient` gradient CSS that follows the unprefixed spec (angles
   *   start vertical and go clockwise). DOES NOT include `linear-gradient`
   *   prefix
   *
   * or null, if no stops are given.
   */
  exports.createLinearGradientCss = function (angle, stops) {
    if (!stops || !stops.length) {
      return null;
    }

    stops = parseStops(stops);

    var obj = {},
        firstStop = stops[0],
        firstColor = firstStop[1],
        lastStop = stops[stops.length - 1],
        lastColor = lastStop[1];

    //background = first stop color
    obj.background = firstColor.hexString;
    obj.filter = makeIEFilter(angle, firstColor, lastColor);
    obj.webkitGradient = makeWebkitGradient(angle, stops);
    obj.linearGradientNonStandard = makeLinearGradient(angle, stops);
    obj.linearGradient = makeLinearGradient(90 - angle, stops);

    return obj;
  };

  /**
   * Applies a sensible array of CSS properties to get a background linear
   * gradient.
   *
   * @param {JQuery} element
   * @param {Number} angle
   * @param {Array} stops
   */
  exports.applyLinearGradientBackground = function (element, angle, stops) {
    var css = exports.createLinearGradientCss(angle, stops),
        i;

    if (css !== null) {
      element.css('background', css.background);
      element.css('filter', css.filter);
      element.css('background', '-webkit-gradient' + css.webkitGradient);
      for (i = 0; i < VENDOR_PREFS.length; i++) {
        element.css('background', VENDOR_PREFS[i] + 'linear-gradient' + css.linearGradientNonStandard);
      }
      element.css('background', 'linear-gradient' + css.linearGradient);
    }
  };

  return exports;
});
