var _slicedToArray = function () { function sliceIterator(arr, i) { var _arr = []; var _n = true; var _d = false; var _e = undefined; try { for (var _i = arr[Symbol.iterator](), _s; !(_n = (_s = _i.next()).done); _n = true) { _arr.push(_s.value); if (i && _arr.length === i) break; } } catch (err) { _d = true; _e = err; } finally { try { if (!_n && _i["return"]) _i["return"](); } finally { if (_d) throw _e; } } return _arr; } return function (arr, i) { if (Array.isArray(arr)) { return arr; } else if (Symbol.iterator in Object(arr)) { return sliceIterator(arr, i); } else { throw new TypeError("Invalid attempt to destructure non-iterable instance"); } }; }();

/**
 * @file Functions relating to page navigation and interaction within
 * the mobile scheduler app.
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/*global niagara */

define(['baja!schedule:WeeklySchedule,schedule:CalendarSchedule,' + 'schedule:TriggerSchedule,schedule:AbstractSchedule,' + 'schedule:DailySchedule,mobile:ScheduleServerSideCallHandler,' + 'schedule:WeekSchedule', 'css!mobile/schedule/schedule.ui', 'baja!', 'lex!baja,mobile,schedule', 'jquery', 'jquerymobile', 'Promise', 'underscore', 'mobile/util/time', 'mobile/util/aop', 'mobile/util/slot', 'bajaux/commands/Command', 'bajaux/events', 'mobile/util/mobile/mobile', 'mobile/util/mobile/pages', 'mobile/util/mobile/commands', 'mobile/util/mobile/dialogs', 'mobile/util/mobile/views/RadioButtonView', 'mobile/fieldeditors/fieldeditors', 'mobile/fieldeditors/fieldeditors.dialogs', 'mobile/schedule/schedule', 'mobile/schedule/ScheduleBlock', 'mobile/schedule/schedule.ui.calendar', 'mobile/schedule/util.schedule', 'mobile/schedule/util.schedule.datebox', 'mobile/schedule/schedule.fieldeditors', 'mobile/schedule/schedule.fieldeditors.triggers'], function (types, unusedCss, baja, lexs, $, jqm, Promise, _, timeUtil, aop, slotUtil, Command, events, mobileUtil, pages, commands, dialogs, RadioButtonView, fe, feDialogs, Schedule, ScheduleBlock, calendarUI, scheduleUtil, dateboxUtil) {

  "use strict";

  //imports

  var encodePageId = mobileUtil.encodePageId,
      escapeHtml = mobileUtil.escapeHtml,
      getActivePage = mobileUtil.getActivePage,
      getPageContainer = mobileUtil.getPageContainer,
      hidePageLoadingMsg = mobileUtil.hidePageLoadingMsg,
      linkToOrd = mobileUtil.linkToOrd,
      onPageContainerCreate = mobileUtil.onPageContainerCreate,
      prependEventHandler = mobileUtil.prependEventHandler,
      preventNavbarHighlight = mobileUtil.preventNavbarHighlight,
      setContentHeight = mobileUtil.setContentHeight,
      appendTimeLabels = scheduleUtil.appendTimeLabels,
      relayoutLabelsDiv = scheduleUtil.relayoutLabelsDiv,
      setCurrentSchedule = scheduleUtil.setCurrentSchedule,
      setLastEnteredValue = scheduleUtil.setLastEnteredValue,
      moveDown = slotUtil.moveDown,
      moveToTop = slotUtil.moveToTop,
      moveUp = slotUtil.moveUp,
      MILLIS_IN_HOUR = timeUtil.MILLIS_IN_HOUR,
      MILLIS_IN_MINUTE = timeUtil.MILLIS_IN_MINUTE,
      getAbsTime = dateboxUtil.getAbsTime,
      validateTimeChange = dateboxUtil.validateTimeChange,
      validateTimeScheduleEditor = dateboxUtil.validateTimeScheduleEditor,
      _lexs = _slicedToArray(lexs, 3),
      bajaLex = _lexs[0],
      mobileLex = _lexs[1],
      scheduleLex = _lexs[2],
      WEEKLY_SCHEDULE_TYPE = 'schedule:WeeklySchedule',
      CALENDAR_SCHEDULE_TYPE = 'schedule:CalendarSchedule',
      TRIGGER_SCHEDULE_TYPE = 'schedule:TriggerSchedule',
      ABSTRACT_SCHEDULE_TYPE = 'schedule:AbstractSchedule',
      DAILY_SCHEDULE_TYPE = 'schedule:DailySchedule',
      DATE_SCHEDULE_TYPE = 'schedule:DateSchedule',
      SCHEDULE_SSC_TYPE = 'mobile:ScheduleServerSideCallHandler',
      DISABLED_CLASS = 'ui-disabled',
      BTN_ACTIVE_CLASS = 'ui-btn-active',
      JQM_HEADER_SELECTOR = ':jqmData(role=header)';

  var navBarListItem = function navBarListItem(_ref) {
    var pageId = _ref.pageId,
        tabId = _ref.tabId,
        text = _ref.text;
    return '\n      <li>\n        <a id="' + pageId + '-' + tabId + '" class="to-' + tabId + '" href="#' + tabId + '" data-theme="b">\n          ' + escapeHtml(text) + '\n        </a>\n      </li>';
  };

  var scheduleReadonly = false,
      specialEventsRead = false,
      specialEventsReadonly = false,
      currentSchedule = void 0,
      //currently edited niagara.schedule.Schedule
  scheduleComponent = void 0,
      //keep a version subscribed for bajascript events and SSCs
  scheduleSnapshot = void 0,
      specialEventsComponent = void 0,
      specialEventsSnapshot = void 0,
      editedDaySchedule = void 0,
      //day schedule being edited on editDay or editSpecialEvent
  dayEditor = void 0,
      //current day editor on editDay or editSpecialEvent
  copiedDayEditor = void 0,
      //copied day editor, to be used later when Paste chosen
  createMode = true,
      //whether day editor button should create new block or edit existing one
  blockCommands = void 0;

  function logError(err) {
    baja.error(err);
  }

  /**
   * Completely redraws a schedule display so it reflects the current status
   * of its underlying `schedule:WeeklySchedule` component.
   *
   * @memberOf niagara.schedule.ui
   * @private
   */
  function redrawSchedule() {
    var scheduleDiv = $('#schedule'),
        mainPage = $('#main'),
        labelsDiv = scheduleDiv.find('div.scheduleLabels'),
        blocksDisplayHeight;

    setContentHeight(mainPage);

    return currentSchedule.load(currentSchedule.value()).then(function () {
      currentSchedule.layout();

      blocksDisplayHeight = scheduleDiv.find('div.blocksDisplay').height();

      relayoutLabelsDiv(labelsDiv, blocksDisplayHeight);
    });
  }

  /**
   * @memberOf niagara.schedule.ui
   * @private
   */
  function setCreateMode(footer, isCreate) {
    createMode = isCreate;
    $('.edit .ui-icon', footer).toggleClass('ui-icon-grid', !isCreate).toggleClass('ui-icon-plus', isCreate);
    $('.edit .ui-btn-text', footer).text(mobileLex.get(isCreate ? 'create' : 'edit'));
    $('.delete', footer).toggleClass(DISABLED_CLASS, isCreate);
  }

  /**
   * Returns a 'snapshot' of a mounted Schedule - all slots are fully
   * populated all the way down the component tree. The component returned
   * will be *unmounted*.
   *
   * @inner
   * @private
   * @memberOf niagara.schedule.ui
   *
   * @param {baja.Component} component a mounted Component
   * @returns {Promise} to be resolved with the snapshot
   */
  function getSnapshot(component) {
    baja.strictArg(component, baja.Component);

    if (!component.isMounted()) {
      return Promise.reject("Cannot call 'getSnapshot' on an unmounted Component");
    }

    return component.serverSideCall({
      typeSpec: SCHEDULE_SSC_TYPE,
      methodName: 'getSnapshot'
    });
  }

  /**
   * When the current schedule is unmounted from the station, there's nothing
   * left we can do - just show an error dialog and redirect the user back
   * to their home.
   *
   * @memberOf niagara.schedule.ui
   * @private
   */
  function redirectToHomeOnUnmount() {
    currentSchedule.setModified(false);
    dialogs.ok({
      title: mobileLex.get("schedule.scheduleUnmounted"),
      content: mobileLex.get({
        key: 'schedule.message.scheduleUnmounted',
        args: scheduleComponent.getDisplayName()
      }),
      ok: function ok() {
        linkToOrd(baja.getUserHome());
        //don't call cb.ok() because this dialog should never close
      }
    });
  }

  function setLiveComponent(component) {
    var type = component.getType(),
        cPerm = component.getPermissions(),
        events,
        sePerm;

    if (type.is(WEEKLY_SCHEDULE_TYPE)) {
      events = component.get('schedule').get('specialEvents');
    } else if (type.is(CALENDAR_SCHEDULE_TYPE)) {
      events = component;
    } else if (type.is(TRIGGER_SCHEDULE_TYPE)) {
      events = component.get('dates');
    }

    sePerm = events && events.getPermissions();

    scheduleComponent = component;
    $('#main-title').text(scheduleComponent.getDisplayName());

    specialEventsComponent = events;
    scheduleReadonly = !cPerm.hasOperatorWrite() || component.getFlags() & baja.Flags.READONLY || component.has("ext");
    specialEventsRead = sePerm && sePerm.hasOperatorRead();
    specialEventsReadonly = sePerm && !sePerm.hasOperatorWrite() || events && events.getFlags() & baja.Flags.READONLY || component.has("ext");

    $('#main-save, #editDay-edit, #editDay-delete, #editDay-actions, ' + '#specialEvents-add, #editSpecialEvent-edit, ' + '#editSpecialEvent-delete, #editSpecialEvent-actions').toggleClass(DISABLED_CLASS, scheduleReadonly);
  }

  function setSnapshot(snapshot) {
    currentSchedule = new Schedule(snapshot);
    setCurrentSchedule(currentSchedule);
    scheduleSnapshot = snapshot;

    var type = snapshot.getType();

    if (type.is(WEEKLY_SCHEDULE_TYPE)) {
      specialEventsSnapshot = snapshot.get('schedule').get('specialEvents');
    } else if (type.is(CALENDAR_SCHEDULE_TYPE)) {
      specialEventsSnapshot = snapshot;
    } else if (type.is(TRIGGER_SCHEDULE_TYPE)) {
      specialEventsSnapshot = snapshot.get('dates');
    }

    aop.after(currentSchedule, 'setModified', function (args) {
      var modified = args[0];
      $('.commandsButton').toggleClass('red', modified);
      $('#main-save').toggleClass(DISABLED_CLASS, !modified);
    });
    currentSchedule.setModified(false);
    currentSchedule.setEnabled(false);

    return currentSchedule.initialize($('#schedule').empty());
  }

  /**
   * Retrieves a snapshot of the currently viewed schedule from the server,
   * overwrites the last downloaded instance of it, and updates the
   * schedule display.
   *
   * @memberOf niagara.schedule.ui
   * @returns {Promise} will be resolved when the schedule has been fully
   * reretrieved
   */
  function reretrieveSchedule() {
    var subscriber = new baja.Subscriber(),
        liveComponent;

    subscriber.attach('unmount', redirectToHomeOnUnmount);

    return baja.Ord.make(niagara.view.ord).get({ subscriber: subscriber }).then(function (c) {
      liveComponent = c;
      if (liveComponent.has('schedule')) {
        return liveComponent.get('schedule').loadSlots();
      }
    }).then(function () {
      setLiveComponent(liveComponent);
      return getSnapshot(liveComponent);
    }).then(function (snapshot) {
      return setSnapshot(snapshot);
    });
  }

  /**
   * Show a "schedule saved successfully" dialog.
   *
   * @memberOf niagara.schedule.ui
   * @returns {Promise} promise to be resolved after user has clicked ok
   */
  function confirmSave() {
    return new Promise(function (resolve) {
      dialogs.ok({
        title: mobileLex.get('schedule.saved'),
        content: mobileLex.get('schedule.message.savedSuccessfully'),
        ok: function ok(cb) {
          cb.ok(); //close dialog
          resolve();
        }
      });
    });
  }

  /**
   * @returns {Promise}
   */
  function saveDayEditor() {
    if (!dayEditor || !dayEditor.isModified()) {
      return Promise.resolve();
    } else {
      currentSchedule.setModified(true);
      return dayEditor.save();
    }
  }

  /**
   * Sends our instance of the schedule up to the server to be saved.
   *
   * @returns {Promise} will be resolved when the schedule has been
   * fully saved up to the station and redrawn on the screen.
   */
  function save() {
    return saveDayEditor().then(function () {
      return currentSchedule.save();
    }).then(function () {
      var snapshot = currentSchedule.snapshot;
      delete currentSchedule.snapshot;
      return setSnapshot(snapshot);
    }).then(function () {
      return confirmSave();
    }).then(redrawSchedule).catch(dialogs.error);
  }

  /**
   * @memberOf niagara.schedule.ui
   * @private
   * @returns {Promise}
   */
  function confirmAbandonChanges() {
    return new Promise(function (resolve, reject) {
      if (currentSchedule.isModified()) {
        dialogs.confirmAbandonChanges({
          viewName: escapeHtml(scheduleLex.get('scheduler.weeklySchedule')),
          yes: function yes(cb) {
            save().then(function () {
              cb.ok();
              resolve();
            });
          },
          no: function no(cb) {
            reretrieveSchedule().then(function () {
              cb.ok();
              resolve();
            });
          },
          cancel: function cancel() {
            dialogs.closeCurrent();
            // promise never resolves...
          }
        });
      } else {
        resolve();
      }
    });
  }

  /**
   * Hides JQM loading message.
   */
  function hideLoading() {
    hidePageLoadingMsg();
  }

  /**
   * Adds the navigation header (weekly schedule, special events, etc) to a
   * JQM page.
   * @param {JQuery} page the JQM page to append the navbar to
   */
  function appendNavbar(page) {
    var pageId = page.attr('id'),
        headerDiv = page.children(JQM_HEADER_SELECTOR),
        navbar = $('<div data-role="navbar"/>').appendTo(headerDiv),
        ul = $('<ul/>').appendTo(navbar),
        type = scheduleComponent.getType(),
        isWeekly = type.is(WEEKLY_SCHEDULE_TYPE),
        isTrigger = type.is(TRIGGER_SCHEDULE_TYPE);

    if (isWeekly) {
      ul.append(navBarListItem({
        pageId: pageId,
        tabId: 'main',
        text: scheduleLex.get('scheduler.weeklySchedule')
      }));
    }

    ul.append(navBarListItem({
      pageId: pageId,
      tabId: 'specialEvents',
      text: scheduleLex.get('scheduler.specialEvents')
    }));

    if (isTrigger) {
      ul.append(navBarListItem({
        pageId: pageId,
        tabId: 'triggers',
        text: mobileLex.get('schedule.triggers')
      }));
    }

    if (isWeekly) {
      ul.append(navBarListItem({
        pageId: pageId,
        tabId: 'properties',
        text: scheduleLex.get('scheduler.properties')
      }));
    }

    ul.append(navBarListItem({
      pageId: pageId,
      tabId: 'summary',
      text: scheduleLex.get('summary')
    }));

    if (!specialEventsRead) {
      headerDiv.find('.to-specialEvents').addClass(DISABLED_CLASS);
    }
  }

  /**
   * Displays a field editor for a `schedule:TimeSchedule` in a dialog. Used on
   * both the `editDay` and `editSpecialEvent` pages when editing a draggable
   * schedule block directly using the datebox field editors.
   *
   * @memberOf niagara.schedule.ui
   * @private
   * @param {module:mobile/fieldeditors/BaseFieldEditor} timeScheduleEditor the
   * field editor for the `schedule:TimeSchedule`
   * @returns {Promise}
   */
  function showTimeScheduleDialog(timeScheduleEditor) {
    return new Promise(function (resolve, reject) {
      dialogs.okCancel({

        title: mobileLex.get('schedule.' + (createMode ? 'createBlock' : 'editBlock')),

        content: function content(targetElement) {
          return timeScheduleEditor.buildAndLoad(targetElement).then(function () {
            if (createMode) {
              return;
            }

            //arm an event listener on the time editors so we validate the
            //time range each time a new value is entered
            timeScheduleEditor.jq().on('datebox', 'input:jqmData(role="datebox")', function (e, passed) {
              if (passed.method === 'offset') {
                var changems;
                switch (passed.type) {
                  case 'h':
                    changems = MILLIS_IN_HOUR * passed.amount;
                    break;
                  case 'i':
                    changems = MILLIS_IN_MINUTE * passed.amount;
                    break;
                }
                validateTimeScheduleEditor($(this), dayEditor, timeScheduleEditor, changems);
              }
              return false; //validateTimeSchedule sets the datebox content
            });
          });
        },

        //after the OK button is clicked
        ok: function ok(cb) {
          timeScheduleEditor.save().then(function () {
            var timeSchedule = timeScheduleEditor.value();

            //set last entered value - this will be used in
            //niagara.schedule.ui.day so new blocks will be have this value
            //as default
            setLastEnteredValue(timeSchedule.get('effectiveValue'));
            dayEditor.setModified(true);
            cb.ok();
            resolve(timeSchedule);
          }).catch(function (err) {
            cb.fail(err);
            reject(err);
          });
        },

        //after the dialog is closed
        callbacks: {
          ok: function ok() {

            //Upon hiding the block editor, delete any cruft the datebox plugin
            //left lying around the DOM since datebox doesn't know how to clean
            //up after itself
            $('#okCancelDialog').find('.ui-datebox-screen, .ui-datebox-container').remove();
            getPageContainer().children('.ui-dialog-datebox').remove();
          },
          fail: dialogs.error
        }
      });
    });
  }

  /**
   * Attempts to edit the currently selected schedule block. Shows a field
   * editor for the block's day schedule, with editors for start time, end
   * time, and effective value. Also arms handlers on the field editor to
   * perform validation on the entered time range (prevent overlapping with
   * another block etc).
   *
   * @param {module:mobile/schedule/ScheduleBlock} block the block to edit
   * @returns {Promise} promise to be resolved with the same block after
   * changes are saved
   */
  function editBlock(block) {
    if (!block) {
      return Promise.resolve();
    }

    return fe.makeFor({
      value: block.createTimeSchedule(),
      facets: currentSchedule.value().get('facets'),
      autoInitialize: false
    }).then(function (timeScheduleEditor) {
      timeScheduleEditor.validators().add(function (timeSchedule) {
        var params = {
          startTime: timeSchedule.get('start'),
          finishTime: timeSchedule.get('finish'),
          dayEditor: dayEditor,
          mschange: 0
        };
        return Promise.all([validateTimeChange($.extend(params, { slot: 'start' })), validateTimeChange($.extend(params, { slot: 'finish' }))]);
      });
      return showTimeScheduleDialog(timeScheduleEditor);
    }).then(function (timeSchedule) {
      //re-run constructor to overwrite properties
      ScheduleBlock.call(block, timeSchedule);
      return block;
    });
  }

  /**
   * Show a time schedule edit dialog. If we are in create mode (no block
   * selected), then create a new time schedule and edit it (without the
   * real-time time range validation as this makes no sense on a brand new
   * time schedule). If not in create mode, then just edit the selected
   * block.
   */
  function createOrEdit() {
    if (createMode) {
      var block = new ScheduleBlock(baja.Time.DEFAULT, baja.Time.DEFAULT, currentSchedule.getNewBlockValue());
      return editBlock(block).then(function (block) {
        dayEditor.addBlock(block.start, block.finish, block.value);
      });
    } else {
      return editBlock(dayEditor.getSelectedBlock());
    }
  }

  /**
   * Removes any handlers the last edited day may have registered on
   * `$(document)` - see `niagara.schedule.ui.day.createHandlers()`. Then
   * instantiates a new editor for the currently edited day.
   *
   * @param {Object} params an object literal to be passed into
   * `fieldEditors.makeFor` to create the day field editor
   * @param {baja.Component} params.value the `schedule:DaySchedule` we want to
   * start editing
   * @param {JQuery} [params.element] where to build the `DayEditor`
   * @returns {Promise} will be resolved when the day editor is fully
   * instantiated and loaded
   */
  function resetDayEditor(params) {
    params = baja.objectify(params, 'value');

    if (!params.value) {
      return Promise.resolve();
    }

    if (dayEditor && dayEditor.unbindAll) {
      dayEditor.unbindAll();
      dayEditor.jq().remove();
    }

    return fe.makeFor(_.extend({}, params, {
      container: params.value.getParent(),
      slot: params.value.getPropertyInParent(),
      element: params.element || dayEditor && dayEditor.jq().parent()
    })).then(function (editor) {
      dayEditor = editor;
      return editor.load(params.value);
    });
  }

  //these are the commands shown when clicking "options" on a day editor,
  //equivalent to right-clicking a schedule block in the Workbench schedule
  //editor
  blockCommands = {
    'delete': new Command("%lexicon(schedule:day.delete)%", function () {
      dayEditor.removeBlock(dayEditor.getSelectedBlock());
    }),
    'all_day': new Command("%lexicon(schedule:day.all_day)%", function () {
      var block = dayEditor.getSelectedBlock();
      dayEditor.allDayEvent(block ? block.value : currentSchedule.getNewBlockValue());
    }),
    'apply_weekdays': new Command("%lexicon(schedule:day.apply_weekdays)%", function () {
      currentSchedule.applyMF(dayEditor);
    }),
    'copy_day': new Command("%lexicon(schedule:day.copy_day)%", function () {
      copiedDayEditor = dayEditor;
    }),
    'clear_day': new Command("%lexicon(schedule:day.clear_day)%", function () {
      dayEditor.empty();
    }),
    'clear_week': new Command("%lexicon(schedule:day.clear_week)%", function () {
      currentSchedule.empty();
      return resetDayEditor(editedDaySchedule);
    }),
    'paste_day': new Command("%lexicon(schedule:day.paste_day)%", function () {
      if (copiedDayEditor) {
        copiedDayEditor.copyTo(dayEditor);
      }
    })
  };

  /**
   * Shows a dialog with options (apply M-F, Clear Day, etc) for a schedule
   * block or day. If the current day editor does not have a schedule block
   * selected, then no block-specific actions (e.g. delete event) will be
   * shown, only day-specific ones.
   *
   * @param {Object} optionsToShow a mapping from the different option keys
   * (corresponding to day.* entries in the schedule lexicon) to boolean
   * values indicating whether or not that option should be shown.
   */
  function showBlockActions(optionsToShow) {
    var block = dayEditor.getSelectedBlock(),
        filteredCommands = [],
        title = dayEditor.$dayDisplay + (block ? ' (' + block.toString() + ')' : '');

    _.each(blockCommands, function (command, commandName) {
      if (optionsToShow[commandName]) {
        filteredCommands.push(command);
      }
    });

    commands.showCommandsDialog(filteredCommands, title);
  }

  /**
   * Registers event listeners using the pages framework.
   *
   * @namespace
   * @name niagara.schedule.ui.pages
   * @function
   * @private
   */
  function registerPages() {
    var oldHomeCommand = commands.getHomeCmd(),
        defaultCommands = commands.getDefaultCommands(),
        oldHomeIndex = _.indexOf(defaultCommands, oldHomeCommand),
        saveCommand,
        refreshCommand;

    function doRetrieve() {
      return confirmAbandonChanges().then(function () {
        return reretrieveSchedule();
      });
    }

    function doConfirm() {
      if (currentSchedule.isModified() || dayEditor && dayEditor.isModified()) {
        return new Promise(function (resolve, reject) {
          dialogs.confirmAbandonChanges({
            viewName: scheduleLex.get('scheduler.weeklySchedule'),
            yes: function yes() {
              save().then(function () {
                return oldHomeCommand.invoke();
              }).then(resolve, reject);
            },
            no: function no() {
              currentSchedule.setModified(false); //prevent onbeforeunload
              oldHomeCommand.invoke().then(resolve, reject);
            }
          });
        });
      } else {
        return oldHomeCommand.invoke();
      }
    }

    saveCommand = new Command("%lexicon(mobile:save)%", function () {
      return save();
    });
    refreshCommand = new Command("%lexicon(mobile:refresh)%", function () {
      return doRetrieve();
    });

    oldHomeCommand.toDisplayName().then(function (displayName) {
      defaultCommands[oldHomeIndex] = new Command(displayName, doConfirm);
      commands.setDefaultCommands(defaultCommands);
    });

    /**
     * Adds a save command (only if the current schedule is modified) and a
     * refresh command to the list of default commands.
     */
    function getDefaultCommands(obj) {
      var cmds = currentSchedule.isModified() ? [saveCommand, refreshCommand] : [refreshCommand];
      return cmds.concat(obj.commands);
    }

    pages.register("main", function main() {

      /**
       * Arm save/refresh buttons in footer nav bar, and enable linking to
       * day editor when clicking a day in the main schedule display.
       * @memberOf niagara.schedule.ui.pages.main
       */
      function pagebeforecreate(obj) {
        appendNavbar(obj.page);

        preventNavbarHighlight($('#main-footer'));

        $('#main-save').click(function () {
          var $this = $(this);
          if (!$this.hasClass(DISABLED_CLASS)) {
            $this.addClass(DISABLED_CLASS);
            save().catch(logError);
          }
        });

        $('#main-refresh').click(function () {
          confirmAbandonChanges().then(function () {
            return reretrieveSchedule();
          }).then(redrawSchedule).catch(logError);
        });

        $('#schedule').on('click', 'div.day', function () {
          pages.getHandler('editDay').load($(this).data('daySchedule'));
        });
      }

      /**
       * Sets the height of the `#main` page's content div to fill the screen,
       * then redraws the weekly schedule display to fill that space. Called
       * when the page is loaded or when the window is resized.
       *
       * @memberOf niagara.schedule.ui.pages.main
       * @see niagara.schedule.ui.initializeUI
       */
      function redraw() {
        setContentHeight($('#main'));
        if (currentSchedule) {
          redrawSchedule();
        }
      }

      /**
       * On every page view, if we have a Schedule object loaded, call
       * `redraw()` to update the display.
       *
       * @memberOf niagara.schedule.ui.pages.main
       */
      function pagelayout() {
        $('#main-main').addClass(BTN_ACTIVE_CLASS);
        redraw();
      }

      /**
       * If we're not actually viewing a weekly schedule (i.e. a calendar or
       * trigger schedule), this page shouldn't even be accessible - so redirect
       * the user directly to the special events page.
       * @memberOf niagara.schedule.ui.pages.main
       */
      function pagebeforeshow(obj) {
        if (scheduleComponent && !scheduleComponent.getType().is(WEEKLY_SCHEDULE_TYPE)) {
          jqm.changePage('#specialEvents', {
            changeHash: false,
            transition: 'none'
          });
          obj.event.preventDefault();
        }
      }

      /**
       * @namespace
       * @private
       * @name niagara.schedule.ui.pages.main
       */
      return {
        pagebeforeshow: pagebeforeshow,
        pagebeforecreate: pagebeforecreate,
        pagelayout: pagelayout,
        redraw: redraw,

        getCommands: getDefaultCommands
      };
    }());

    pages.register("editDay", function editDay() {
      /**
       * Returns a click handler to be bound to a back button. If the currently
       * edited day is in a modified state, will pop up a dialog asking the user
       * whether or not to save changes before allowing the backward navigation
       * to take place.
       *
       * @memberOf niagara.schedule.ui.pages.editDay
       * @private
       */
      function checkForChangesFunction(targetPage) {

        return function () {
          if (dayEditor.isModified()) {
            dialogs.confirmAbandonChanges({
              yes: saveDayEditor,
              redirect: targetPage,
              viewName: mobileLex.get('schedule.dayEditor')
            });
            return false;
          }
        };
      }

      /**
       * Loads the available actions for the currently edited day (copy/paste,
       * all day, apply MF, etc.). The actions shown will differ depending on
       * whether or not a schedule block is currently selected.
       * @memberOf niagara.schedule.ui.pages.editDay
       * @private
       */
      function loadBlockActions() {
        var selectedBlock = dayEditor.getSelectedBlock(),
            hasBlocks = !!dayEditor.scheduleBlocks.length,
            hasSelected = !!selectedBlock,
            hasCopied = !!copiedDayEditor;

        showBlockActions({
          'delete': hasSelected,
          'paste_day': hasCopied,
          'all_day': true,
          'apply_weekdays': hasBlocks,
          'copy_day': hasBlocks,
          'clear_day': hasBlocks,
          'clear_week': true
        });
      }

      /**
       * @memberOf niagara.schedule.ui.pages.editDay
       */
      function pagecreate(obj) {
        var backButton = obj.page.children(JQM_HEADER_SELECTOR).find('a.profileHeaderBack');
        backButton.on('click', checkForChangesFunction('#main'));
      }

      /**
       * Sets the `#editDay` page's content div to fill the screen, then
       * redraws the day editor to fill that space. Does not add/remove blocks
       * to reflect the structure of the underlying day schedule - only moves
       * the existing divs around to be correctly laid out.
       *
       * Runs when the page is shown or when the window is resized.
       *
       * @memberOf niagara.schedule.ui.pages.editDay
       * @see niagara.schedule.ui.initializeUI
       */
      function relayout() {
        var dayEditorDiv = $('#editDay-day');
        setContentHeight($('#editDay'));
        if (dayEditor) {
          dayEditor.layout();
        }
        relayoutLabelsDiv(dayEditorDiv.find('.scheduleLabels'), dayEditorDiv.find('.blocksDisplay').height());
      }

      /**
       * Completely rebuilds the structure of the day editor to reflect the
       * current state of the day schedule, throwing away any user edited
       * changes.
       *
       * Runs each time the page is loaded (when clicking on a day on the main
       * schedule tab), and after clicking Refresh. `relayout` will be called
       * after the day editor is rebuilt.
       *
       * @memberOf niagara.schedule.ui.pages.editDay
       * @private
       * @returns {Promise}
       */
      function rebuild() {
        var dayDiv = $('#editDay-day').empty(),
            parent = editedDaySchedule.getParent(),
            parentName = parent && parent.getName(),
            label = parentName && bajaLex.get(parentName),
            params = {
          value: editedDaySchedule,
          element: dayDiv,
          readonly: scheduleReadonly,
          label: label
        };

        appendTimeLabels(dayDiv);

        return resetDayEditor(params).then(function () {
          setCreateMode($('#editDay-footer'), true);
          relayout();
        });
      }

      /**
       * Repaints the day editor to ensure that any user entered changes
       * (from clicking the create/delete/edit footer buttons) are reflected
       * on the screen.
       *
       * @memberOf niagara.schedule.ui.pages.editDay
       * @private
       */
      function refresh() {
        dayEditor.refreshWidgets();
        setCreateMode($('#editDay-footer'), true);
        relayout();
      }

      /**
       * Arms click handlers on the edited day.
       * @memberOf niagara.schedule.ui.pages.editDay
       */
      function pagebeforecreate() {
        $('#editDay-day').on('selectionchange', function () {
          //toggle edit/create when we select/deselect a block
          var selected = !!dayEditor.getSelectedBlock();
          setCreateMode($('#editDay-footer'), !selected);
        }).on('dblclick', function () {
          //edit a block by double clicking
          editBlock(dayEditor.getSelectedBlock()).catch(logError);
        });

        $('#editDay-footer').on('click', 'a', function () {
          var $this = $(this);

          switch ($this.attr('id')) {

            case 'editDay-save':
              saveDayEditor().catch(logError);
              break;

            case 'editDay-refresh':
              if (dayEditor.isModified()) {
                dialogs.confirmAbandonChanges({
                  yes: function yes(cb) {
                    saveDayEditor().then(cb.ok, cb.fail);
                  },
                  no: function no(cb) {
                    rebuild().then(cb.ok, cb.fail);
                  },
                  viewName: mobileLex.get('schedule.dayEditor')
                });
              } else {
                rebuild().catch(logError);
              }
              break;

            case 'editDay-actions':
              loadBlockActions();
              break;

            case 'editDay-edit':
              createOrEdit().then(refresh);
              break;

            case 'editDay-delete':
              dayEditor.removeBlock(dayEditor.getSelectedBlock());
              refresh();
              break;
          }
        });
      }

      /**
       * Updates the day editor div to show the current state of the edited
       * day.
       * @memberOf niagara.schedule.ui.pages.editDay
       */
      function pagebeforeshow() {
        refresh();
        setCreateMode($('#editDay-footer'), true);
      }

      /**
       * Loads, and instigates a page change to, the page to edit the specified
       * day. The current day editor will be completely reloaded and refreshed
       * to begin editing the day.
       *
       * @memberOf niagara.schedule.ui.pages.editDay
       * @param {baja.Component} daySchedule the `schedule:DaySchedule` to edit
       */
      function load(daySchedule) {
        editedDaySchedule = daySchedule;

        rebuild().then(function () {
          jqm.changePage('#editDay');
        }).catch(logError);
      }

      /**
       * Ensures that the day editor paints correctly given the current
       * visible height.
       * @memberOf niagara.schedule.ui.pages.editDay
       */
      function pagelayout() {
        relayout();
      }

      /**
       * If the day editor is modified, prompts for changes before navigating
       * away.
       * @memberOf niagara.schedule.ui.pages.editDay
       */
      function pagebeforechange(obj) {
        if (dayEditor.isModified()) {
          dialogs.confirmAbandonChanges({
            yes: function yes(cb) {
              saveDayEditor().then(cb.ok, cb.fail);
            },
            no: function no(cb) {
              resetDayEditor(editedDaySchedule).then(cb.ok, cb.fail);
            },
            redirect: obj.nextPage,
            viewName: mobileLex.get('schedule.dayEditor')
          });
          obj.event.preventDefault();
        }
      }

      /**
       * @namespace
       * @private
       * @name niagara.schedule.ui.pages.editDay
       */
      return {
        load: load,
        rebuild: rebuild,
        relayout: relayout,
        pagebeforechange: pagebeforechange,
        pagebeforecreate: pagebeforecreate,
        pagebeforeshow: pagebeforeshow,
        pagecreate: pagecreate,
        pagelayout: pagelayout
      };
    }());

    pages.register("properties", function () {
      var propertyEditors, effectiveRangeEditor;

      /**
       * @memberOf niagara.schedule.ui.pages.properties
       * @private
       * @param {jQuery} targetElement the element containing the calendar
       * datebox
       * @param {baja.AbsTime} newTime the new time (month) to load into the calendar
       * datebox
       */
      function doCalendarUpdate(targetElement, newTime) {
        return effectiveRangeEditor.read().then(function (effectiveRange) {
          return calendarUI.showEffectiveRangeCalendar(targetElement, effectiveRange, newTime);
        });
      }

      /**
       * Validates and saves the Properties tab field editors for effective
       * date range, facets, etc.
       * @memberOf niagara.schedule.ui.pages.properties
       * @private
       * @returns {Promise} will be resolved when all editors are validated
       * and saved
       */
      function validateAndSaveEditors() {
        var editors = propertyEditors.slice();

        editors.push(effectiveRangeEditor);

        var validates = _.map(editors, function (editor) {
          return editor.isModified() && editor.validate();
        });

        return Promise.all(validates).then(function () {
          var saves = _.map(editors, function (editor) {
            return editor.save();
          });
          return Promise.all(saves);
        }).then(function () {
          currentSchedule.setModified(true);
        });
      }

      /**
       * Arms handlers for save/cancel and to ensure the calendar preview
       * updates its display when a different month is shown.
       * @memberOf niagara.schedule.ui.pages.properties
       */
      function pagebeforecreate(obj) {
        //if user clicks +/- a bunch of times in a row, don't go through
        //a round trip to the server until they've settled on a month
        var debouncedCalendarUpdate = _.debounce(function (targetElement, newTime) {
          doCalendarUpdate(targetElement, newTime).catch(logError);
        }, 600);

        appendNavbar(obj.page);

        $('#properties-save').click(function () {
          validateAndSaveEditors().catch(logError);
        });

        $('#properties-calendars').on('datebox', 'input:jqmData(role="datebox")', function (e, passed) {
          if (passed.method === 'offset') {
            //+ or - button clicked
            var newTime = baja.AbsTime.make({ jsDate: passed.newDate });
            debouncedCalendarUpdate($('#properties-calendars'), newTime);
          }
        });

        preventNavbarHighlight($('#properties-footer'));
      }

      /**
       * Instantiates the field editors for the schedule's properties
       * (facets, default output, cleanup special events). After this function
       * (asynchronously) completes, the `propertyEditors` variable will be
       * populated with an array containing field editor instances for the
       * schedule's `defaultOutput`, `facets`, and `cleanupExpiredEvents`
       * properties.
       *
       * @memberOf niagara.schedule.ui.pages.properties
       * @private
       * @param {JQuery} element the div in which to display the field editors
       * @returns {Promise}
       */
      function loadPropertiesEditors(element) {
        element.empty();
        propertyEditors = [];

        var slots = ['defaultOutput', 'facets', 'cleanupExpiredEvents'],
            schedule = currentSchedule.value();

        var loads = _.map(slots, function (slot) {
          var div = $('<div></div>').addClass(slot).appendTo(element),
              labeled = fe.toLabeledEditorContainer(String(slot), div);

          return fe.makeFor({
            element: labeled,
            container: schedule,
            slot: slot,
            facets: schedule.get('facets'),
            readonly: scheduleReadonly
          }).then(function (ed) {
            propertyEditors.push(ed);
          });
        });

        return Promise.all(loads);
      }

      /**
       * Loads and shows the necessary field editors and calendar preview
       * for the properties tab.
       * @memberOf niagara.schedule.ui.pages.properties
       */
      function pagebeforeshow() {
        var sched = currentSchedule.value();

        loadPropertiesEditors($('#properties-editors')).catch(logError);

        calendarUI.loadScheduleEditor($('#properties'), sched.get('effective'), scheduleReadonly).then(function (editor) {
          calendarUI.bindEditorToCalendar(editor, $('#properties-calendars'));
          effectiveRangeEditor = editor;
        }).catch(function (err) {
          baja.error("properties: could not load schedule editor: " + err);
        });

        calendarUI.showEffectiveRangeCalendar($('#properties-calendars'), sched.get('effective'), baja.AbsTime.now()).catch(logError);

        $('#properties-properties').addClass(BTN_ACTIVE_CLASS);
        $('#properties-title').text(scheduleComponent.getDisplayName());
      }

      function pageshow() {
        hideLoading();
      }

      /**
       * @namespace
       * @private
       * @name niagara.schedule.ui.pages.properties
       */
      return {
        pagebeforecreate: pagebeforecreate,
        pagebeforeshow: pagebeforeshow,
        pageshow: pageshow,
        getCommands: getDefaultCommands
      };
    }());

    pages.register("specialEvents", function specialEvents() {
      var radioView, SpecialEventsRadioView, events;

      /**
       * A radio button view for showing the current schedule's special
       * events on the `specialEvents` page.
       *
       * @class
       * @memberOf niagara.schedule.ui.pages.specialEvents
       * @private
       * @extends niagara.util.mobile.RadioButtonView
       */
      SpecialEventsRadioView = baja.subclass(function () {
        baja.callSuper(SpecialEventsRadioView, this, arguments);
      }, RadioButtonView);

      /**
       * Only show those slots that are of type `schedule:AbstractSchedule`.
       * @param {baja.Property} slot
       */
      SpecialEventsRadioView.prototype.shouldIncludeSlot = function shouldIncludeSlot(slot) {
        return slot.isProperty() && slot.getType().is(ABSTRACT_SCHEDULE_TYPE);
      };

      SpecialEventsRadioView.prototype.doLoad = function (value) {
        if (value.getSlots().properties().dynamic().toArray().length) {
          return RadioButtonView.prototype.doLoad.call(this, value);
        } else {
          this.jq().text(mobileLex.get('schedule.message.noSpecialEvents'));
        }
      };

      /**
       * Makes multiline labels for the radio buttons - display name on top,
       * display value below.
       */
      SpecialEventsRadioView.prototype.makeLabel = function (complex, slot) {
        var pageId = encodePageId(String(slot)),
            display = complex.getDisplay(slot),
            displayName = baja.SlotPath.unescape(String(slot)),
            label = $('<label data-theme="c"/>').attr('for', pageId);

        label.append($('<span/>').text(displayName)).append($('<br/>')).append($('<span/>').text(display));

        return label;
      };

      /**
       * After the user selects a special event, enable the footer buttons
       * for editing.
        */
      aop.after(SpecialEventsRadioView.prototype, 'selectionChanged', function (args) {
        if (!specialEventsReadonly) {
          $('#specialEvents-footer').find('a').removeClass(DISABLED_CLASS);
          var slot = String(args[0]),
              value = this.value(),
              slots = value.getSlots().is(ABSTRACT_SCHEDULE_TYPE).toArray(),
              firstSlot = slots[0],
              lastSlot = slots[slots.length - 1];

          if (firstSlot && slot === String(firstSlot)) {
            $('#specialEvents-priorityUp').addClass(DISABLED_CLASS);
          }

          if (lastSlot && slot === String(lastSlot)) {
            $('#specialEvents-priorityDown').addClass(DISABLED_CLASS);
          }
        } else {
          $('#specialEvents-edit').removeClass(DISABLED_CLASS);
        }
      });

      /**
       * Refreshes the special events radio buttons to reflect any changes
       * the user may have made on the edit screen.
       *
       * @memberOf niagara.schedule.ui.pages.specialEvents
       * @private
       * @returns {Promise} will be resolved when the special events radio
       * button list has been fully refreshed
       */
      function redrawSpecialEvents() {
        return radioView.initialize($('#specialEvents-eventList').empty()).then(function () {
          return radioView.load(events);
        }).then(function () {
          radioView.setSelectedSlot(radioView.$selectedSlot);
        });
      }

      /**
       * Enable click handlers for selecting a special event and manipulating
       * it via the buttons in the footer bar.
       *
       * @memberOf niagara.schedule.ui.pages.specialEvents
       */
      function pagebeforecreate(obj) {
        appendNavbar(obj.page);
        radioView = new SpecialEventsRadioView();

        var navBar = $('#specialEvents-navbar');

        preventNavbarHighlight(navBar);

        navBar.on('click', 'a', function () {
          var $this = $(this),
              selectedSlot = String(radioView.getSelectedSlot());

          if ($this.hasClass(DISABLED_CLASS)) {
            return;
          }

          switch ($(this).attr('id')) {
            case "specialEvents-add":
              pages.getHandler('addSpecialEvent').load(events);
              break;
            case "specialEvents-edit":
              pages.getHandler('editSpecialEvent').load(radioView.getSelectedValue());
              break;
            case "specialEvents-priorityUp":
              moveUp(events, selectedSlot).catch(logError);
              redrawSpecialEvents().catch(logError);
              currentSchedule.setModified(true);
              break;
            case "specialEvents-priorityDown":
              moveDown(events, selectedSlot).catch(logError);
              redrawSpecialEvents().catch(logError);
              currentSchedule.setModified(true);
              break;
            case "specialEvents-delete":
              dialogs.yesNo({
                content: mobileLex.get({
                  key: 'schedule.message.confirmDeleteSpecialEvent',
                  def: '{0}',
                  args: [baja.SlotPath.unescape(String(selectedSlot))]
                }),
                yes: function yes(cb) {
                  events.remove(selectedSlot).then(function () {
                    radioView.setSelectedSlot(null);
                    return redrawSpecialEvents();
                  }).then(function () {
                    currentSchedule.setModified(true);
                    cb.ok();
                  });
                }
              });
              break;
            case "specialEvents-rename":
              feDialogs.fieldEditor({
                title: scheduleLex.get('composite.rename'),
                value: baja.SlotPath.unescape(selectedSlot)
              }).then(function (newName) {
                if (newName === null) {
                  return;
                }
                newName = baja.SlotPath.escape(newName);
                if (newName !== selectedSlot) {
                  return events.rename({ slot: selectedSlot, newName: newName }).then(function () {
                    currentSchedule.setModified(true);
                    return redrawSpecialEvents();
                  });
                }
              }).catch(dialogs.error);
              break;
          }
        });
      }

      /**
       * Populate the list of special events to be displayed.
       * @memberOf niagara.schedule.ui.pages.specialEvents
       */
      function pagebeforeshow() {
        events = specialEventsSnapshot;
        redrawSpecialEvents().then(function () {
          if (!radioView.getSelectedSlot()) {
            $('#specialEvents-footer').find('a').addClass(DISABLED_CLASS);
          }

          if (!specialEventsReadonly) {
            $('#specialEvents-add').removeClass(DISABLED_CLASS);
          }
          $('#specialEvents-specialEvents').addClass(BTN_ACTIVE_CLASS);
          $('#specialEvents-title').text(scheduleComponent.getDisplayName());
          $('#specialEvents-edit').find('.ui-btn-text').text(mobileLex.get(specialEventsReadonly ? 'view' : 'edit'));
        });
      }

      /**
       * @namespace
       * @private
       * @name niagara.schedule.ui.pages.specialEvents
       */
      return {
        pagebeforecreate: pagebeforecreate,
        pagebeforeshow: pagebeforeshow,
        getCommands: getDefaultCommands
      };
    }());

    pages.register('triggers', function triggers() {
      var triggersEditor;

      /**
       * @memberOf niagara.schedule.ui.pages.triggers
       */
      function pagebeforeshow() {
        $('#triggers-triggers').addClass(BTN_ACTIVE_CLASS);
        $('#triggers-remove').addClass(DISABLED_CLASS);

        fe.makeFor({
          value: scheduleSnapshot.get('times'),
          key: 'triggers-list',
          element: $('#triggers-triggerList').empty()
        }).then(function (editor) {
          triggersEditor = editor;
        });
      }

      /**
       * @memberOf niagara.schedule.ui.pages.triggers
       */
      function pagebeforecreate(obj) {
        appendNavbar(obj.page);

        $('#triggers-triggerList').on(events.MODIFY_EVENT, baja.throttle(function () {
          $('#triggers-remove').toggleClass(DISABLED_CLASS, !triggersEditor.getSelectedSlots().length);
        }, 100)).on('change', '#selectAll', function () {
          var checked = $(this).is(':checked');
          $('#triggers-triggerList').find('.ui-li-static input[type=checkbox]').prop('checked', checked).checkboxradio('refresh');
        });

        $('#triggers-add').click(function () {
          feDialogs.fieldEditor({
            title: mobileLex.get('schedule.addTrigger'),
            value: scheduleSnapshot.get('times'),
            key: 'triggers-add'
          }).then(function (value) {
            if (value !== null) {
              currentSchedule.setModified(true);
            }
          }).catch(dialogs.error);
        });

        $('#triggers-remove').click(function () {
          triggersEditor.removeSelected().then(function () {
            currentSchedule.setModified(true);
            $('#triggers-triggerList').trigger('updatelayout');
            $('#triggers-remove').addClass(DISABLED_CLASS);
          }).catch(logError);
        });

        preventNavbarHighlight($('#triggers-navbar'));
      }

      /**
       * @namespace
       * @private
       * @name niagara.schedule.ui.pages.triggers
       */
      return {
        pagebeforeshow: pagebeforeshow,
        pagebeforecreate: pagebeforecreate,
        getCommands: getDefaultCommands
      };
    }());

    pages.register('addSpecialEvent', function addSpecialEvent() {
      var editedEvents, newScheduleEditor;

      /**
       * Sequential async workflow to add a new special event. First the
       * special event editor must be saved, then the new special event added
       * to the current list of special events. Finally it must retrieve
       * a display string from the station and update the slot's display
       * value (since toString on an AbstractSchedule is too complicated to
       * really do browserside).
       *
       * @memberOf niagara.schedule.ui.pages.addSpecialEvent
       * @private
       * @field
       */
      function addNewSpecialEvent(name) {
        var specialEvent,
            slot = baja.SlotPath.escape(name);

        return newScheduleEditor.save().then(function () {
          var savedSchedule = newScheduleEditor.value();

          if (scheduleComponent.getType().is(WEEKLY_SCHEDULE_TYPE)) {
            specialEvent = baja.$(DAILY_SCHEDULE_TYPE, {
              days: savedSchedule
            });
          } else {
            specialEvent = savedSchedule;
          }

          return editedEvents.add({
            slot: slot,
            value: specialEvent
          });
        }).then(function () {
          return scheduleComponent.serverSideCall({
            typeSpec: SCHEDULE_SSC_TYPE,
            methodName: 'getDailyScheduleDisplayString',
            value: specialEvent
          });
        }).then(function (displayString) {
          //complex.getDisplay(slot) just returns slot.$display
          //TODO: is there a non-ugly way of doing this?
          editedEvents.getSlot(slot).$display = displayString;
          moveToTop(editedEvents, slot);
          currentSchedule.setModified(true);
          history.back();
        });
      }

      /**
       * When creating a new DateSchedule, set the default values for
       * day/month/year to match the current date.
       *
       * @memberOf niagara.schedule.ui.pages.addSpecialEvent
       * @private
       */
      function setNewDateScheduleDefaults(div) {
        var date = new Date();

        div.find('.schedule-DayOfMonthSchedule').find('select').val(String(date.getDate())).selectmenu('refresh');

        div.find('.schedule-MonthSchedule').find('select').val(String(date.getMonth())).selectmenu('refresh');

        div.find('.schedule-YearSchedule').find('select').val(String(date.getFullYear())).selectmenu('refresh');
      }

      /**
       * When creating a new special event, start with a default name -
       * "Event" + (an integer as high as necessary to make it a unique slot
       * name).
       * @memberOf niagara.schedule.ui.pages.addSpecialEvent
       * @private
       * @returns {String} a default name for a new special event
       */
      function generateDefaultName() {
        var i = 0,
            name = "Event";
        while (editedEvents.has(name)) {
          i++;
          name = "Event" + i;
        }
        return name;
      }

      /**
       * Add event handlers to redraw the field editor depending on what
       * type of new special event is selected, and save it once editing is
       * complete.
       * @memberOf niagara.schedule.ui.pages.addSpecialEvent
       */
      function pagebeforecreate() {
        var typesList = $('#addSpecialEvent-typesList');
        if (!scheduleComponent.getType().is(WEEKLY_SCHEDULE_TYPE)) {
          typesList.find('option[value="schedule:ScheduleReference"]').remove();
        }

        /*
         * Sequential async workflow to display a field editor for a newly
         * created special event.
         */
        typesList.change(function () {
          var typeSpec = $(this).val(),
              newSchedule = baja.$(typeSpec);

          fe.makeFor({
            value: newSchedule,
            element: $('#addSpecialEvent-editors').empty()
          }).then(function (editor) {
            var dom = editor.jq();

            if (newSchedule.getType().is(DATE_SCHEDULE_TYPE)) {
              setNewDateScheduleDefaults(dom);
            }
            dom.trigger('updatelayout');
            newScheduleEditor = editor;
          }).catch(dialogs.error);
        });

        $('#addSpecialEvent-editors').on(events.MODIFY_EVENT, function () {
          $('#addSpecialEvent-save').removeClass(DISABLED_CLASS);
        });

        preventNavbarHighlight($('#addSpecialEvent-footer'));

        $('#addSpecialEvent-save').click(function () {
          var name = $('#addSpecialEvent-name').val();

          if (editedEvents.has(name)) {
            dialogs.ok({
              content: mobileLex.get({
                key: 'schedule.message.nameInUse',
                args: [name]
              })
            });
          } else {
            addNewSpecialEvent(name).catch(dialogs.error);
          }
        });
      }

      /**
       * Loads, and instigates a page change to, the addSpecialEvent page.
       * Shows the default field editor and a default slot name.
       * @memberOf niagara.schedule.ui.pages.addSpecialEvent
       * @param {baja.Component} events the `schedule:CompositeSchedule` in the
       * current schedule's `schedule/specialEvents` slot
       */
      function load(events) {
        editedEvents = events;
        jqm.changePage($('#addSpecialEvent'));
        $('#addSpecialEvent-typesList').trigger('change');
        $('#addSpecialEvent-name').val(generateDefaultName());
      }

      /**
       * @namespace
       * @private
       * @name niagara.schedule.ui.pages.addSpecialEvent
       */
      return {
        load: load,
        pagebeforecreate: pagebeforecreate
      };
    }());

    pages.register('editSpecialEvent', function editSpecialEvent() {
      var scheduleEditor,
          editedSpecialEvent,
          dateRangeModified,
          selectedSectionIndex = 0,
          debouncedCalendarUpdate;

      /**
       * Sequential async workflow to save the effective date range on an
       * edited special event. Will retrieve an updated toString from the
       * server for proper display back on the Special Events tab.
       *
       * @memberOf niagara.schedule.ui.pages.editSpecialEvent
       * @private
       * @returns {Promise} will be resolved when the special event has been
       * saved
       */
      function saveScheduleEditor(scheduleEditor) {
        return scheduleEditor.save().then(function () {
          return scheduleComponent.serverSideCall({
            typeSpec: SCHEDULE_SSC_TYPE,
            methodName: 'getDailyScheduleDisplayString',
            value: editedSpecialEvent
          });
        }).then(function (displayString) {
          var slot = editedSpecialEvent.getPropertyInParent();
          slot.$display = displayString; //TODO: shouldn't this be displayNames?
          currentSchedule.setModified(true);
        });
      }

      /**
       * Updates the editSpecialEvent calendar to show the currently effective
       * date range (based on user input on the Active Dates tab).
       * @memberOf niagara.schedule.ui.pages.editSpecialEvent
       * @private
       * @param {jQuery} targetElement DOM element containing the datebox
       * calendar
       * @param {baja.AbsTime} newTime the newly selected date
       */
      function doCalendarUpdate(targetElement, newTime) {
        return scheduleEditor.read().then(function (schedule) {
          dateRangeModified = false;
          return calendarUI.showEffectiveRangeCalendar(targetElement, schedule, newTime);
        });
      }

      debouncedCalendarUpdate = _.debounce(doCalendarUpdate, 600);

      /**
       * Saves the editors for the current special event's active date range
       * schedule and day schedule.
       *
       * Parallel async workflow to save both the day editor and the effective
       * date range editor on the currently edited special event.
       *
       * @memberOf niagara.schedule.ui.pages.editSpecialEvent
       * @private
       * @returns {Promise} when the day editor and schedule editors have
       * all been saved
       */
      function saveEditors() {
        return saveDayEditor().then(function () {
          return scheduleEditor.isModified() && saveScheduleEditor(scheduleEditor);
        });
      }

      /**
       * Returns a click handler to be bound to a back button. If the currently
       * edited day is in a modified state, will pop up a dialog asking the user
       * whether or not to save changes before allowing the backward navigation
       * to take place.
       *
       * @memberOf niagara.schedule.ui.pages.editSpecialEvent
       * @private
       */
      function checkForChangesFunction(targetPage) {

        return function () {
          if (dayEditor && dayEditor.isModified() || scheduleEditor.isModified()) {
            dialogs.confirmAbandonChanges({
              yes: function yes(cb) {
                saveEditors().then(cb.ok, cb.fail);
              },
              redirect: targetPage,
              viewName: editedSpecialEvent.getDisplayName()
            });
            return false;
          }
        };
      }

      /**
       * Enables/disables footer bar links depending on which tab is being
       * shown. Edit, Delete, Options, and Refresh will only be available
       * when viewing the Day Editor tab (edit/delete only if a schedule
       * block is selected), while the OK (save) button can be accessible from
       * any tab if the user has entered a change.
       *
       * Called whenever the user switches tabs or makes an edit to a field
       * editor.
       *
       * @memberOf niagara.schedule.ui.pages.editSpecialEvent
       * @private
       */
      function updateLinks() {
        if (selectedSectionIndex === 2) {
          if (specialEventsReadonly) {
            $('#editSpecialEvent-refresh').removeClass(DISABLED_CLASS);
          } else {
            $('#editSpecialEvent-actions, #editSpecialEvent-refresh').removeClass(DISABLED_CLASS);

            $('#editSpecialEvent-edit').removeClass(DISABLED_CLASS);
            $('#editSpecialEvent-delete').toggleClass(DISABLED_CLASS, !dayEditor.getSelectedBlock());
          }
        } else {
          $('#editSpecialEvent-footer').find('a[id!="editSpecialEvent-save"]').addClass(DISABLED_CLASS);
        }
      }

      /**
       * Redraws the content of the special event editor tabs. The
       * content div of the #editSpecialEvent page will be set to display
       * full screen, and the day editor drawn accordingly. If the date range
       * editor tab is taller than the day editor tab, then the content div
       * will be expanded to fit.
       *
       * Called upon pageshow and window resize.
       *
       * @memberOf niagara.schedule.ui.pages.editSpecialEvent
       * @private
       * @see niagara.schedule.ui.initializeUI
       */
      function relayout() {
        var dayWrapper = $('#editSpecialEvent-dayWrapper'),
            eventDayEditor = $('#editSpecialEvent-day');
        setContentHeight($('#editSpecialEvent'), dayWrapper);

        var height = Math.max(dayWrapper.height(), $('#editSpecialEvent-editors').children('.editor').height());

        $('#editSpecialEvent-content').height(height);

        if (dayEditor) {
          dayEditor.layout();
        }

        relayoutLabelsDiv(eventDayEditor.children('.scheduleLabels'), eventDayEditor.find('.blocksDisplay').height());
        updateLinks();
      }

      /**
       * Blanks out the day editor and redraws to fill available screen space.
       * Will be called when explicitly refreshing the day editor (by clicking
       * Refresh), when the page is shown, or when the window is resized.
       *
       * @memberOf niagara.schedule.ui.pages.editSpecialEvent
       * @private
       * @returns {Promise} will be resolved when the day editor has been
       * redrawn
       */
      function rebuild() {
        var dayDiv = $('#editSpecialEvent-day').empty();

        appendTimeLabels(dayDiv);

        return resetDayEditor({
          value: editedDaySchedule,
          label: baja.SlotPath.unescape(editedSpecialEvent.getName()),
          readonly: specialEventsReadonly,
          element: dayDiv
        }).then(function () {
          $('#editSpecialEvent-edit').addClass(DISABLED_CLASS);
          $('#editSpecialEvent-delete').addClass(DISABLED_CLASS);
          relayout();
        });
      }

      /**
       * Repaints the day editor to ensure that any user entered changes
       * (from clicking the create/delete/edit footer buttons) are reflected
       * on the screen.
       *
       * @memberOf niagara.schedule.ui.pages.editSpecialEvent
       * @private
       */
      function refresh() {
        if (dayEditor) {
          dayEditor.refreshWidgets();
        }
        setCreateMode($('#editSpecialEvent-footer'), true);
        relayout();
      }

      /**
       * Shows available actions to perform on the currently edited special
       * event's day schedule. Copy/paste and Clear Week functions are always
       * unavailable for a special event.
       * @memberOf niagara.schedule.ui.pages.editSpecialEvent
       * @private
       */
      function loadBlockActions() {
        var selectedBlock = dayEditor.getSelectedBlock(),
            hasBlocks = !!dayEditor.scheduleBlocks.length,
            hasSelected = !!selectedBlock;

        showBlockActions({
          'delete': hasSelected,
          'paste_day': false,
          'all_day': true,
          'apply_weekdays': false,
          'copy_day': false,
          'clear_day': hasBlocks,
          'clear_week': false
        });
      }

      /**
       * Performs the scrolling between calendar/date range/day editor tabs.
       * @memberOf niagara.schedule.ui.pages.editSpecialEvent
       * @private
       * @param {Number} index the tab index. 0 = Calendar Preview, 1 =
       * Active Dates, 2 = Day Editor.
       */
      function scrollSections(index) {
        var contentDiv = $('#editSpecialEvent-content'),
            calendarsDiv,
            absTime,
            contentWidth = $(document.documentElement).outerWidth();

        //viewing calendar
        if (index === 0 && dateRangeModified) {
          calendarsDiv = $('#editSpecialEvent-calendars');
          absTime = getAbsTime(calendarsDiv);
          doCalendarUpdate(calendarsDiv, absTime).catch(baja.error);
        }

        contentDiv.children().each(function () {
          var $this = $(this),
              elementIndex = $this.index();
          $this.animate({
            left: contentWidth * (elementIndex - index),
            right: contentWidth * (index - elementIndex)
          }, 'fast');
        });

        selectedSectionIndex = index;

        updateLinks();
      }

      /**
       * @memberOf niagara.schedule.ui.pages.editSpecialEvent
       */
      function pagecreate(obj) {
        var backButton = obj.page.children(JQM_HEADER_SELECTOR).find('a.profileHeaderBack');
        backButton.on('click', checkForChangesFunction('#specialEvents'));
      }

      /**
       * Arm handlers for the save button, the `DayEditor`, and for updating
       * the calendar preview whenever the schedule field editor is changed by
       * the user or the calendar's display month is changed.
       *
       * @memberOf niagara.schedule.ui.pages.editSpecialEvent
       */
      function pagebeforecreate() {
        if (!scheduleComponent.getType().is(WEEKLY_SCHEDULE_TYPE)) {
          $('#editSpecialEvent-dayEditorLink').parent().remove();
          $('#editSpecialEvent-edit').parent().remove();
          $('#editSpecialEvent-delete').parent().remove();
          $('#editSpecialEvent-actions').parent().remove();
          $('#editSpecialEvent-refresh').parent().remove();
        }

        $('#editSpecialEvent-day').on('selectionchange', function () {
          var selected = !!dayEditor.getSelectedBlock();
          setCreateMode($('#editSpecialEvent-footer'), !selected);
        }).on('dblclick', function () {
          editBlock(dayEditor.getSelectedBlock()).catch(logError);
        });

        $('#editSpecialEvent-navbar').on('click', 'a', function () {

          switch ($(this).attr('id')) {
            case 'editSpecialEvent-calendarLink':
              scrollSections(0);
              break;
            case 'editSpecialEvent-activeDatesLink':
              scrollSections(1);
              break;
            case 'editSpecialEvent-dayEditorLink':
              scrollSections(2);
              break;
          }
        });

        var specialEventFooter = $('#editSpecialEvent-footer');

        preventNavbarHighlight(specialEventFooter);

        specialEventFooter.on('click', 'a', function () {
          var $this = $(this);
          switch ($this.attr('id')) {
            case 'editSpecialEvent-save':
              saveEditors().then(function () {
                history.back();
              }, dialogs.error);
              break;
            case 'editSpecialEvent-refresh':
              if (dayEditor.isModified()) {
                dialogs.confirmAbandonChanges({
                  yes: saveDayEditor,
                  callbacks: function callbacks() {
                    var calendarDiv = $('#editSpecialEvent-calendars'),
                        absTime = getAbsTime(calendarDiv);
                    rebuild().catch(logError);
                    doCalendarUpdate(calendarDiv, absTime).catch(logError);
                  },
                  viewName: mobileLex.get('schedule.dayEditor')
                });
              } else {
                rebuild().catch(logError);
              }

              break;
            case 'editSpecialEvent-actions':
              loadBlockActions();
              break;
            case 'editSpecialEvent-edit':
              createOrEdit().then(refresh);
              break;
            case 'editSpecialEvent-delete':
              dayEditor.removeBlock(dayEditor.getSelectedBlock());
              refresh();
              break;
          }
        });

        $('#editSpecialEvent-content').on(events.MODIFY_EVENT, function () {
          dateRangeModified = true;
        });

        $('#editSpecialEvent-calendars').on('datebox', 'input:jqmData(role="datebox")', function (e, passed) {

          if (passed.method === 'offset') {
            //+ or - button clicked
            var calendarDiv = $('#editSpecialEvent-calendars'),
                newTime = baja.AbsTime.make({ jsDate: passed.newDate });

            Promise.resolve(debouncedCalendarUpdate(calendarDiv, newTime)).catch(logError);
          }
        });
      }

      /**
       * Loads, and instigates page change to, the editSpecialEvent page.
       * Instantiates a new `niagara.fieldEditors.schedule.DayEditor` for the
       * edited schedule's `day` property.
       *
       * @memberOf niagara.schedule.ui.pages.editSpecialEvent
       * @param {baja.Component} specialEvent the `schedule:DailySchedule`
       * representing the edited special event
       */
      function load(specialEvent) {
        editedSpecialEvent = specialEvent;
        editedDaySchedule = editedSpecialEvent.get('day');

        $('#editSpecialEvent-eventName').text(baja.SlotPath.unescape(specialEvent.getName()));

        rebuild().then(function () {
          jqm.changePage('#editSpecialEvent');
          return calendarUI.loadScheduleEditor($('#editSpecialEvent'), editedSpecialEvent.get('days') || editedSpecialEvent, specialEventsReadonly);
        }).then(function (editor) {
          var calendarDiv = $('#editSpecialEvent-calendars'),
              absTime = getAbsTime(calendarDiv);
          scheduleEditor = editor;
          return doCalendarUpdate(calendarDiv, absTime);
        }).catch(function (err) {
          logError("editSpecialEvent.pagebeforeshow: failed to load " + "schedule editor: " + err);
        });
      }

      /**
       * Loads the calendar display preview and special event schedule
       * field editor.
       * @memberOf niagara.schedule.ui.pages.editSpecialEvent
       */
      function pagebeforeshow() {
        if (!editedSpecialEvent) {
          jqm.changePage('#specialEvents');
          return;
        }

        refresh();
        setCreateMode($('#editSpecialEvent-footer'), true);
      }

      /**
       * Loads the `niagara.fieldEditors.schedule.DayEditor` for the edited
       * special event's `day` property. (Done in `pagelayout` because to draw
       * the editor properly we must know the div's height - which can't be
       * obtained in `pagebeforeshow`.)
       *
       * @memberOf niagara.schedule.ui.pages.editSpecialEvent
       */
      function pagelayout() {
        if (!dayEditor) {
          return;
        }

        document.title = $('#editSpecialEvent-header').children('h1').text();

        relayout();
        scrollSections(selectedSectionIndex || 0);
      }

      /**
       * If the day editor is modified, prompts for changes before navigating
       * away.
       * @memberOf niagara.schedule.ui.pages.editSpecialEvent
       */
      function pagebeforechange(obj) {
        if (dayEditor && dayEditor.isModified() || scheduleEditor.isModified()) {
          dialogs.confirmAbandonChanges({
            yes: function yes(cb) {
              saveEditors().then(cb.ok, cb.fail);
            },
            no: function no(cb) {
              resetDayEditor(editedDaySchedule).then(cb.ok, cb.fail);
            },
            redirect: obj.nextPage,
            viewName: scheduleLex.get('summary.specialEvent')
          });
          obj.event.preventDefault();
        }
      }

      $(window).resize(baja.throttle(function () {
        scrollSections(selectedSectionIndex || 0);
      }));

      /**
       * @namespace
       * @private
       * @name niagara.schedule.ui.pages.editSpecialEvent
       */
      return {
        pagebeforechange: pagebeforechange,
        pagebeforecreate: pagebeforecreate,
        pagebeforeshow: pagebeforeshow,
        pagecreate: pagecreate,
        pagelayout: pagelayout,
        load: load,
        rebuild: rebuild,
        relayout: relayout
      };
    }());

    pages.register('summary', function summary() {
      var debouncedCalendarUpdate;

      /**
       * Parses the text value of a datebox `<input>` into a `baja.AbsTime`.
       * @memberOf niagara.schedule.ui.pages.summary
       * @private
       * @param {String} str the datebox text input value
       * @returns {baja.AbsTime} the parsed `AbsTime` value
       */
      function dateboxToAbsTime(str) {
        var year = str.substring(0, 4),
            month = str.substring(5, 7),
            day = str.substring(8, 10),
            date = baja.Date.make({
          year: Number(year),
          month: Number(month) - 1,
          day: Number(day)
        }),

        //bump the time forward in hopes of not getting results from yesterday
        //TODO: TIME ZONES AHOY
        time = baja.Time.make(MILLIS_IN_HOUR * 9);
        return baja.AbsTime.make({ date: date, time: time });
      }

      /**
       * @memberOf niagara.schedule.ui.pages.summary
       * @private
       */
      function doCalendarUpdate(targetElement, newTime) {
        return calendarUI.showEffectiveRangeCalendar(targetElement, currentSchedule.value(), newTime).catch(logError);
      }

      debouncedCalendarUpdate = _.debounce(doCalendarUpdate, 600);

      /**
       * Arms click handlers on the effective range calendar to show a summary
       * for the clicked day, and to update the effective range display when
       * a new month is selected.
       * @memberOf niagara.schedule.ui.pages.summary
       */
      function pagebeforecreate(obj) {
        appendNavbar(obj.page);

        $('#summary-calendars').on('datebox', 'input:jqmData(role="datebox")', function (e, passed) {
          if (passed.method === 'offset') {
            //+ or - button clicked
            var calendarDiv = $('#summary-calendars'),
                newTime = baja.AbsTime.make({ jsDate: passed.newDate });
            debouncedCalendarUpdate(calendarDiv, newTime);
          } else if (passed.method === 'set' && scheduleComponent.getType().is(WEEKLY_SCHEDULE_TYPE)) {
            calendarUI.showDaySummary($('#summary-summaryList'), currentSchedule.value(), dateboxToAbsTime(passed.value)).catch(logError);
          }
        });
      }

      /**
       * Shows the effective range calendar and day summary.
       * @memberOf niagara.schedule.ui.pages.summary
       */
      function pagebeforeshow() {
        calendarUI.showEffectiveRangeCalendar($('#summary-calendars'), currentSchedule.value(), baja.AbsTime.now()).catch(logError);

        $('#summary-summary').addClass(BTN_ACTIVE_CLASS);
        $('#summary-title').text(scheduleComponent.getDisplayName());
      }

      /**
       * @namespace
       * @name niagara.schedule.ui.pages.summary
       * @private
       */
      return {
        pagebeforecreate: pagebeforecreate,
        pagebeforeshow: pagebeforeshow,
        getCommands: getDefaultCommands
      };
    }());
  }

  /**
   * Prevent blowups if we hit refresh on a sub-page and thus no schedule
   * currently loaded. `editDay` redirects to `main`, `addSpecialEvent` and
   * `editSpecialEvent` redirect to `specialEvents`, and all other pages just
   * preload the current schedule before displaying.
   *
   * This function is bound to `pagebeforechange` in `initializeUI`.
   *
   * @memberOf niagara.schedule.ui
   * @private
   * @param {JQuery.Event} event the `pagebeforechange` event
   * @param {Object} data the JQM data passed along with the event
   */
  function preloadOrRedirect(event, data) {
    if (!currentSchedule) {
      var id;

      if (typeof data.toPage === 'string') {
        id = jqm.path.parseUrl(data.toPage).hash.replace('#', '') || 'main';
      } else {
        id = data.toPage.attr('id');
      }

      event.preventDefault();
      event.stopImmediatePropagation();

      reretrieveSchedule().then(function () {
        data.options.changeHash = true;
        if (id === 'editDay') {
          jqm.changePage('#main', data.options);
        } else if (id === 'addSpecialEvent' || id === 'editSpecialEvent') {
          jqm.changePage('#specialEvents', data.options);
        } else {
          data.options.changeHash = false;
          jqm.changePage('#' + id, data.options);
        }
      });
    }
  }

  /**
   * Registers jQuery custom selectors and JQM page event listeners. Runs
   * automatically when `schedule.ui.js` runs.
   */
  function initializeUI() {
    registerPages();

    $(window).on('resize', baja.throttle(function () {
      getActivePage().trigger('pagelayout');
    }, 500));

    window.onbeforeunload = function () {
      if (currentSchedule.isModified()) {
        return mobileLex.get({
          key: 'message.viewModified',
          args: [scheduleLex.get('scheduler.weeklySchedule')]
        });
      }
    };

    // When the DOM is ready, register the command handler directly on the commands button...
    $(function () {
      $(".commandsButton").click(commands.showCommandsHandler);
    });

    onPageContainerCreate(function () {
      prependEventHandler(getPageContainer(), 'pagebeforechange', preloadOrRedirect);
    });
  }

  return initializeUI;
});
