/**
 * @file Functions relating to displaying and updating calendar
 * widgets within the mobile scheduler app.
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/*global niagara */

define(['baja!baja:Month', 'baja!', 'jquery', 'jquerymobile', 'jqmDatebox', 'underscore', 'mobile/util/mobile/dialogs', 'mobile/util/mobile/mobile', 'mobile/util/time', 'mobile/fieldeditors/fieldeditors', 'bajaux/events', 'mobile/schedule/util.schedule.datebox'], function (unusedTypes, baja, $, jqm, jqmDatebox, _, dialogs, mobileUtil, timeUtil, fe, events, dateboxUtil) {
  "use strict";

  var SPINNER_DELAY = 2000;

  /**
   * @private
   * @exports mobile/schedule/schedule.ui.calendar
   */
  var exports = {};

  /**
   * Creates a Bajascript `BVector`, holding a `BWeeklySchedule` and a
   * `BAbsTime`. This vector will be sent to
   * `BScheduleServiceSideCallHandler#getHighlightedDates`.
   * 
   * @param {baja.Component} schedule a `schedule:WeeklySchedule` to send
   * @param {baja.AbsTime} absTime the time to send
   */
  function makeVector(schedule, absTime) {
    return baja.$('baja:Vector', { time: absTime, schedule: schedule });
  }

  function appendRefBase(schedule) {
    var obj = {
      slot: 'refBase',
      value: baja.Ord.make(niagara.view.ord),
      flags: baja.Flags.HIDDEN | baja.Flags.TRANSIENT
    };
    //BScheduleReference ordinarily has a mounted component to work with -
    //since we are working with an unmounted snapshot, let's give it the ORD of
    //the currently viewed schedule.
    //i freely grant you there is a little bit of voodoo at work here - this may
    //or may not be the "right" way to do it - but based on some amateurish
    //reverse engineering of uncommented code, it works. ag the schedule-rt
    //codebase for 'refBase' for usage.
    if (schedule.has('refBase')) {
      schedule.set(obj);
    } else {
      schedule.add(obj);
    }
  }

  /**
   * Converts an AbsTime to a datebox-friendly format ready to load into
   * a calendar.
   * @param {baja.AbsTime} absTime
   * @returns {String}
   */
  function toDateboxFormat(absTime) {
    var date = absTime.getDate(),
        year = date.getYear(),
        month = date.getMonth().getOrdinal() + 1;
    return year + '-' + (month < 10 ? '0' : '') + month + '-01';
  }

  /**
   * Calls a server side handler method on `ScheduleServerSideCallHandler`.
   * @private
   * @param {baja.Complex} schedule a `schedule:AbstractSchedule`
   * @param {baja.AbsTime} absTime an AbsTime
   * @param {String} methodName
   * @returns {Promise} promise to be resolved with the result of the
   * server side call
   */
  function doSSC(schedule, absTime, methodName) {
    schedule = schedule.newCopy();
    appendRefBase(schedule);

    var vec = makeVector(schedule, absTime);

    return baja.Ord.make(niagara.view.ord).get().then(function (component) {
      return component.serverSideCall({
        typeSpec: 'mobile:ScheduleServerSideCallHandler',
        methodName: methodName,
        value: vec
      });
    }).then(function (val) {
      return JSON.parse(val);
    });
  }

  /**
   * Given a BAbstractSchedule and an AbsTime, will make a server side call
   * to `BScheduleServerSideCallHandler#getHighlightedDates` and return its
   * output (an array of dates that are in the same month as the AbsTime and
   * are active in the given schedule).
   * 
   * @param {baja.Component} schedule a BAbstractSchedule
   * @param {baja.AbsTime} absTime get dates for this month
   * @returns {Promise} promise to be resolved with an array of highlighted
   * dates
   */
  function getActiveDates(schedule, absTime) {
    return doSSC(schedule, absTime, 'getHighlightedDates');
  }

  /**
   * Get a daily summary of events for the given date, using a server side
   * call to `BScheduleServerSideCallHandler#getDaySummary`, for display on the
   * summary tab.
   * 
   * @param {baja.Component} schedule a BWeeklySchedule
   * @param {baja.AbsTime} absTime the day to show a summary for
   * @returns {Promise} promise to be resolved with an array of day summaries
   */
  function getDaySummary(schedule, absTime) {
    return doSSC(schedule, absTime, 'getDaySummary');
  }

  /**
   * Displays/updates a JQM Datebox calendar in the target div. It will 
   * retrieve a list of the effective dates and highlight those dates in the
   * datebox calendar.
   * 
   * @param {jQuery} targetElement the div in which to show the calendar
   * @param {baja.Component} schedule a `schedule:AbstractSchedule` to check
   * for effective dates
   * @param {baja.AbsTime} absTime the target date to show a calendar for (will
   * be truncated to month)
   * @returns {Promise} promise to be resolved once target element has a
   * range calendar rendered
   */
  exports.showEffectiveRangeCalendar = function (targetElement, schedule, absTime) {
    var ticket = mobileUtil.spinnerTicket(SPINNER_DELAY);

    return getActiveDates(schedule, absTime).then(function (activeDates) {
      var input = dateboxUtil.getDateboxInput(targetElement);

      //highlight active dates - will redraw in trigger('datebox')
      input.datebox({ highDates: activeDates });
      input.datebox('setTheDate', toDateboxFormat(absTime));
      input.trigger('datebox', { method: 'close' });
      input.datebox('getTheDate');

      ticket.hide();
    });
  };

  function makeDaySummaryList(events) {
    var ul = $('<ul class="daySummary" data-role="listview"/>');
    baja.iterate(events, function (eventObj) {
      var li = $('<li/>'),
          title = $('<label class="daySummaryTitle"/>').appendTo(li),
          subtitle = $('<label class="daySummarySubtitle"/>').appendTo(li);

      title.text(eventObj.timestamp);
      subtitle.text(eventObj.output + ' (' + eventObj.source + ')');
      li.appendTo(ul);
    });
    return ul;
  }

  /**
   * 
   * @param targetElement
   * @param schedule
   * @param absTime
   * @returns {Promise}
   */
  exports.showDaySummary = function (targetElement, schedule, absTime) {
    var ticket = mobileUtil.spinnerTicket(SPINNER_DELAY);

    return getDaySummary(schedule, absTime).then(function (daySummary) {
      var ul = makeDaySummaryList(daySummary);
      targetElement.html(ul);
      ul.listview();

      //scroll down to show the summary output
      $.mobile.silentScroll(targetElement.offset().top);
      ticket.hide();
    });
  };

  /**
   * Loads field editors for the effective date range of a schedule - will
   * show day, month, and year dropdowns for both start and end effective dates.
   * 
   * @param {JQuery} page the page in which to show the date range field
   * editor
   * @param {baja.Component} schedule the `schedule:WeeklySchedule` whose
   * `'effective'` property we will edit
   * @param {Boolean} readonly
   * @returns {module:mobile/fieldeditors/BaseFieldEditor} a field editor for a
   * `schedule:DateRangeSchedule`
   */
  exports.loadScheduleEditor = function (page, schedule, readonly) {

    var contentDiv = page.children(':jqmData(role="content")'),
        editorDiv = contentDiv.find('div:jqmData(role="calendareditor")');

    return fe.makeFor({
      value: schedule,
      element: editorDiv.empty(),
      readonly: readonly
    });
  };

  /**
   * Binds an editor's `modified.fieldeditor` event to a calendar div, so
   * that whenever the editor changes, the calendar display will perform the
   * server side call to display the new effective date range.
   * 
   * @param {module:mobile/fieldeditors/BaseFieldEditor} editor the editor to bind to
   * a calendar
   * @param {jQuery} calendarDiv the div where the datebox calendar lives
   */
  exports.bindEditorToCalendar = function (editor, calendarDiv) {
    var editorDiv = editor.jq();

    editorDiv.off(events.MODIFY_EVENT);
    editorDiv.on(events.MODIFY_EVENT, _.debounce(function () {
      var input = dateboxUtil.getDateboxInput(calendarDiv),
          theDate = input.datebox('getTheDate');

      //TODO: temporary hack until time zones are fixed
      theDate = new Date(theDate.getTime() + timeUtil.MILLIS_IN_DAY);

      editor.read().then(function (schedule) {
        return exports.showEffectiveRangeCalendar(calendarDiv, schedule, baja.AbsTime.make({ jsDate: theDate }));
      }).catch(dialogs.error);
    }, 500));
  };

  return exports;
});
