/**
 * @file A view for the property sheet app containing a property
 * sheet list view plus extra buttons.
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

define(['lex!mobile', 'baja!', 'jquery', 'jquerymobile', 'Promise', 'mobile/fieldeditors/fieldeditors.dialogs', 'mobile/util/slot', 'mobile/util/mobile/mobile', 'mobile/util/mobile/commands', 'mobile/util/mobile/dialogs', 'mobile/util/mobile/views/PageView', 'mobile/propsheet/PropertySheetListView', 'bajaux/events'], function (lexs, baja, $, jqm, Promise, feDialogs, slotUtil, mobileUtil, commands, dialogs, PageView, PropertySheetListView, events) {

  "use strict";

  //local vars

  var footerBarHtml,
      navbarHtml,
      mobileLex = lexs[0],
      escapeHtml = mobileUtil.escapeHtml,
      LOAD_EVENT = events.LOAD_EVENT,
      MODIFY_EVENT = events.MODIFY_EVENT,
      UNMODIFY_EVENT = events.UNMODIFY_EVENT,


  //exports
  PropertySheet;

  footerBarHtml = '<div data-role="footer" data-position="fixed" data-theme="b" data-tap-toggle="false">\n' + '<div data-role="navbar"><ul>\n' + '<li><a data-icon="check" class="saveLink">{save}</a></li>\n' + '<li><a data-icon="refresh" class="refreshLink">{refresh}</a></li>\n' + '<li><a data-icon="gear" class="actionsLink">{actions}</a></li>\n' + '</ul></div>\n' + '</div>';

  navbarHtml = '<div data-role="navbar">' + '<ul>' + '<li>' + '<a class="componentLink">{title}</a>' + '</li>' + '</ul>' + '</div>';

  function validationError(err) {
    dialogs.ok({
      content: err,
      title: mobileLex.get({
        key: 'propsheet.validationError',
        def: 'Validation Error'
      })
    });
  }

  /**
   * A property sheet view consisting of a JQM page whose content is formed
   * by a `PropertySheetListView`.
   * 
   * @class
   * @extends niagara.util.mobile.PageView
   * @memberOf niagara.propSheet
   */
  PropertySheet = baja.subclass(function PropertySheet() {
    baja.callSuper(PropertySheet, this, arguments);
  }, PageView);

  PropertySheet.prototype.instantiateContentView = function () {
    return new PropertySheetListView();
  };

  /**
   * Adds a `viewloadvalue` listener to this view's div. Once the property
   * sheet's value has been fully loaded, it will show/hide the commands button
   * depending on whether the property sheet is showing a Component or just a
   * Struct. It will also update the footer bar to disable the Save button, and
   * either enable or disable the Actions button depending on whether the
   * component has any actions to be fired.
   * 
   * @name niagara.propSheet.PropertySheet#doInitialize
   * @function
   * @see niagara.util.mobile.PageView#doInitialize
   */
  PropertySheet.prototype.doInitialize = function (element) {
    var that = this;

    element.on([LOAD_EVENT, MODIFY_EVENT, UNMODIFY_EVENT].join(' '), function () {
      that.updateBars();
    });

    element.on('click', 'a.saveLink', function () {
      var $this = $(this);

      that.contentView.$saveCommand.invoke().then(function () {
        $this.addClass('ui-disabled');
      }).catch(validationError);
    });

    element.on('click', 'a.refreshLink', function () {
      that.contentView.$refreshCommand.invoke();
    });

    element.on('click', 'a.actionsLink', function () {
      feDialogs.showAvailableActions(that.value());
    });

    return PageView.prototype.doInitialize.call(this, element);
  };

  /**
   * Appends a footer bar to the JQM page for this view, and a view button to
   * the page's header. The footer bar contains link buttons for Save, Refresh,
   * and Actions.
   * 
   * @name niagara.propSheet.PropertySheet#createPage
   * @function
   * @see niagara.util.mobile.PageView#createPage
   */
  PropertySheet.prototype.createPage = function () {
    var page = PageView.prototype.createPage.call(this),
        headerDiv = page.children(':jqmData(role=header)'),
        navbar = $(navbarHtml.patternReplace({
      title: mobileLex.getSafe('loading')
    })),
        footer = $(footerBarHtml.patternReplace({
      save: escapeHtml(mobileLex.get('save')),
      refresh: escapeHtml(mobileLex.get('refresh')),
      actions: escapeHtml(mobileLex.get('actions'))
    }));

    page.append(footer);
    mobileUtil.preventNavbarHighlight(footer);

    headerDiv.append(commands.getCommandsButton().removeClass(this.$ignoreProfileClasses));
    headerDiv.append(navbar);

    return page;
  };

  /**
   * When removing a property sheet from the DOM structure, also examine every
   * datebox input (`$(this).jqmData('datebox').pickPage`) and remove it from
   * the JQM page container.
   * 
   * @name niagara.propSheet.PropertySheet#stop
   * @function
   * @see niagara.util.mobile.mixins.subscribableMixin.stop
   */
  PropertySheet.prototype.doDestroy = function () {
    this.getContentDiv().find('input:jqmData(role=datebox)').each(function () {
      var datebox = $(this).jqmData('datebox'),
          pickPage = datebox && datebox.pickPage;
      if (pickPage) {
        pickPage.remove();
      }
    });
  };

  /**
   * Finds actions on this property sheet's component that are fireable (not
   * hidden).
   * 
   * @name niagara.propSheet.PropertySheet#hasFireableActions
   * @function
   * 
   * @returns {Boolean} true if this property sheet's component has an action
   * that can be fired
   */
  PropertySheet.prototype.hasFireableActions = function () {
    return !!this.value().getSlots(function filter(slot) {
      return slotUtil.isFireable(slot);
    }).next();
  };

  /**
   * Enables/disables the save/action buttons on the footer, depending on
   * whether the given sheet has unsaved changes / fireable actions. The 
   * enable/disable is performed simply by adding the `ui-disabled`
   * class to the `<a>` links contained in the footer buttons. JQM
   * automatically checks for the presence of this class and prevents click
   * handlers from executing if found.
   * 
   * This function also sets the menu button in the header bar to turn red
   * depending on whether the sheet has unsaved changes.
   * 
   * @name niagara.propSheet.PropertySheet#updateBars
   * @function
   * @private
   * 
   * @param {niagara.propSheet.PropertySheet} sheet the property sheet
   * whose state the footer bar should reflect
   */
  PropertySheet.prototype.updateBars = function updateBars() {
    var that = this,
        value = that.value(),
        header = that.getHeaderDiv(),
        footer = that.getFooterDiv(),
        footerLinks = footer.find('a'),
        actionsLink = footerLinks.filter('.actionsLink'),
        commandsButton = header.find('.commandsButton'),
        saveLink = footerLinks.filter('.saveLink');

    if (value.getType().isComponent()) {
      commandsButton.show();
    } else {
      commandsButton.hide();
    }

    actionsLink.toggleClass('ui-disabled', !this.hasFireableActions());
    saveLink.toggleClass('ui-disabled', !this.isModified());
    commandsButton.toggleClass('red', this.isModified());
  };

  return PropertySheet;
});
