/**
 * @copyright 2017 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * @private
 * @module mobile/fieldeditors/mobile/UnitEditor
 */
define(['baja!', 'jquery', 'underscore', 'mobile/fieldeditors/MobileFieldEditor'], function (baja, $, _, MobileFieldEditor) {

  'use strict';

  var dimensionRegex = /\(\S*\)/;

  var getUnitDb = _.once(retrieveUnitDb);

  /**
   * @private
   * @class
   * @alias module:mobile/fieldeditors/mobile/UnitEditor
   * @extends module:mobile/fieldeditors/MobileFieldEditor
   */
  var UnitEditor = function UnitEditor() {
    MobileFieldEditor.apply(this, arguments);
  };
  UnitEditor.prototype = Object.create(MobileFieldEditor.prototype);
  UnitEditor.prototype.constructor = UnitEditor;

  /**
   * @param {JQuery} dom
   */
  UnitEditor.prototype.doInitialize = function (dom) {
    var quantitySelect = $('<select data-role="selectmenu" data-theme="a" name="quantity" />').append('<option>loading</option>'),
        unitSelect = $('<select data-role="selectmenu" data-theme="a" name="units" />').append('<option>loading</option>');

    dom.append(quantitySelect);
    dom.append(unitSelect);

    return getUnitDb().then(function (unitDb) {
      //populate quantity select...
      quantitySelect.html(_.map(unitDb.getQuantities(), function (quantity) {
        var quantityName = quantity.getName();
        return $('<option/>').attr('value', quantityName || '').text(quantityName);
      }));

      //upon selecting a new quantity, grab the units from the quantity db
      //and populate the units select
      quantitySelect.on('change', function () {
        var quantityName = $(this).find('option:selected').text(),
            units = unitDb.getUnits(quantityName);

        unitSelect.html(_.map(units, function (unit) {
          return $('<option/>').val(unit.getUnitName()).text(unit.getUnitName() + ' (' + unit.getSymbol() + ')').jqmData('unit', unit);
        }));

        unitSelect.selectmenu('refresh').trigger('updatelayout');
      });

      quantitySelect.appendTo(dom);
      unitSelect.appendTo(dom);
    });
  };

  UnitEditor.prototype.doLoad = function (unit) {
    var dom = this.jq(),
        quantitySelect = dom.find('select[name=quantity]'),
        unitSelect = dom.find('select[name=units]');

    return getUnitDb().then(function (db) {
      var quantity = db.getQuantityFromUnit(unit);
      if (quantity) {
        quantitySelect.val(quantity.getName()).trigger('change');
        unitSelect.val(unit.getUnitName()).trigger('change');
      }
    });
  };

  UnitEditor.prototype.doRead = function () {
    var unitSelect = this.jq().find('select[name=units]'),
        option = unitSelect.find('option:selected');
    return option.jqmData('unit');
  };

  function retrieveUnitDb() {
    return baja.UnitDatabase.get().then(function (db) {
      var quantities = db.getQuantities();

      function getQuantities() {
        return quantities;
      }

      function getQuantitiesByDimensionName(dimensionName) {
        return _.filter(quantities, function (quantity) {
          var q = quantity.getName(),
              dimensionMatch = q.match(dimensionRegex);
          return dimensionMatch && dimensionMatch[0];
        });
      }

      function getQuantityByName(quantityName) {
        return _.find(quantities, function (quantity) {
          return quantity.getName() === quantityName;
        });
      }

      function getQuantityFromUnit(unit) {
        var quantities = getQuantitiesByDimensionName(String(unit.getDimension()));

        return _.find(quantities, function (q) {
          return _.find(q.getUnits(), function (u) {
            return unit.equivalent(u);
          });
        });
      }

      function getUnits(quantityName) {
        var q = getQuantityByName(quantityName);
        return q && q.getUnits();
      }

      function getUnitFromSymbolAndDimension(unitSymbol, unitDimension) {
        var quantities = getQuantitiesByDimensionName(unitDimension);

        function bySymbol(unit) {
          return unit.getSymbol() === unitSymbol;
        }

        // there can be multiple quantities with the same dimension, so we
        // have to search each quantity to see if it contains the actual
        // unit.
        for (var i = 0; i < quantities.length; i++) {
          var unit = _.find(quantities[i].getUnits(), bySymbol);
          if (unit) {
            return unit;
          }
        }
      }

      return {
        getQuantities: getQuantities,
        getQuantityFromUnit: getQuantityFromUnit,
        getUnits: getUnits,
        getUnitFromSymbolAndDimension: getUnitFromSymbolAndDimension
      };
    });
  }

  return UnitEditor;
});
