/**
 * @file Dialogs relating to field editors, including action invocation.
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

define(['baja!', 'lex!mobile', 'jquery', 'Promise', 'mobile/fieldeditors/fieldeditors', 'mobile/fieldeditors/fieldeditors.composite', 'bajaux/commands/Command', 'mobile/util/mobile/dialogs', 'mobile/util/mobile/commands', 'mobile/util/slot'], function (baja, lexs, $, Promise, fe, composite, Command, dialogs, commands, slotUtil) {

  "use strict";

  var AllSlotsEditor = composite.allSlots(),
      mobileLex = lexs[0];

  /**
   * Displays a single field editor in a popup dialog. The field editor will
   * be shown with OK and Cancel buttons. Cancel will simply cause the dialog
   * to close, and OK will save the field editor and resolve the resulting
   * value.
   * 
   * @memberOf niagara.util.mobile.dialogs
   * 
   * @param {Object} params an object literal containing dialog parameters
   * 
   * @param {String} [params.title] the dialog title
   * 
   * @param {baja.Value} params.value the value to be edited using a field
   * editor
   * 
   * @param {Object} [params.editorParams] any extra field editor parameters
   * to be passed into `niagara.fieldEditors.getFieldEditor` when constructing
   * the field editor
   * 
   * @returns {Promise} promise to be resolved with the value the user entered
   * into the field editor, or with `null` if the user clicked cancel
   * 
   * @example
   * require('mobile/fieldeditors/fieldeditors.dialogs').fieldEditor({
   *   title: 'Please select a stooge',
   *   value: baja.DynamicEnum.make({
   *     range: baja.EnumRange.make({
   *       ordinals: [0, 1, 2],
   *       tags: ['Moe', 'Larry', 'Curly']
   *     })
   *   })
   * })
   *   .then(function (selectedValue) {
   *     if (selectedValue === null) {
   *       alert('you clicked cancel');
   *     } else {
   *       alert('you selected "' + selectedValue.getTag() + '".');
   *     }
   *   });
    */
  function fieldEditor(params) {
    var value = params.value,
        editor;

    function showEditor(Ctor) {
      return new Promise(function (resolve, reject) {
        dialogs.okCancel({
          title: params.title || String(value.getType()),

          content: function content(contentDiv) {
            params.element = contentDiv;
            return fe.makeFor(Ctor, params).then(function (e) {
              editor = e;
            });
          },

          ok: function ok(cb) {
            var batch = new baja.comm.Batch();

            editor.save({ batch: batch }).then(function () {
              return editor.read();
            }).then(function (value) {
              resolve(value);
              cb.ok();
            }, function (err) {
              reject(err);
              cb.fail(err);
            });

            batch.commit();
          },

          cancel: function cancel(cb) {
            resolve(null);
            cb.ok();
          }
        });
      });
    }

    return fe.isRegistered(value.getType()).then(function (result) {
      return showEditor(result ? null : AllSlotsEditor);
    });
  }

  /**
   * Dialog to fire an action on a component. Note that because this dialog
   * type is asynchronous by nature, it will therefore _not_ return a handler
   * object the same way `yesNoCancel` etc. will.
   * 
   * If the specified action requires an argument, `action()` will pop up a
   * `fieldEditor` for that argument before firing the action. If no argument is
   * required, `action()` will simply fire the action.
   * 
   * In either case, if the action slot has the `CONFIRM_REQUIRED` flag set, a
   * yes/no dialog will be shown for confirmation before firing the action.
   * 
   * @memberOf niagara.util.mobile.dialogs
   * @param {Object} params an object containing parameters for the dialog
   * @param {baja.Component} params.component the component on which to fire
   * an action
   * @param {String|baja.Slot} params.slot the slot for the action to fire
   * @param {baja.Value} [params.parameter] the parameter to give to the action -
   * if omitted, will retrieve the default action parameter asynchronously from
   * the server - or will just not use one if not needed
   * @param {Object} [params.editorParams] parameters to be passed to the
   * field editor for the action parameter, if one is used
   * @returns {Promise} promise to be resolved with the returned value from the
   * action invocation, or with `null` if the user canceled invocation
   */
  function action(params) {
    params = baja.objectify(params);

    var actionArg = null,
        slot = params.slot,
        actionSlot,
        arg = baja.def(params.parameter, ""),
        comp = params.component,
        paramType,
        showDialog;

    baja.strictArg(comp, baja.Component);

    // Make sure we have an Action
    actionSlot = comp.getSlot(slot);
    if (!actionSlot.isAction()) {
      throw new Error("slot '" + slot + "' is not an action");
    }

    paramType = actionSlot.getParamType();

    if (!arg.equals("") && paramType !== null) {
      if (arg.getType().is(paramType)) {
        actionArg = arg;
      } else if (typeof arg === "string" && paramType.isSimple()) {
        actionArg = paramType.getInstance().decodeFromString(arg);
      }
    }

    // Invoke the Action
    function invoke() {
      // Manually do a sync so we can a response ASAP.
      baja.clock.schedule(function () {
        baja.station.sync();
      }, 500);

      return comp.invoke({ slot: actionSlot, value: actionArg });
    }

    showDialog = actionArg === null && paramType !== null;

    // Check for dialog input and invoke the dialog
    function dialogInvoke() {

      // Dialog for Action?
      if (showDialog) {
        // Make network call to get the default argument for the Action invocation
        return comp.getActionParameterDefault({ slot: actionSlot }).then(function (actionDefArg) {
          actionArg = actionDefArg;

          // Use facets if not already provided
          if (params && !params.facets) {
            params.facets = slotUtil.getFacets(comp);
          }

          // Ask the user to enter a value for the Action
          return fieldEditor($.extend(params, {
            title: comp.getDisplayName(actionSlot),
            value: actionArg
          })).then(function (val) {
            if (val !== null) {
              actionArg = val;
              return invoke();
            } else {
              return null;
            }
          });
        });
      } else {
        return invoke();
      }
    }

    // Confirm required?
    if ((comp.getFlags(actionSlot) & baja.Flags.CONFIRM_REQUIRED) === baja.Flags.CONFIRM_REQUIRED) {
      return new Promise(function (resolve, reject) {
        dialogs.yesNo({
          content: mobileLex.get({
            key: "invoke.confirm",
            args: [comp.getDisplayName(actionSlot)]
          }),
          yes: function yes() {
            dialogInvoke().then(resolve, reject);
          },
          no: function no() {
            resolve(null);
          }
        });
      });
    } else {
      var prom = dialogInvoke();
      // If there are going to be no further dialogs then close the current one
      if (!showDialog) {
        dialogs.closeCurrent();
      }
      return prom;
    }
  }

  function showAvailableActions(component) {
    var cmds = [];

    component.getSlots(function filter(slot) {
      return slotUtil.isFireable(slot);
    }).each(function (slot) {
      cmds.push(new Command(component.getDisplayName(slot), function () {
        return action({
          component: component,
          slot: slot,
          facets: component.get('facets') || {}
        });
      }));
    });

    commands.showCommandsDialog(cmds, mobileLex.get('actions'));
  }

  return {
    action: action,
    fieldEditor: fieldEditor,
    showAvailableActions: showAvailableActions
  };
});
