/*
 * Copyright 2005 Tridium, Inc. All Rights Reserved.
 */

/* jshint latedef: nofunc, jquery: true */
/* globals hx: true, document: true, smartTable: false, save: false, alert: false, console: false */

/**
 * JavaScript support for Scheduler.
 * @private
 * @deprecated since Niagara 4.4. Will be removed in a future release.
 */
var schedule = new Schedule();

function Schedule() {
  'use strict';

  var DEBUG_MOUSE_MOVE = false;

  // Replace with keyword CONST once supported by all browsers
  var constant = {};

  /**
   * Text to display when a day event status is null.  Initialized to 'Null' and updated by a call
   * to {@link setNullStatus}.  Read in {@link createEventForValue} and in {@link valueChange} when
   * the null checkbox is checked.
   *
   * @type {string}
   */
  constant.NULL_STATUS_LEX = 'nullStatusLex';

  /**
   * Set in {@link initBajaType} when the bajaType is "baja:Boolean".  Will set
   * {@link IS_ENUM_VALUE} if true.  If this flag is set, the {@link RANGE} values are searched for
   * text or an ordinal in {@link getDisplayText} and {@link getValueEncoding}.
   *
   * @type {string}
   */
  constant.IS_BOOLEAN_VALUE = 'isBooleanValue';

  /**
   * Set in {@link initBajaType} when the bajaType is "baja:String".  Checked when the parsed value
   * is falsy to allow empty Strings for day event values.
   *
   * @type {string}
   */
  constant.IS_STRING_VALUE = 'isStringValue';

  /**
   * Set in {@link initBajaType} when the bajaType is "baja:Double".  Sets the value box to be
   * right-aligned and causes extra parsing in {@link parseValue}.
   *
   * @type {string}
   */
  constant.IS_DOUBLE_VALUE = 'isDoubleValue';

  /**
   * Set in {@link initBajaType} when the bajaType is "baja:DynamicEnum" or {@link IS_BOOLEAN_VALUE}
   * is set (bajaType is "baja:Boolean").  Determines what "value-" class to apply when creating a
   * new event (see {@link createNewEvent}) or changing the event value (see {@link valueChange}).
   * If this flag is set, the {@link RANGE} values are searched for text or an ordinal in
   * {@link getDisplayText} and {@link getValueEncoding}.
   *
   * @type {string}
   */
  constant.IS_ENUM_VALUE = 'isEnumValue';

  constant.TOP_LIMIT = 'topLimit';
  constant.BOTTOM_LIMIT = 'bottomLimit';

  constant.RANGE = 'range';

  constant.READ_ONLY = 'readOnly';
  constant.MOVING = 'moving';   // in the middle of moving

  /**
   * Initially false.  Set to true in {@link ondown} and {@link allDay} when creating new events.
   *
   * If not set when {@link ondayselectedmousedown} is called and the command day is not the same as
   * the previously selected event's day, the previously selected event is deselected.
   *
   * If set when {@link ondayselectedmousemove} is called, {@link onmousemove} is called instead of
   * {@link onselectedmousemouse}.
   *
   * If not set when {@link onmousemove} is called and {@link TOP} is also not set,
   * {@link onselectedmousemove} is called and {@link onmousemove} short-circuits.
   *
   * If set when {@link onselectedmouseup} is called, that method short-circuits to
   * {@link onmouseup} and then this flag is reset.  If set when {@link ondayselectedmouseup} is
   * called, {@link onmouseup} is called instead of {@link onselectedmouseup}.
   *
   * If not set when {@link onmouseup} is called, that method does nothing.  {@link onmouseup} will
   * reset this flag if the the start and finish times are identical and the start time is not 12am.
   * {@link onmouseup} will also reset this flag when the method is finished.
   *
   * If set when {@link timeChange} is called, that method does nothing.
   *
   * If set when {@link setBottom} is called, the offset is set 0.25 instead of 0.0.
   *
   * @type {string}
   */
  constant.NEWDIV = 'newDiv';

  constant.MOVE_TOP = 'moveTop';
  constant.MOVE = 'move';
  constant.TOP = 'top';
  constant.ANCHOR = 'anchor';

  /**
   * Currently selected div.  Initialized to null.
   *
   * Set in {@link onreadonlyselectedmousedown}, to the mousedown event's element, or in
   * {@link ondown}, to the newly created day event.
   *
   * Used in {@link finishleftclick} to set the value input to the selected value and the null
   * status checkbox to the null status of the selected day event.
   *
   * If undefined or null in {@link onselectedmousemove}, the mouse cursor will be changed to
   * indicate whether over the top/bottom edges or in the middle of a day event.  If defined in
   * {@link onselectedmousemove}, it will be passed to {@link moveMiddle}, to move the whole event,
   * or {@link onmousemove}, to change the bottom position, or handled in
   * {@link onselectedmousemove} itself to change the top position.
   *
   * Used in {@link moveMiddle} to set mouse top and bottom limits and have the selected day event
   * position changed.
   *
   * In {@link onselectedmouseup}, if the {@link IN_ACTION} flag is set, the selected div will
   * replace the mouse event's elem.  At the end of {@link onselectedmouseup}, the selected div is
   * cleared.
   *
   * If undefined or null in {@link onmousemove}, that method will short-circuit.  Otherwise, it
   * will be used to change the bottom position.
   *
   * If undefined or null in {@link onmouseup}, that method will short-circuit.  Otherwise, it will
   * be removed in this method if the start and finish times are the same and the start is not 12am.
   * At the end of {@link onmouseup}, the selected div is cleared.
   *
   * If undefined or null in {@link ondayselectedmousemove}, that method will short-circuit.
   * Otherwise, the selected div will be passed to either {@link onmousemove}, if the {@link NEWDIV}
   * flag is set, or {@link onselectedmousemove}.
   *
   * If undefined or null in {@link ondayselectedmouseup}, that method will short-circuit.
   * Otherwise, the selected div will be passed to either {@link onmouseup}, if the {@link NEWDIV}
   * flag is set, or {@link onselectedmouseup}.
   *
   * Cleared also in {@link contextMenu}, {@link daymousedown}, {@link allday}, and
   * {@link ondayselectedmousedown}.
   *
   * @type {string}
   */
  constant.SELECTED_DIV = 'selectedDiv';

  constant.LAST_DIV_VALUE = 'lastDivValue';        // value of lastDiv.  Retains value even if lastDiv is reset
  constant.LAST_DIV = 'lastDiv';                   // last selected div?
  constant.LAST_DAY = 'lastDay';

  /**
   * Set to true if {@link valueChange} is called with finalChecking set to false.  It is initially
   * false and reset to false in {@link valueBlur} if the flag is set.
   *
   * Value is checked in {@link onreadonlyselectedmousedown}, {@link ondown}, and
   * {@link ondayselectedmousedown} and, if set to true, {@link valueBlur} is called with the values
   * stored in {@link LAST_CHANGE}.
   *
   * @type {string}
   */
  constant.VALUE_CHANGE_FLAG = 'valueChangeFlag';

  /**
   * Stores an array of values: event, sourceId, targetId, and scope of the last change.  Set in
   * {@link valueChange} when called with finalChecking set to false.  It is read in
   * {@link onreadonlyselectedmousedown}, {@link ondown}, and {@link ondayselectedmousedown} and the
   * last three values are passed to {@link valueBlur} (event is set to null).
   *
   * @type {string}
   */
  constant.LAST_CHANGE = 'lastChange';

  constant.COPIED_DAY = 'copiedDay';
  constant.COPIED_DAY_FORM_VALUE = 'copiedDayFormValue';

  constant.IN_ACTION = 'inAction';

  constant.COMMAND_DAY = 'commandDay';

  constant.OLD_CLIENTY = 'oldClientY';

  this.init = function (scope) {
    var $ = jQuery;

    // Initialize values
    schedule.data(scope, constant.NULL_STATUS_LEX, 'Null');

    schedule.data(scope, constant.TOP_LIMIT, 0.0);
    schedule.data(scope, constant.BOTTOM_LIMIT, 1.0);

    schedule.data(scope, constant.READ_ONLY, false);
    schedule.data(scope, constant.MOVING, false);
    schedule.data(scope, constant.NEWDIV, false);

    schedule.data(scope, constant.MOVE_TOP, null);
    schedule.data(scope, constant.MOVE, null);

    schedule.data(scope, constant.SELECTED_DIV, null);
    schedule.data(scope, constant.LAST_DIV_VALUE, null);
    schedule.data(scope, constant.LAST_DIV, null);
    schedule.data(scope, constant.LAST_DAY, null);

    schedule.data(scope, constant.VALUE_CHANGE_FLAG, false);
    schedule.data(scope, constant.COPIED_DAY, null);
    schedule.data(scope, constant.COPIED_DAY_FORM_VALUE, null);

    schedule.data(scope, constant.IN_ACTION, false);

    schedule.data(scope, constant.COMMAND_DAY, -1);
    schedule.data(scope, constant.OLD_CLIENTY, 0);

    var $body = $('body');

    if (DEBUG_MOUSE_MOVE) {
      // Debugging
      var $showCoord = $("<div id='showCoord'></div>");
      $showCoord.appendTo($body);
      $showCoord.append('mousePosition: ', "<span id='mouseX'></span>", ', ', "<span id='mouseY'></span>", "<br>");
      var $optional = $("<span id='optional'></span>");
      $optional.append('relativeY: ', "<span id='relativeY'></span>", '; mouse%: ', "<span id='mousePercent'></span>", "<br>");
      $showCoord.append($optional);

      $showCoord.css({
        'display': 'none',
        'position': 'absolute',
        'float': 'left',
        'width': '200px',
        'border': '1px solid red',
        'color': 'red',
        'background-color': 'rgba(0, 0, 0, 0.2)',
        'font-size': '90%'
      });

      // Not sure why this doesn't work.  Perhaps event chaining,
      //  since we don't use jQuery to set the other events.
      // So we'll just call debugMouseMove directly from the other mouseMove handlers.
      //$('document').mousemove(schedule.debugMouseMove);
    }

  };

  this.debugMouseMove = function (scope, event) {
    if (!DEBUG_MOUSE_MOVE) {
      return;
    }

    var $ = jQuery;
    var $elem = $(event.target);
    var dayEventEl;
    if ($elem.hasClass('schedule-HxScheduler-dayEvent')) {
      dayEventEl = event.target.parentNode;
    }

    var mousePosition = hx.getMousePositionViewport(event);

    $('#showCoord').show();

    $('#mouseX').html(mousePosition[0].toFixed(2));
    $('#mouseY').html(mousePosition[1].toFixed(2));

    var $optional = $('#optional');
    if (dayEventEl) {
      $optional.show();
      var elementBounds = hx.getElementBoundsViewport(dayEventEl);
      var relativeY = mousePosition[1] - elementBounds[1];
      var mousePercent = (relativeY) / dayEventEl.offsetHeight;
      $('#relativeY').html(relativeY.toFixed(2));
      $('#mousePercent').html(mousePercent.toFixed(2));
    } else {
      $optional.hide();
    }

    $('#showCoord').css({
      left: event.pageX + 250 > $('body').width() ? event.pageX - 250 : event.pageX + 20,
      top: event.pageY + 100 > $('body').height() ? event.pageY - 75 : event.pageY + 20
    });
  };

  /**
   * Called onload to initialize the type flags.
   *
   * @param bajaType
   * @param scope
   */
  this.initBajaType = function (bajaType, scope) {
    schedule.data(scope, constant.IS_BOOLEAN_VALUE, bajaType === "baja:Boolean");
    schedule.data(scope, constant.IS_STRING_VALUE, bajaType === "baja:String");
    schedule.data(scope, constant.IS_DOUBLE_VALUE, bajaType === "baja:Double");
    schedule.data(scope, constant.IS_ENUM_VALUE, bajaType === "baja:DynamicEnum" || schedule.data(scope, constant.IS_BOOLEAN_VALUE));

    var isDoubleValue = schedule.data(scope, constant.IS_DOUBLE_VALUE);
    if (isDoubleValue) {
      var $ = jQuery;
      $('.schedule-HxScheduler-modBox input[type="text"]').css('text-align', 'right');
    }
  };

  this.leaveTable = function(elem, event, scope) {
    this.ondayselectedmouseup(elem, event, scope);
    document.body.style.cursor = "auto";
  };

  this.setReadOnly = function (scope, bool) {
    schedule.data(scope, constant.READ_ONLY, bool);
  };

  /**
   * Called onload to set the {@link NULL_STATUS_LEX} constant to the 'day.null' in in the schedule
   * lexicon.
   *
   * @param scope
   * @param str
   */
  this.setNullStatus = function (scope, str) {
    schedule.data(scope, constant.NULL_STATUS_LEX, str);
  };

  /**
   * Tied to the onmousedown and ontouchstart events of a day event when the schedule is read-only.
   * Also called from {@link on selectedmousedown}.  Changes the selected day event and then calls
   * {@link finishleftclick}.
   *
   * @param elem div that represents a day event
   * @param event
   * @param scope
   */
  this.onreadonlyselectedmousedown = function (elem, event, scope) {
    var $ = jQuery;

    // See if we have an outstanding change
    if (schedule.data(scope, constant.VALUE_CHANGE_FLAG)) {
      var lastChange = schedule.data(scope, constant.LAST_CHANGE);
      schedule.valueBlur(null, lastChange[1], lastChange[2], lastChange[3]);
    }

    schedule.fixDaySelection(elem, event, scope, 2);

    schedule.data(scope, constant.IN_ACTION, false);

    // Change selected from the previously selected day event to this new one.
    schedule.deselectSelected(scope);
    $(elem).addClass("schedule-HxScheduler-dayEventSelected");

    schedule.data(scope, constant.SELECTED_DIV, elem);
    schedule.data(scope, constant.LAST_DIV, elem);
    schedule.data(scope, constant.LAST_DIV_VALUE, $(elem).attr('value'));
    schedule.data(scope, constant.LAST_DAY, elem.parentNode);
    return schedule.finishleftclick(elem, event, scope);
  };

  /**
   * Calculate the top and bottom limits that an event can be resized to so it doesn't bump into or overlap an adjacent event.
   * The global topLimit and bottomLimit will be set
   * @param elem
   * @param mousePercent
   * @returns mousePercent
   */
  this.setLimits = function (scope, elem, mousePercent) {
    // If elem is a dayEvent, then we use the bounds of the element.  mousePercent is not passed in
    //  If not, then we're creating a new dayEvent, so we use the position of the mouse to set the bounds
    //  In the latter case, we'll also return the revised mousePercent in case it went out of bounds
    var $ = jQuery;
    var $elem = $(elem);

    var $siblings = $elem.siblings('.schedule-HxScheduler-dayEvent');
    var siblingTop;
    var siblingBottom;
    var currentTop;
    var currentBottom;
    if ($elem.hasClass('schedule-HxScheduler-dayEvent')) {
      // Existing event
      currentTop = parseFloat(elem.style.top) / 100.0;
      currentBottom = 1.0 - parseFloat(elem.style.bottom) / 100.0;
    } else {
      // Creating a new event
      currentTop = Math.max(mousePercent, schedule.data(scope, constant.TOP_LIMIT));
      currentBottom = Math.min(mousePercent, schedule.data(scope, constant.BOTTOM_LIMIT));
    }

    // The fudgeFactor accounts for mouse imprecision to prevent 2 adjacent events from overlapping
    var fudgeFactor = 0.01;

    // We can't rely on the siblings being sorted by position so we must iterate
    $siblings.each(function () {
      //console.log('currentTop: ' + currentTop + '; currentBottom: ' + currentBottom);
      siblingBottom = 1.0 - parseFloat(this.style.bottom) / 100.0;
      if ((siblingBottom - fudgeFactor) <= currentTop && siblingBottom > schedule.data(scope, constant.TOP_LIMIT)) {
        schedule.data(scope, constant.TOP_LIMIT, siblingBottom);
      }
      siblingTop = parseFloat(this.style.top) / 100.0;
      if ((siblingTop + fudgeFactor) >= currentBottom && siblingTop < schedule.data(scope, constant.BOTTOM_LIMIT)) {
        schedule.data(scope, constant.BOTTOM_LIMIT, siblingTop);
      }
      //console.log('siblingTop: ' + siblingTop + ' (' + this.style.top + ')' + '; siblingBottom: ' + siblingBottom + ' (' + this.style.bottom + ')');
      //console.log('topLimit: ' + schedule.data(scope, constant.TOP_LIMIT) + '; bottomLimit: ' + schedule.data(scope, constant.BOTTOM_LIMIT));
    });

    //console.log('topLimit: ' + schedule.data(scope, constant.TOP_LIMIT) + '; bottomLimit: ' + schedule.data(scope, constant.BOTTOM_LIMIT));
    if (typeof mousePercent !== 'undefined') {
      mousePercent = Math.max(mousePercent, schedule.data(scope, constant.TOP_LIMIT));
      mousePercent = Math.min(mousePercent, schedule.data(scope, constant.BOTTOM_LIMIT));
      return mousePercent;
    }
  };

  /**
   * Called from {@link onreadonlyselectedmousedown}, which is called from
   * {@link onselectedmousedown}.  Determines whether the day event itself is being moved or just
   * its top and bottom edges.  Then, updates the status.value.value field and status.status null
   * checkbox with the day event values.
   *
   * @param elem div that represents a day event
   * @param event
   * @param scope
   */
  this.finishleftclick = function (elem, event, scope) {
    if (schedule.data(scope, constant.IN_ACTION) && !schedule.data(scope, constant.READ_ONLY)) {
      return smartTable.endEvent(event);
    }

    schedule.data(scope, constant.IN_ACTION, true);

    schedule.setLimits(scope, elem);

    var mousePercent = schedule.getMousePercent(elem, event);
    var topPercent = (elem.offsetTop) / elem.parentNode.offsetHeight;
    var bottomPercent = (elem.offsetTop + elem.offsetHeight) / elem.parentNode.offsetHeight;
    schedule.data(scope, constant.MOVE, null);

    var edge = schedule.detectEdge(topPercent, bottomPercent, mousePercent);
    if (edge === 'top') {
      schedule.data(scope, constant.TOP, false);
      schedule.data(scope, constant.ANCHOR, bottomPercent);
      schedule.data(scope, constant.MOVE, null);
    } else if (edge === 'bottom') {
      schedule.data(scope, constant.TOP, true);
      schedule.data(scope, constant.ANCHOR, topPercent);
      schedule.data(scope, constant.MOVE, null);
    } else {
      schedule.data(scope, constant.TOP, true);
      schedule.data(scope, constant.MOVE, mousePercent);          // starting point for the mouse
      schedule.data(scope, constant.MOVE_TOP, topPercent);        // starting point for top
      schedule.data(scope, constant.ANCHOR, bottomPercent - topPercent);
    }

    // Update the value input (status.value.value) and the null checkbox (status.status) based on
    // the selected day event.
    try {
      var path = scope;
      if (path && path.length > 0) {
        path = path + ".";
      }

      var selectedDiv = schedule.data(scope, constant.SELECTED_DIV);

      var valueElem = schedule.getElem(path + "status.value.value");
      valueElem.value = schedule.getValueEncoding(selectedDiv.getAttribute("value"), scope);

      var nullStatusElem = schedule.getElem(path + "status.status");
      nullStatusElem.checked = selectedDiv.getAttribute("nullstatus") === "true";

      if (!schedule.data(scope, constant.READ_ONLY)) {
        smartTable.disableInput(valueElem, !nullStatusElem.checked);
      }
    } catch (e) {
      console.log(e);
    }

    schedule.setExact(scope);

    return smartTable.endEvent(event);
  };

  /**
   * Tied to the onmousedown and ontouchstart events of a day event when the schedule is NOT
   * read-only.
   *
   * @param elem div that represents a day event
   * @param event
   * @param scope
   */
  this.onselectedmousedown = function (elem, event, scope) {
    schedule.data(scope, constant.TOP_LIMIT, 0.0);
    schedule.data(scope, constant.BOTTOM_LIMIT, 1.0);
    return schedule.onreadonlyselectedmousedown(elem, event, scope);
  };

  /**
   * Tied to the oncontextmenu event of a day event div when the schedule is not read-only.
   *
   * @param elem div that represents a day event
   * @param event
   * @param scope
   */
  this.contextMenu = function (elem, event, scope) {
    if (event.button === 2 || smartTable.forceRightClick) {
      schedule.data(scope, constant.SELECTED_DIV, null);
      var commands = smartTable.fixCommands(scope, 2);
      smartTable.showRightClickMenu(event, commands);
      schedule.data(scope, constant.LAST_DAY, elem.parentNode);
      return smartTable.endEvent(event);
    }
  };

  this.moveMiddle = function (elem, event, scope) {
    if (schedule.data(scope, constant.MOVING)) {
      return;
    } else {
      schedule.data(scope, constant.MOVING, true);
    }

    document.body.style.cursor = "grabbing";
    var height = schedule.data(scope, constant.ANCHOR);
    var oldTop = schedule.data(scope, constant.MOVE_TOP);

    var selectedDiv = schedule.data(scope, constant.SELECTED_DIV);

    var mousePercent = schedule.getMousePercent(elem, event);
    mousePercent = schedule.setLimits(scope, selectedDiv, mousePercent);

    var percentChange = mousePercent - schedule.data(scope, constant.MOVE);

    var top    = oldTop + percentChange;
    var bottom = oldTop + percentChange + height;

    if (top < schedule.data(scope, constant.TOP_LIMIT)) {
      top = schedule.data(scope, constant.TOP_LIMIT);
      bottom = top + height;
    }

    if (bottom > schedule.data(scope, constant.BOTTOM_LIMIT)) {
      bottom = schedule.data(scope, constant.BOTTOM_LIMIT);
      top = bottom - height;
    }

    var path = scope;
    if (path && path.length > 0) {
      path = path + ".";
    }
    schedule.setInfo(path + "start", top);
    schedule.setInfo(path + "finish", bottom);

    if (selectedDiv.style.bottom.length > 0) {
      selectedDiv.style.bottom = "";
      selectedDiv.style.height = height * 100.0 + "%";
    }
    selectedDiv.style.top = top * 100.0 + "%";
    schedule.data(scope, constant.MOVING, false);
  };

  this.detectEdge = function (topPercent, bottomPercent, mousePercent) {
    var margin = 0.04;
    if ((bottomPercent - topPercent) < margin * 2.5) {             // Small blocks need a smaller margin to leave room for the middle
      margin = 0.02;
    }

    // If mouse is within 4% from bottom &&
    //   height of event is at least 8% ||                    As a result of this test, it can be more difficult to grab the bottom of a short event
    //   top of event is less then 50% from top of dayBox     If event is in the top half of the day, more likely to want bottom.
    //
    if ((mousePercent + margin > bottomPercent) &&
      ((bottomPercent - topPercent > margin * 2) || (topPercent < 0.5))) {
      //console.log("bottom");
      return 'bottom';
      // If mouse is within 0.04% from top
    } else if (mousePercent - margin < topPercent) {
      //console.log("top");
      return 'top';
    } else {
      return null;
    }
  };

  /**
   * @returns  the position of the mouse as a percentage from the top of the dayGrid
   */
  this.getMousePercent = function (elem, event) {
    return (hx.getMousePositionViewport(event)[1] - hx.getElementBoundsViewport(elem.parentNode)[1]) / elem.parentNode.offsetHeight;
  };

  /**
   * Tied to the onmousemove and ontouchmove events of a day event when the scheule is not read-
   * only.  Also called from {@link onmousemove} when not creating a new day event ({@link NEWDIV}
   * is set) and moving the top of an event ({@link TOP} is not set).  Also called from
   * {@link ondayselectedmousemove} when not creating a new day event.
   *
   * If a day event is not selected, sets cursor based on the mouses position within a day event.
   * If a day event is selected, calls {@link moveMiddle}, to move the whole day event, or {@link
   * onmousemove}, to change the bottom position, or this method changes the top position of the
   * selected day event.
   *
   * @param elem
   * @param event
   * @param scope
   */
  this.onselectedmousemove = function (elem, event, scope) {
    schedule.debugMouseMove(scope, event);
    var selectedDiv = schedule.data(scope, constant.SELECTED_DIV);
    var mousePercent;
    var topPercent;
    var bottomPercent;
    if (!selectedDiv) {
      // Detect edges
      mousePercent = schedule.getMousePercent(elem, event);
      topPercent = (elem.offsetTop) / elem.parentNode.offsetHeight;
      bottomPercent = (elem.offsetTop + elem.offsetHeight) / elem.parentNode.offsetHeight;

      var edge = schedule.detectEdge(topPercent, bottomPercent, mousePercent);
      if (edge) {
        document.body.style.cursor = "n-resize";
      } else {
        document.body.style.cursor = "pointer";
      }

      return smartTable.endEvent(event);
    }

    // If there is already a selected one, use it
    elem = selectedDiv;

    if (schedule.data(scope, constant.MOVE) !== null) {
      // Moving the whole block
      schedule.moveMiddle(elem, event, scope);
      return smartTable.endEvent(event);
    }

    // Not sure what TOP is.  It's set to true if we're dragging bottom or middle
    if (schedule.data(scope, constant.TOP)) {
      // Dragging the bottom
      schedule.onmousemove(elem, event, scope);
      return smartTable.endEvent(event);
    }

    // Dragging the top - we get here from onMouseMove
    mousePercent = schedule.getMousePercent(elem, event);
    bottomPercent = schedule.data(scope, constant.ANCHOR);

    mousePercent = schedule.setLimits(scope, elem, mousePercent);

    topPercent = mousePercent;

    topPercent = Math.max(topPercent, schedule.data(scope, constant.TOP_LIMIT));
    if (topPercent >= bottomPercent - 0.0208) {
      topPercent = bottomPercent - 0.0208;
    }

    //setInfo
    var path = scope;
    if (path && path.length > 0) {
      path = path + ".";
    }
    schedule.setInfo(path + "start", topPercent);
    schedule.data(scope, constant.LAST_DIV).style.top = (topPercent * 100.0) + "%";
    schedule.setBottom(scope, schedule.data(scope, constant.LAST_DIV), topPercent, schedule.data(scope, constant.ANCHOR));
    return smartTable.endEvent(event);
  };

  this.modified = function (scope) {
    if (save !== null && !schedule.data(scope, constant.READ_ONLY)) {
      save.modified();
    }
  };

  /**
   * Tied to the onmouseup and ontouchend events of a day event when the schedule is NOT
   * read-only.
   *
   * @param elem
   * @param event
   * @param scope
   */
  this.onselectedmouseup = function (elem, event, scope) {
    schedule.data(scope, constant.MOVE, null);

    if (schedule.data(scope, constant.IN_ACTION)) {
      schedule.data(scope, constant.IN_ACTION, false);
      elem = schedule.data(scope, constant.SELECTED_DIV);
    } else {
      return;
    }

    var lastDiv = schedule.data(scope, constant.LAST_DIV);
    if (lastDiv) {
      // Short-circuit to onmouseup if creating a new event
      if (schedule.data(scope, constant.NEWDIV)) {
        schedule.onmouseup(elem, event, scope);
        schedule.data(scope, constant.NEWDIV, false);
        return;
      }

      schedule.modified(scope);
      schedule.fixDaySelection(elem, event, scope, 2);

      var path = scope;
      if (path && path.length > 0) {
        path = path + ".";
      }

      var dayElem = lastDiv.parentNode;
      if (dayElem !== null) {
        var id = schedule.getDayId(scope, dayElem.id);
        var formDay = schedule.getElem(path + "day" + id);
        var index = -1;
        var child = dayElem.firstChild;
        while (child !== lastDiv) {
          child = child.nextSibling;
          if (child.className.indexOf("schedule-HxScheduler-dayEvent") > -1) {
            index++;
          }
        }

        if (formDay === null) {
          return;
        }

        var value = formDay.value;
        if (value === null) {
          return;
        }

        var modify = value.split("|")[index] + "|";
        //modify
        var start = schedule.encode(path + "start");
        var finish = schedule.encode(path + "finish");
        var nullStatus = lastDiv.getAttribute("nullstatus") === "true";
        var indexOf = value.indexOf(modify);
        var endIndex = indexOf + modify.length;
        modify = start + "," + finish + "," + nullStatus + "," + schedule.getValueEncoding(lastDiv.getAttribute("value"), scope) + "|";

        var newValue = value.substring(0, indexOf) + modify + value.substring(endIndex);
        formDay.value = newValue;

        var amount = (0.0 + lastDiv.offsetTop + lastDiv.offsetHeight) / (-2.0 + lastDiv.parentNode.offsetHeight);

        var topPercent = (lastDiv.offsetTop) / lastDiv.parentNode.offsetHeight;
        schedule.setBottom(scope, lastDiv, topPercent, amount);
      }
      schedule.data(scope, constant.SELECTED_DIV, null);
      schedule.data(scope, constant.MOVE, null);
      document.body.style.cursor = "pointer";
    }
  };

  /**
   * Handles the onmousedown, ontouchstart, and oncontextmenu events of the dayGrid background slots
   * that each represent an hour of the day.
   *
   * @param elem background slots on the dayGrid that each represent an hour of the day
   * @param event mouse event
   * @param scope
   */
  this.onmousedown = function (elem, event, scope) {
    schedule.fixDaySelection(elem, event, scope, 1);

    if (event.button === 2 || smartTable.forceRightClick) {
      return;
    }

    schedule.ondown(elem, event, scope);
  };

  /**
   * Initiate creation of a new day event.  Called from onmousedown if not a right-click.
   *
   * @param elem background slots on the dayGrid that each represent an hour of the day
   * @param event mouse event from which the mouse position is read to determine the top position of
   * the new event div
   * @param scope
   */
  this.ondown = function (elem, event, scope) {
    // See if we have an outstanding change
    if (schedule.data(scope, constant.VALUE_CHANGE_FLAG)) {
      var lastChange = schedule.data(scope, constant.LAST_CHANGE);
      schedule.valueBlur(null, lastChange[1], lastChange[2], lastChange[3]);
    }

    schedule.deselectSelected(scope);

    // Determine the top position based on the mouse position relative to the dayGrid
    schedule.data(scope, constant.TOP_LIMIT, 0.0);
    schedule.data(scope, constant.BOTTOM_LIMIT, 1.0);
    var mousePercent = schedule.getMousePercent(elem, event);
    mousePercent = schedule.setLimits(scope, elem, mousePercent);
    if (mousePercent > 0.985) {
      return;
    }

    schedule.data(scope, constant.ANCHOR, mousePercent);
    schedule.data(scope, constant.TOP, true);

    //reinit
    schedule.data(scope, constant.MOVE_TOP, null);
    schedule.data(scope, constant.MOVE, null);
    schedule.data(scope, constant.IN_ACTION, false);

    var $ = jQuery;

    var newEvent = schedule.createEventForValue(scope, mousePercent * 100 + "%", "1px");
    var $newEvent = $(newEvent);

    // Set the new event as selected
    $newEvent.addClass("schedule-HxScheduler-dayEventSelected");

    // Add the new event to the dayGrid
    elem.parentNode.appendChild(newEvent);

    schedule.data(scope, constant.NEWDIV, true);

    var path = scope;
    if (path && path.length > 0) {
      path = path + ".";
    }

    // Set the start and finish times based on the new event
    schedule.setInfo(path + "start", mousePercent);
    schedule.setInfo(path + "finish", mousePercent);

    schedule.data(scope, constant.SELECTED_DIV, newEvent);
    schedule.data(scope, constant.LAST_DIV, newEvent);
    schedule.data(scope, constant.LAST_DIV_VALUE, $newEvent.attr('value'));
  };

  /**
   * Retrieve and check status.value.value.  Retrieve the units, too, if present.
   *
   * Called from {@link ondown} when creating a new event value, from {@link valueChange} when the
   * status.value.value or status.status changes, and from {@link allDay} when the event values for
   * a day are replaced with a single event that fills the entire day.
   *
   * @param scope
   * @param path usually just scope with an extra '.' appended to it
   * @param finalChecking true when called from {@link ondown} and {@link allDay}; true if
   * {@link valueChange} is called from {@link valueBlur}; when true, extra parsing of numeric
   * values is performed and the last value is used if the form value is not valid
   * @returns {[formValue, units, input]}
   */
  this.parseValue = function (scope, path, finalChecking) {
    var $ = jQuery;
    var $form = $(schedule.getElem(path + "status.value.value"));
    var formValue = $form.val();

    // Special handling for a numeric schedule
    var isDoubleValue = schedule.data(scope, constant.IS_DOUBLE_VALUE);
    if (isDoubleValue) {
      if (finalChecking) {
        // If Integer, tack on .0
        if (schedule.isInteger(formValue)) {
          formValue = formValue + '.0';
        }

        // If not a float, we'll set to empty
        if (schedule.isFloat(formValue)) {
          // It's a valid float.  Let's make sure there's at least 1 decimal place specified
          var decimalPlaces = formValue.split(/[\.,]/)[1].length;
          if (decimalPlaces === 0) {
            formValue = parseFloat(formValue).toFixed(1).toString();
          }
        } else {
          // Set to an empty string so the last value is retrieved below and used
          formValue = '';
        }
      }
    }

    if (finalChecking) {
      var isStringValue = schedule.data(scope, constant.IS_STRING_VALUE);
      if (!formValue && !isStringValue) {
        // If we didn't get a good value, use the last selected value
        formValue = schedule.data(scope, constant.LAST_DIV_VALUE);
      }
    }

    // Look for units on the schedule
    var $last = $form.parent().contents().last();
    var units = "";
    if ($last.get(0).nodeType === 3) {
      // If it's a Text node...
      units = $last.text();
    }

    // Update status.value.value with any changes made here
    $form.val(formValue);
    $form.attr('value', formValue);
    return [formValue, units, $form.get(0)];
  };

  /**
   * Called from the fixEvents command of the special events view (see BHxSpecialEventsView) to
   * populate the dayGrid when a special event is selected in the events table.
   *
   * @param elem dayGrid
   * @param scope
   * @param top top position of the event div [%]
   * @param height height of the event div [%]
   * @param isStatusNull whether the event is set to null
   * @param valueNoUnits used to set the div class and value attribute
   * @param valueWithUnits used for the div display text
   */
  this.createDiv = function (elem, scope, top, height, isStatusNull, valueNoUnits, valueWithUnits) {
    elem.appendChild(schedule.createNewEvent(scope, top + "%", height + "%", isStatusNull, valueNoUnits, valueWithUnits));
  };

  /**
   * Create an event based on the current status.value.value and status.status values.  The top
   * position and height are set explicitly and not based on the start and finish time inputs.
   * Called from {@link allDay}, where the top is zero and the height is 100, and {@link ondown},
   * where the top is based on the mouse position and the height is always 1px.
   */
  this.createEventForValue = function (scope, top, height) {
    var valueNoUnits = '';
    var valueWithUnits = '';

    var path = scope;
    if (path && path.length > 0) {
      path = path + ".";
    }

    var value = schedule.parseValue(scope, path, true);
    var isStatusNull = schedule.getElem(path + "status.status").checked;

    // An empty string will be falsy but should still be used if this is a string schedule.
    var isStringValue = schedule.data(scope, constant.IS_STRING_VALUE);
    if (value[0] || isStringValue) {
      valueNoUnits = value[0];
      valueWithUnits = schedule.getDisplayText(value[0], scope) + value[1];
    }

    return schedule.createNewEvent(scope, top, height, isStatusNull, valueNoUnits, valueWithUnits);
  };

  /**
   * Clones the template and updates it based on the current supplied arguments.
   *
   * @param scope
   * @param top top position of the event div
   * @param height height of the event div
   * @param isStatusNull whether the event is set to null
   * @param valueNoUnits used to set the div class and value attribute
   * @param valueWithUnits used for the div display text
   * @return {Node|*}
   */
  this.createNewEvent = function (scope, top, height, isStatusNull, valueNoUnits, valueWithUnits) {
    var path = scope;
    if (path && path.length > 0) {
      path = path + ".";
    }

    var template = hx.$(path + "template");
    var newEvent = template.cloneNode(true);

    newEvent.id = "";
    newEvent.name = "";
    newEvent.style.display = "";
    newEvent.style.top = top;
    // Account for the border-top (1px) and padding-top (2px) values in the height
    newEvent.style.height = 'calc(' + height + ' - 3px)';

    var isEnumValue = schedule.data(scope, constant.IS_ENUM_VALUE);
    if (isStatusNull) {
      newEvent.className = "schedule-HxScheduler-dayEvent value-null";
    } else if (!isEnumValue) {
      newEvent.className = "schedule-HxScheduler-dayEvent value-default";
    } else {
      newEvent.className = "schedule-HxScheduler-dayEvent value-" + valueNoUnits;
    }

    // Set the text displayed on the day event
    if (isStatusNull) {
      newEvent.innerHTML = schedule.safe(schedule.data(scope, constant.NULL_STATUS_LEX));
    } else {
      newEvent.innerHTML = schedule.safe(valueWithUnits);
    }

    newEvent.setAttribute("value", valueNoUnits);
    newEvent.setAttribute("nullstatus", "" + isStatusNull);
    return newEvent;
  };

  this.onmousemove = function (elem, event, scope) {
    schedule.debugMouseMove(scope, event);

    var selectedDiv = schedule.data(scope, constant.SELECTED_DIV);
    if (!selectedDiv) {
      document.body.style.cursor = "auto";
      return smartTable.endEvent(event);
    }

    if (!schedule.data(scope, constant.NEWDIV) && !schedule.data(scope, constant.TOP)) {
      schedule.onselectedmousemove(elem, event, scope);
      return smartTable.endEvent(event);
    }

    if (schedule.data(scope, constant.IN_ACTION) && schedule.data(scope, constant.MOVE) !== null) {
      schedule.moveMiddle(elem, event, scope);
      return smartTable.endEvent(event);
    }

    // Resize from the bottom
    if (schedule.data(scope, constant.OLD_CLIENTY) === event.clientY) {
      return;
    }
    schedule.data(scope, constant.OLD_CLIENTY, event.clientY);

    var mousePercent = schedule.getMousePercent(elem, event);
    mousePercent = schedule.setLimits(scope, selectedDiv, mousePercent);

    var height = mousePercent - schedule.data(scope, constant.ANCHOR);

    // If mouse moved up, then ignore
    if (height <= 0) {
      return;
    }

    var atLimit = mousePercent >= schedule.data(scope, constant.BOTTOM_LIMIT);
    var smallest = height <= 0.0208;

    if (atLimit) {
      schedule.setBottom(scope, selectedDiv, schedule.data(scope, constant.ANCHOR), schedule.data(scope, constant.BOTTOM_LIMIT));
      mousePercent = schedule.data(scope, constant.BOTTOM_LIMIT);
    }

    if (smallest) {
      mousePercent = schedule.data(scope, constant.ANCHOR) + 0.0208;
    }

    var path = scope;
    if (path && path.length > 0) {
      path = path + ".";
    }

    height = mousePercent - schedule.data(scope, constant.ANCHOR);
    schedule.setInfo(path + "finish", mousePercent);
    if (!atLimit) {
      selectedDiv.style.height = height * 100.0 + "%";
    }
    return smartTable.endEvent(event);
  };

  this.onmouseup = function (elem, event, scope) {
    var selectedDiv = schedule.data(scope, constant.SELECTED_DIV);
    if (!selectedDiv) {
      return;
    }

    if (!schedule.data(scope, constant.NEWDIV)) {
      return;
    }

    var lastDiv = schedule.data(scope, constant.LAST_DIV);
    var dayElem = lastDiv.parentNode;
    if (dayElem) {
      var id = schedule.getDayId(scope, dayElem.id);

      var path = scope;
      if (path && path.length > 0) {
        path = path + ".";
      }

      var formDay = schedule.getElem(path + "day" + id);
      var start = schedule.encode(path + "start");
      var finish = schedule.encode(path + "finish");

      if (start === finish && start !== "00:00:00.000") {
        selectedDiv.parentNode.removeChild(selectedDiv);
        schedule.data(scope, constant.SELECTED_DIV, null);
        schedule.data(scope, constant.LAST_DIV, null);
        schedule.data(scope, constant.MOVE, null);
        schedule.data(scope, constant.NEWDIV, false);
        return;
      }

      var nullStatus = lastDiv.getAttribute("nullstatus") === "true";
      formDay.value += start + "," + finish + "," + nullStatus + "," + schedule.getValueEncoding(lastDiv.getAttribute("value"), scope) + "|";
      var amount = (lastDiv.offsetTop + lastDiv.offsetHeight) / lastDiv.parentNode.offsetHeight;
      var topPercent = (lastDiv.offsetTop) / lastDiv.parentNode.offsetHeight;
      schedule.setBottom(scope, lastDiv, topPercent, amount);

      schedule.fixDaySelection(elem, event, scope, 2);
    }

    schedule.modified(scope);
    schedule.data(scope, constant.SELECTED_DIV, null);
    schedule.data(scope, constant.MOVE, null);
    schedule.data(scope, constant.NEWDIV, false);
  };

    /*
     * "hh:mm:ss.mmm".
     */
  this.encode = function (path) {
    var encoded = "";
    var min = parseInt(schedule.getElem(path + ".time.min").value);
    var twelveHourElem = schedule.getElem(path + ".time.twelveHour");
    var hour;
    if (twelveHourElem) {
      hour = parseInt(twelveHourElem.value) + 1;

      var ampm = parseInt(schedule.getElem(path + ".time.ampm").value);


      if (ampm === 1 && hour !== 12) {
        hour += 12;
      } else if (ampm === 0 && hour === 12) {
        hour = 0;
      }
    } else {
      var hour24 = schedule.getElem(path + ".time.hour").value;
      hour = parseInt(hour24);
    }

    if (hour < 10) {
      encoded += '0';
    }
    encoded += hour;
    encoded += ':';
    if (min < 10) {
      encoded += '0';
    }
    encoded += min;
    encoded += ':00.000';

    return encoded;
  };

  /**
   * Decode the time and replace the values in the modbox
   * @param value
   * @param path
   */
  this.decode = function (value, path) {
    var split = value.split(':');
    var hour = schedule.parseIntFix(split[0]);
    var minute = schedule.parseIntFix(split[1]);
    var ampm = 0;


    var minElem = schedule.getElem(path + ".time.min");
    minElem.value = minute;

    var twelveHourElem = schedule.getElem(path + ".time.twelveHour");
    if (twelveHourElem) {
      if (hour > 11) {
        ampm = 1;
        hour = hour - 12;
      }

      if (hour === 0) {
        hour = 12;
      }
      twelveHourElem.value = parseInt(hour) - 1;
      var ampmElem = schedule.getElem(path + ".time.ampm");
      ampmElem.value = ampm;
    } else {
      var hour24Elem = schedule.getElem(path + ".time.hour");
      hour24Elem.value = parseInt(hour);
    }
  };

  this.parseIntFix = function (val) {
    if (val.charAt(0) === '0') {
      val = val.substring(1, val.length);
    }
    return parseInt(val);
  };

    /*
     * return (percent of day)/100
     */
  this.getTime = function (path) {
    var min = parseInt(schedule.getElem(path + ".time.min").value);

    var twelveHourElem = schedule.getElem(path + ".time.twelveHour");
    var hour;
    if (twelveHourElem) {
      hour = parseInt(twelveHourElem.value) + 1;
      var ampm = parseInt(schedule.getElem(path + ".time.ampm").value);

      if (ampm === 1 && hour !== 12) {
        hour += 12;
      } else if (ampm === 0 && hour === 12) {
        hour = 0;
      }
    } else {
      hour = parseInt(schedule.getElem(path + ".time.hour").value);
    }

    var percentOfDay = (0.0 + hour + (min / 60.0)) / 24.0;
    return percentOfDay;
  };

  /**
   * Uses the percent (in terms of a day) to find the hour, min, and ampm values with which to call
   * {@link setTimeInfo}.
   *
   * @param path the time value to modify once the hour, min, and ampm values are determined
   * @param percent percent of a day used to determine the hour, min, and ampm values
   */
  this.setInfo = function (path, percent) {
    if (percent > 0.99) {
      percent = 1.0;
    }

    var time = percent * 24.0;
    var hour = Math.floor(time);
    var min = Math.floor((time - hour) * 60.0);
    var ampm = 0;

    // Round up to the next hour if minutes are 45 or more
    if (min >= 45.0) {
      hour += 1;
      min = 0;
    }

    // Round to 30 if minutes are between 15 and 45 (non-inclusive); otherwise, round down to 0
    var top = 15.0 < min && min < 45.0;
    if (top) {
      min = 30;
    } else {
      min = 0;
    }

    // Set pm if hour is 12 or more
    if (hour > 11) {
      ampm = 1;
    }

    if (hour === 24 && ampm === 1 && min === 0) {
      hour = 12;
      ampm = 0;
    }

    hour = hour % 12;
    if (hour === 0) {
      hour = 12;
    }

    schedule.setTimeInfo(path, hour, min, ampm);
  };

  this.setTimeInfo = function (path, hour, min, ampm) {
    var twelveHourElem = schedule.getElem(path + ".time.twelveHour");
    if (twelveHourElem) {
      twelveHourElem.value = hour - 1;

      var ampmElem = schedule.getElem(path + ".time.ampm");
      ampmElem.value = ampm;
    } else {

      if (!ampm && hour === 12) {
        hour = 0;
      } else if (ampm && hour !== 12) {
        hour = hour + 12;
      }

      var hourElem = schedule.getElem(path + ".time.hour");
      hourElem.value = hour;
    }

    var minElem = schedule.getElem(path + ".time.min");
    minElem.value = min;
  };

  this.getElem = function (listName) {
    var colList = document.getElementById(listName);
    if (colList === null) {
      colList = document.getElementsByName(listName)[0];
    }
    return colList;
  };

  this.setExact = function (scope) {
    var lastDiv = schedule.data(scope, constant.LAST_DIV);
    if (!lastDiv) {
      return;
    }

    var path = scope;
    if (path && path.length > 0) {
      path = path + ".";
    }

    var dayElem = lastDiv.parentNode;
    if (dayElem !== null) {
      var id = schedule.getDayId(scope, dayElem.id);
      var formDay = schedule.getElem(path + "day" + id);
      var index = -1;
      var child = dayElem.firstChild;
      while (child !== lastDiv) {
        child = child.nextSibling;
        if (child.className.indexOf("schedule-HxScheduler-dayEvent") > -1) {
          index++;
        }
      }

      var value = formDay.value;
      var modify = value.split("|")[index] + "|";

      var start = modify.split(",")[0];
      var finish = modify.split(",")[1];

      try {
        schedule.decode(start, path + "start");
        schedule.decode(finish, path + "finish");
      }
      catch (err) {
        //ignore IE error
        //alert(index);
        //alert(value);
      }
    }

  };

  this.valueFocus = function (e, sourceId, targetId, scope) {
  };

  /**
   * Tied to the blur event of status.value.value.  Also called from
   * {@link onreadonlyselectedmousedown}, {@link ondown}, and {@link ondayselectedmousedown} when
   * the {@link VALUE_CHANGE_FLAG} is set with the values stored in {@link LAST_CHANGE}.  Calls
   * {@link valueChange} if the {@link VALUE_CHANGE_FLAG} is set and resets
   * {@link VALUE_CHANGE_FLAG}.
   *
   * @param e event
   * @param sourceId
   * @param targetId always empty
   * @param scope
   */
  this.valueBlur = function (e, sourceId, targetId, scope) {
    if (schedule.data(scope, constant.VALUE_CHANGE_FLAG)) {
      schedule.data(scope, constant.VALUE_CHANGE_FLAG, false);
      schedule.valueChange(e, sourceId, targetId, scope, true);
    }
  };

  /**
   * Called twice when the field loses focus: once for the onchange event and once for onblur.  The
   * onblur event makes sure that the final checking of the input is done.
   *
   * Tied to the change event of status.value.value and status.status and keyup on
   * status.value.value.
   *
   * @param e event
   * @param sourceId
   * @param targetId always empty
   * @param scope
   * @param finalChecking only set when called from {@link valueBlur}
   */
  this.valueChange = function (e, sourceId, targetId, scope, finalChecking) {
    // Chrome triggers onchange with arrow keys.  Ignore them
    if (e && (e.key === 'ArrowRight' || e.key === 'ArrowLeft' || e.key === 'ArrowUp' || e.key === 'ArrowDown')) {
      return;
    }

    var input = this;
    var selectionStart;
    var selectionEnd;
    if (input.type === 'text') {
      // Chrome doesn't remember cursor position, so save it
      selectionStart = input.selectionStart;
      selectionEnd = input.selectionEnd;
    }

    // finalChecking is only set when called from valueBlur
    finalChecking = (finalChecking === true);
    if (!finalChecking) {
      schedule.data(scope, constant.VALUE_CHANGE_FLAG, true);
      // Remember arguments
      schedule.data(scope, constant.LAST_CHANGE, [e, sourceId, targetId, scope]);
    }

    var $ = jQuery;

    var lastDiv = schedule.data(scope, constant.LAST_DIV);
    var $lastDiv;
    if (lastDiv) {
      $lastDiv = $(lastDiv);

      // Remove any existing 'value-' class; replaced below with a value class for the new value
      $lastDiv.removeClass(function (index, css) {
        return (css.match(/(^|\s)value-\S+/g) || []).join(' ');
      });

      var path = scope;
      if (path && path.length > 0) {
        path = path + ".";
      }

      var value = schedule.parseValue(scope, path, finalChecking);
      var isEnumValue = schedule.data(scope, constant.IS_ENUM_VALUE);
      if (isEnumValue && value[0] === null) {
        // Might be ad-hoc enum value
        value[0] = lastDiv.getAttribute('value');
      }

      var isStatusNull = schedule.getElem(path + "status.status").checked;
      var isStringValue = schedule.data(scope, constant.IS_STRING_VALUE);
      try {
        if (isStatusNull) {
          lastDiv.innerHTML = schedule.data(scope, constant.NULL_STATUS_LEX);
        } else if (!finalChecking || (value[0] || isStringValue)) {
          lastDiv.innerHTML = schedule.safe(schedule.getDisplayText(value[0], scope) + value[1]);
        }

        lastDiv.setAttribute("nullstatus", isStatusNull);
        lastDiv.setAttribute("value", schedule.getDisplayText(value[0], scope));

        if (isStatusNull) {
          $lastDiv.addClass("value-null");
        } else if (!isEnumValue) {
          $lastDiv.addClass("value-default");
        } else if (value[0] || isStringValue) {
          $lastDiv.addClass("value-" + value[0]);
        }

        var dayElem = lastDiv.parentNode;
        if (dayElem !== null) {
          var id = schedule.getDayId(scope, dayElem.id);
          var formDay = schedule.getElem(path + "day" + id);
          var index = -1;
          var child = dayElem.firstChild;
          while (child !== lastDiv) {
            child = child.nextSibling;
            if (child.className.indexOf("schedule-HxScheduler-dayEvent") > -1) {
              index++;
            }
          }

          var dayValue = formDay.value;
          var modify = dayValue.split("|")[index] + "|";
          //modify
          var start = schedule.encode(path + "start");
          var finish = schedule.encode(path + "finish");
          var indexOf = dayValue.indexOf(modify);
          var newModify = start + "," + finish + "," + isStatusNull + "," + schedule.getValueEncoding(lastDiv.getAttribute("value"), scope) + "|";
          var newValue = dayValue.substring(0, indexOf) + newModify + dayValue.substring(indexOf + modify.length, dayValue.length);
          formDay.value = newValue;
          schedule.modified(scope);
        }
      } catch (ev) {
        console.log(ev);
      }
    }

    if (input.type === 'text') {
      // For Chrome
      input.selectionStart = selectionStart;
      input.selectionEnd = selectionEnd;
    }
  };

  this.timeChange = function (e, sourceId, targetId, scope) {
    var $ = jQuery;
    if (schedule.data(scope, constant.NEWDIV)) {
      return;
    }
    var lastDiv = schedule.data(scope, constant.LAST_DIV);
    if (lastDiv) {
      var path = scope;
      if (path && path.length > 0) {
        path = path + ".";
      }
      var dayElem = lastDiv.parentNode;
      if (dayElem !== null) {
        // Verify that start is before end and adjust the height of the block
        var time = schedule.getTime(path + targetId);
        var topPercent;
        var bottomPercent;
        if (targetId === "start") {
          topPercent = Math.max(time, schedule.data(scope, constant.TOP_LIMIT));

          // getTime() returns 0.0 for 12:00am.  But for bottom, we'll change to 1.0
          bottomPercent = schedule.getTime(path + "finish");
          if (bottomPercent === 0) {
            bottomPercent = 1.0;
          }
        } else {
          // getTime() returns 0.0 for 12:00am.  But for bottom, we'll change to 1.0
          if (time === 0.0) {
            time = 1.0;
          }
          bottomPercent = Math.min(time, schedule.data(scope, constant.BOTTOM_LIMIT));
          topPercent = (lastDiv.offsetTop) / lastDiv.parentNode.offsetHeight;
        }

        if (topPercent >= bottomPercent) {
          alert("The start time must be before the finish time.");
          // set back to original value
          schedule.setExact(scope);
          return;
        }

        var id = schedule.getDayId(scope, dayElem.id);
        var formDay = schedule.getElem(path + "day" + id);

        // Find the index of the current event (lastDiv)
        var $children = $(dayElem).children(".schedule-HxScheduler-dayEvent");
        var index = $children.index($(lastDiv));

        // We cannot assume that the children are in order, so we'll have to search one by one
        var value = formDay.value;

        var splitValues = value.split('|');
        splitValues.length = splitValues.length - 1; // Get rid of the last empty element

        var modify = splitValues[index] + "|";
        var currentTimes = modify.split(',');
        var currentStart = currentTimes[0];
        var currentFinish = currentTimes[1];

        var minimumStart = '00:00:00.000';
        var maximumFinish = '24:00:00.000';
        // Loop through all the times and find the one that is < start time and > end time
        $.each(splitValues, function (i, val) {
          if (i === index) {
            // continue
            return true;
          }
          var split = val.split(',');
          var s = split[0];
          var f = split[1];
          minimumStart = f > minimumStart && f <= currentStart ? f : minimumStart;
          maximumFinish = s < maximumFinish && s >= currentFinish ? s : maximumFinish;
        });

        var newStart = schedule.encode(path + "start");
        var newFinish = schedule.encode(path + "finish");

        if (newStart < minimumStart) {
          alert('Start time cannot be before finish time of previous event');
          // set back to original value
          schedule.setExact(scope);
          return;
        }
        if (newFinish > maximumFinish) {
          alert('Finish time cannot be after start time of next event');
          // set back to original value
          schedule.setExact(scope);
          return;
        }

        // Update the event in the grid
        schedule.setBottom(scope, lastDiv, topPercent, bottomPercent);
        if (targetId === "start") {
          lastDiv.style.top = topPercent * 100.0 + "%";
        }


        var nullStatus = lastDiv.getAttribute("nullstatus") === "true";
        var indexOf = value.indexOf(modify);
        modify = newStart + "," + newFinish + "," + nullStatus + "," + schedule.getValueEncoding(lastDiv.getAttribute("value"), scope) + "|";

        var newValue = value.substring(0, indexOf) + modify + value.substring(indexOf + modify.length, value.length);
        formDay.value = newValue;
        schedule.modified(scope);
      }
    }
  };

  /**
   * Remove the dayEventSelected class from the previously selected event.
   *
   * @param scope
   */
  this.deselectSelected = function (scope) {
    var lastDiv = schedule.data(scope, constant.LAST_DIV);
    if (lastDiv) {
      jQuery(lastDiv).removeClass("schedule-HxScheduler-dayEventSelected");
    }
  };

  this.deleteSelected = function (scope) {
    var lastDiv = schedule.data(scope, constant.LAST_DIV);
    if (lastDiv) {
      schedule.modified(scope);
      var path = scope;
      if (path && path.length > 0) {
        path = path + ".";
      }
      var dayElem = lastDiv.parentNode;
      if (dayElem !== null) {
        var id = schedule.getDayId(scope, dayElem.id);
        var formDay = schedule.getElem(path + "day" + id);
        //alert(formDay.value);
        var index = -1;
        var child = dayElem.firstChild;
        while (child !== lastDiv) {
          child = child.nextSibling;
          if (child.className.indexOf("schedule-HxScheduler-dayEvent") > -1) {
            index++;
          }
        }

        var value = formDay.value;
        var remove = value.split("|")[index] + "|";
        var indexOf = value.indexOf(remove);
        var newValue = value.substring(0, indexOf) + value.substring(indexOf + remove.length, value.length);
        formDay.value = newValue;

        //remove
        lastDiv.parentNode.removeChild(lastDiv);
      }

      schedule.data(scope, constant.LAST_DIV, null);
    }
  };

  /**
   * Tied to oncontextmenu event of the dayGrid.
   *
   * @param elem
   * @param event
   * @param scope
   */
  this.daymousedown = function (elem, event, scope) {
    if (event.button === 2 || smartTable.forceRightClick) {
      schedule.data(scope, constant.SELECTED_DIV, null);

      // TODO pass in zero if nothing is selected so the delete command is hidden (maybe other commands as well)
      var commands = smartTable.fixCommands(scope, 1);
      smartTable.showRightClickMenu(event, commands);

      schedule.data(scope, constant.LAST_DAY, elem);

      return smartTable.endEvent(event);
    }
  };

  /**
   * Tied to the clearDay button and context menu command.  Also called from {@link clearWeek},
   * {@link applyMF}, {@link pasteDay}, and {@link allDay}.
   *
   * @param elem
   * @param event
   * @param scope
   */
  this.clearDay = function (elem, event, scope) {
    schedule.modified(scope);
    schedule.clearDayNoModify(elem, event, scope);
  };

  /**
   * Called from {@link clearDay} and from the fixEvents command of the special events view (see
   * BHxSpecialEventsView).
   *
   * @param elem
   * @param event
   * @param scope
   */
  this.clearDayNoModify = function (elem, event, scope) {
    //if (!elem) throw 'null element';
    var path = scope;
    if (path && path.length > 0) {
      path = path + ".";
    }

    var dayElem = elem;
    if (dayElem === null) {
      dayElem = schedule.data(scope, constant.LAST_DAY);
    }

    if (dayElem === null) {
      dayElem = schedule.getElem(schedule.data(scope, constant.COMMAND_DAY));
    }

    // Just use the numeric part
    var id = schedule.getDayId(scope, dayElem.id);
    var formDay = schedule.getElem(path + "day" + id);
    if (formDay !== null) {
      var childNodes = dayElem.childNodes;
      for (var i = childNodes.length - 1; i >= 0; i--) {

        if (childNodes[i].className.indexOf("schedule-HxScheduler-dayEvent") > -1) {
          dayElem.removeChild(childNodes[i]);
        }
      }
      formDay.value = "";
    }
  };

  /**
   * Tied to the clearWeek button and context menu command.
   *
   * @param elem
   * @param event
   * @param scope
   */
  this.clearWeek = function (elem, event, scope) {
    var path = scope;
    if (path && path.length > 0) {
      path = path + ".";
    }
    for (var i = 0; i < 7; i++) {
      var currentDay = schedule.getElem(path + i);
      schedule.clearDay(currentDay, event, scope);
    }
  };

  /**
   * Tied to the applyMF button and context menu command.
   *
   * @param elem
   * @param event
   * @param scope
   */
  this.applyMF = function (elem, event, scope) {
    schedule.modified(scope);
    var path = scope;
    if (path && path.length > 0) {
      path = path + ".";
    }
    var dayElem = schedule.data(scope, constant.LAST_DAY);

    if (dayElem === null) {
      dayElem = schedule.getElem(schedule.data(scope, constant.COMMAND_DAY));
    }

    if (dayElem !== null) {
      var id = schedule.getDayId(scope, dayElem.id);
      var intId = parseInt(id);
      var formDay = schedule.getElem(path + "day" + id);
      var events = schedule.copy(dayElem, event, scope);
      schedule.data(scope, constant.LAST_DAY, schedule.getElem(intId));

      for (var i = 1; i < 6; i++) {
        if (intId === i) {
          continue;
        }
        var currentDay = schedule.getElem(path + i);
        var currentFormDay = schedule.getElem(path + "day" + i);
        schedule.clearDay(currentDay, event, scope);

        for (var j = 0; events !== null && j < events.length; j++) {
          currentDay.appendChild(events[j].cloneNode(true));
        }

        currentFormDay.value = formDay.value;
      }
    }
  };

  this.copyDay = function (elem, event, scope) {
    var path = scope;
    if (path && path.length > 0) {
      path = path + ".";
    }
    var dayElem = schedule.data(scope, constant.LAST_DAY);

    if (dayElem === null) {
      dayElem = schedule.getElem(schedule.data(scope, constant.COMMAND_DAY));
    }

    if (dayElem !== null) {
      var id = schedule.getDayId(scope, dayElem.id);
      schedule.data(scope, constant.COPIED_DAY, schedule.copy(dayElem, event, scope));
      schedule.data(scope, constant.COPIED_DAY_FORM_VALUE, schedule.getElem(path + "day" + id).value);
    }
  };

  this.pasteDay = function (elem, event, scope) {
    schedule.modified(scope);
    var path = scope;
    if (path && path.length > 0) {
      path = path + ".";
    }

    if (schedule.hasPaste(scope)) {
      var id = schedule.data(scope, constant.COMMAND_DAY);
      var dayElem = schedule.getElem(path + id);
      var formDay = schedule.getElem(path + "day" + id);
      var events = schedule.data(scope, constant.COPIED_DAY);
      schedule.clearDay(dayElem, event, scope);

      for (var i = 0; i < events.length; i++) {
        dayElem.appendChild(events[i].cloneNode(true));
      }

      formDay.value = schedule.data(scope, constant.COPIED_DAY_FORM_VALUE);
    }
  };

  this.hasPaste = function (scope) {
    return (schedule.data(scope, constant.COPIED_DAY) !== null);
  };

  this.copy = function (elem, event, scope) {
    var $ = jQuery;

    var dayElem = elem;

    var childNodes = dayElem.childNodes;

    var events = [0];
    var eventsAmount = 0;
    for (var i = 0; i < childNodes.length; i++) {
      if (childNodes[i].className.indexOf("schedule-HxScheduler-dayEvent") > -1) {
        var node = childNodes[i].cloneNode(true);
        $(node).removeClass("schedule-HxScheduler-dayEventSelected");
        events[eventsAmount] = node;
        eventsAmount++;
      }
    }
    if (eventsAmount === 0) {
      return null;
    }
    return events;
  };

  this.allDay = function (elem, event, scope) {
    schedule.deselectSelected(scope);

    schedule.clearDay(schedule.data(scope, constant.LAST_DAY), event, scope);

    var dayElem = schedule.data(scope, constant.LAST_DAY);
    if (dayElem === null) {
      dayElem = schedule.getElem(schedule.data(scope, constant.COMMAND_DAY));
    }

    if (dayElem === null) {
      return null;
    }

    var $ = jQuery;

    var newEvent = schedule.createEventForValue(scope, "0%", "100%");
    var $newEvent = $(newEvent);

    // Set the new event as selected
    $newEvent.addClass("schedule-HxScheduler-dayEventSelected");

    // Add the new event to the dayGrid
    dayElem.appendChild(newEvent);

    schedule.data(scope, constant.NEWDIV, true);

    var path = scope;
    if (path && path.length > 0) {
      path = path + ".";
    }

    // Set the start and finish times based on the new all-day event
    schedule.setInfo(path + "start", 0.0);
    schedule.setInfo(path + "finish", 1.0);

    schedule.data(scope, constant.SELECTED_DIV, null);
    schedule.data(scope, constant.LAST_DIV, newEvent);

    // modify
    var id = schedule.getDayId(scope, dayElem.id);
    var formDay = schedule.getElem(path + "day" + id);
    var start = schedule.encode(path + "start");
    var finish = schedule.encode(path + "finish");
    var isStatusNull = newEvent.getAttribute("nullstatus") === "true";
    var modify = start + "," + finish + "," + isStatusNull + "," + schedule.getValueEncoding(newEvent.getAttribute("value"), scope) + "|";
    formDay.value = modify;
  };

  this.setBottom = function (scope, elem, topPercent, bottomPercent) {
    if (topPercent >= bottomPercent) {
      return;
    }
    var offset = 0.0;
    if (schedule.data(scope, constant.NEWDIV)) {
      offset = 0.25;
    }

    schedule.data(scope, constant.LAST_DIV).style.height = "";
    elem.style.bottom = ((1.000 - bottomPercent) * 100.0 - offset) + "%";

    return;
  };

  /**
   * Tied to dayGrid onmousemove and ontouchmove events.  If a day event is selected, call
   * {@link onmousemove} if it is a new event or {@link onselectedmousemove} otherwise.
   *
   * @param elem
   * @param event
   * @param scope
   * @returns {boolean}
   */
  this.ondayselectedmousemove = function (elem, event, scope) {
    schedule.debugMouseMove(scope, event);

    var selectedDiv = schedule.data(scope, constant.SELECTED_DIV);
    if (!selectedDiv) {
      return false;
    }

    if (schedule.data(scope, constant.NEWDIV)) {
      schedule.onmousemove(selectedDiv, event, scope);
    } else {
      schedule.onselectedmousemove(selectedDiv, event, scope);
    }
  };

  /**
   * Tied to dayGrid onmouseup and ontouchend events.  If a day event is selected, call
   * {@link onmouseup} if it is a new event or {@link onselectedmouseup} otherwise.
   *
   * @param elem
   * @param event
   * @param scope
   * @returns {boolean}
   */
  this.ondayselectedmouseup = function (elem, event, scope) {
    var selectedDiv = schedule.data(scope, constant.SELECTED_DIV);
    if (!selectedDiv) {
      return false;
    }

    if (schedule.data(scope, constant.NEWDIV)) {
      schedule.onmouseup(selectedDiv, event, scope);
    } else {
      schedule.onselectedmouseup(selectedDiv, event, scope);
    }
  };

  /**
   * Tied to dayGrid onmousedown and ontouchstart events.  If a day event is selected but in a
   * different day and not creating a new event, de-select the previously selected day event.
   *
   * @param elem
   * @param event
   * @param scope
   */
  this.ondayselectedmousedown = function (elem, event, scope) {
    var $ = jQuery;

    // See if we have an outstanding change
    if (schedule.data(scope, constant.VALUE_CHANGE_FLAG)) {
      var lastChange = schedule.data(scope, constant.LAST_CHANGE);
      schedule.valueBlur(null, lastChange[1], lastChange[2], lastChange[3]);
    }

    schedule.fixDaySelection(elem, event, scope, 1);

    var lastDiv = schedule.data(scope, constant.LAST_DIV);
    if (lastDiv) {
      var parentNode = lastDiv.parentNode;
      if (parentNode) {
        var commandDay = schedule.data(scope, constant.COMMAND_DAY);
        var isNewDiv = schedule.data(scope, constant.NEWDIV);
        if (parseInt(schedule.getDayId(scope, parentNode.id)) !== commandDay && !isNewDiv) {
          schedule.data(scope, constant.SELECTED_DIV, null);
          //schedule.data(scope, constant.NEWDIV, false);

          $(lastDiv).removeClass("schedule-HxScheduler-dayEventSelected");
          schedule.data(scope, constant.LAST_DIV, null);
        }
      }
    }

    return smartTable.endEvent(event);
  };

  /**
   * Called onload to set the range of values allowed for boolean and enum schedules.
   *
   * @param rangeEncoding comma delimited list of property=value pairs
   * @param scope
   */
  this.setValueRange = function (rangeEncoding, scope) {
    // Convert to valid Json by replacing = with : and quoting the property names
    var removeBracesRegEx = /{(.*)}/;
    var toJSONRegEx = /['"]?(.+?)['"]?\s*[:=]\s*(.+?)(?:(, ?) *|$)/g;
    var toJSONRegExReplace = '"$1": $2$3';

    var noBraces = removeBracesRegEx.exec(rangeEncoding)[1];
    var json = '{' + noBraces.replace(toJSONRegEx, toJSONRegExReplace) + '}';
    var range = JSON.parse(json);
    schedule.data(scope, constant.RANGE, range);
  };

  this.getValueEncoding = function (displayText, scope) {
    var $ = jQuery;

    var isEnumValue = schedule.data(scope, constant.IS_ENUM_VALUE);
    if (!isEnumValue) {
      // Not enum, so just return
      return displayText;
    }

    var myRange = schedule.data(scope, constant.RANGE);
    if (myRange.length === 0 && !schedule.isInteger(displayText)) {
      // For enum without range, must be integer
      throw "getValueEncoding(): Only integer allowed for Enum - " + displayText;
    }

    var result;
    $.each(myRange, function (text, encoded) {
      if (schedule.slotPathUnescape(text) === displayText) {
        result = encoded;
      }
    });

    if (result === undefined) {
      //this could be a dynamic enum for a integer value not in the range
      return displayText;
    }
    return result;
  };

  this.getDisplayText = function (encoding, scope) {
    var $ = jQuery;

    var isEnumValue = schedule.data(scope, constant.IS_ENUM_VALUE);
    if (!isEnumValue) {
      // Not enum, just return
      return encoding;
    }

    // Either an enum or boolean schedule
    var isBooleanValue = schedule.data(scope, constant.IS_BOOLEAN_VALUE);
    if (isBooleanValue) {
      if (encoding.toString() !== 'true' && encoding.toString() !== 'false') {
        // Must be boolean
        throw "getDisplayText(): Only boolean allowed for Boolean - " + encoding;
      }
    } else if (!schedule.isInteger(encoding)) {
      // For enum, must be integer
      throw "getDisplayText(): Only integer allowed for Enum - " + encoding;
    }

    var result;
    $.each(schedule.data(scope, constant.RANGE), function (text, encoded) {
      if ((isBooleanValue && encoded.toString() === encoding) || encoded === parseInt(encoding)) {
        result = schedule.slotPathUnescape(text);
      }
    });

    if (result === undefined) {
      //this could be a dynamic enum for a integer value not in the range
      return encoding;
    }
    return result;
  };

  this.safe = function (value) {
    var $ = jQuery;

    return $('<span>').text(value).html();
  };

  /* Needs to match safeEncoding in BHxWeeklyScheduler.java */
  this.safeEncode = function (object) {

    var text = String(object);
    while (text.indexOf("|") > -1) {
      text = text.replace("|", String.fromCharCode(65532));
    }

    while (text.indexOf(",") > -1) {
      text = text.replace(",", String.fromCharCode(65533));
    }

    return text;
  };

  this.slotPathUnescape = function (str) {
    if (str.length === 0) {
      return str;
    }

    // Convert from $xx
    str = str.replace(/\$[0-9a-fA-F]{2}/g, function (s) {
      return String.fromCharCode(parseInt(s.substring(1, s.length), 16));
    });

    // Convert from $uxxxx
    str = str.replace(/\$u[0-9a-fA-F]{4}/g, function (s) {
      return String.fromCharCode(parseInt(s.substring(2, s.length), 16));
    });

    return str;
  };

  this.multiSelectChanged = function (elem) {
    var formValue = "";
    for (var i = 0; i < elem.options.length; i++) {
      if (elem.options[i].selected) {

        if (formValue.length > 0) {
          formValue += ";";
        }

        formValue += elem.options[i].value;
      }
    }
    hx.setFormValue(elem.name + "Multiple", formValue);
  };

  this.fixDaySelection = function (elem, event, scope, selectionType) {
    smartTable.fixCommands(scope, selectionType);

    var dayBoxNode = elem;
    while (dayBoxNode !== null && dayBoxNode.className !== null && dayBoxNode.className.indexOf('Box') === -1) {
      dayBoxNode = dayBoxNode.parentNode;
    }

    if (dayBoxNode === null || dayBoxNode.className === null) {
      return;
    }

    // lastDay expects to be a dayGrid, so get the dayGrid from this dayBox
    var $ = jQuery;
    schedule.data(scope, constant.LAST_DAY, $('.schedule-HxScheduler-dayGrid', $(dayBoxNode)).first().get(0));


    if (dayBoxNode.getAttribute('day') !== null) {
      schedule.data(scope, constant.COMMAND_DAY, parseInt(dayBoxNode.getAttribute('day')));
    }

    var dayBoxNodeTd = dayBoxNode.parentNode;
    dayBoxNode.className = "schedule-HxScheduler-dayBox schedule-HxScheduler-dayBoxSelected";
    elem = dayBoxNodeTd.previousSibling;
    while (elem !== null) {
      if (elem.firstChild !== null) {
        elem.firstChild.className = "schedule-HxScheduler-dayBox";
      }

      elem = elem.previousSibling;
    }

    elem = dayBoxNodeTd.nextSibling;
    while (elem !== null) {
      if (elem.firstChild !== null) {
        elem.firstChild.className = "schedule-HxScheduler-dayBox";
      }

      elem = elem.nextSibling;
    }
  };

  this.checkDisableUp = function (scope) {
    return schedule.shouldEnableMove(scope, 'up');
  };

  this.checkDisableDown = function (scope) {
    return schedule.shouldEnableMove(scope, 'down');
  };

  /**
   * Determines whether the move up or move down button should be enabled.  If there is only one
   * event, both buttons are disabled.  If there are no selected events or more than one event is
   * selected, both button are disabled.  Otherwise, if there is more than one event and only one is
   * selected, disable the up button if the selected row is at the top and disable the down button
   * if the selected row is at the bottom.
   *
   * @param scope
   * @param context either 'up' for the move up button or 'down' for the move down button
   * @returns {boolean} true to enable the button; false to disable
   */
  this.shouldEnableMove = function (scope, context) {
    var $ = jQuery;
    scope = scope ? hx.escapeSelector(scope + '.') : '';

    var $events = $('#' + scope + 'records .ux-table-row');
    var count = $events.length;

    // Can't go up or down if there's only one event in the table
    if (count === 1) {
      return false;
    }

    var $selected = $('#' + scope + 'records .ux-table-row.selected');
    // Disable if multi-select or if nothing is selected
    if ($selected.length !== 1) {
      return false;
    }

    // Get the index of the single selected row
    var index = $events.index($selected);
    if (context === 'up') {
      // Return true if the selected row is not the top row
      return index > 0;
    } else {
      // Return true if the selected row is not the bottom row
      return index < count - 1;
    }
  };

  this.integerRegex = /^[-+]?\d+$/;
  this.floatRegex = /^[-+]?\d*?(\.|,)\d+|\d+(\.|,)$/;

  this.testNum = function (numStr) {
    console.log(numStr + ' ==> integer: ' + schedule.isInteger(numStr) + '; float: ' + schedule.isFloat(numStr));
  };

  this.isInteger = function (numStr) {
    return schedule.integerRegex.test(numStr);
  };

  this.isFloat = function (numStr) {
    return schedule.floatRegex.test(numStr);
  };

  /**
   * Get the last segment, which should be an int
   * @param id
   * @returns {*}
   */
  this.getDayId = function (scope, id) {
    var intId = id.substring(scope.length + 1);
    if (!schedule.isInteger((intId))) {
      throw "Must be integer: " + intId;
    }
    return intId;
  };

  this.data = function (scope) {
    var $ = jQuery;
    var args = Array.prototype.slice.call(arguments, this.data.length);
    scope = scope ? hx.escapeSelector(scope + '.') : '';
    var anchor = $('#' + scope + 'records');
    return anchor.data.apply(anchor, args);
  };
}
