/**
 * @copyright 2020 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/* eslint-env browser */

/**
 * API Status: **Private**
 * @module baja/env/mux/BoxEnvelopeMux
 */
define(['bajaScript/sys', 'bajaScript/env/mux/BoxEnvelope'], function (baja, BoxEnvelope) {
  'use strict';

  /**
   * This class does the work of bundling up BOX messages into envelopes and sending the individual
   * fragments across the wire.
   *
   * The BOX messages will be packaged up into a BOX frame to be processed as normal on the server.
   * This BOX frame is muxing-specific - individual BOX frames as generated by baja.comm calls will
   * _not_ go out over the wire with muxing enabled. Each BOX frame as received by the server
   * represents one BOX envelope.
   *
   * @class
   * @alias module:baja/comm/BoxEnvelopeMux
   * @param {baja/env/mux/muxUtils~MuxSettings} params
   */
  var BoxEnvelopeMux = function BoxEnvelopeMux(params) {
    this.$sendToServer = params.sendToServer;
    this.$maxEnvelopeSize = params.maxEnvelopeSize;
    this.$maxMessageSize = params.maxMessageSize;

    /**
     * Envelope containing box messages that are sitting, waiting for their timeout to elapse,
     * so they can be sent up to the server.
     * @type {module:baja/env/mux/BoxEnvelope}
     */
    this.$bufferEnvelope = null;
  };

  /**
   * @param {object} msg - BOX message to be JSON-stringified and sent
   * @returns {number} the id of the BoxEnvelope in which the data will be sent
   */
  BoxEnvelopeMux.prototype.sendMessage = function (msg) {
    var envData = BoxEnvelope.toBytes(JSON.stringify(msg));
    if (envData.length > this.$maxEnvelopeSize) {
      throw new Error('single message overflowed envelope size (see box.mux.maxEnvelopeSize system property)');
    }
    if (!this.$canBuffer(envData)) {
      this.$flush();
    }
    this.$buffer(envData);
    return this.$getBufferEnvelope().getEnvelopeId();
  };

  /**
   * @private
   * @returns {module:baja/env/mux/BoxEnvelope} the envelope currently used to
   * buffer up outgoing data
   */
  BoxEnvelopeMux.prototype.$getBufferEnvelope = function () {
    return this.$bufferEnvelope || (this.$bufferEnvelope = this.$startFreshEnvelope());
  };

  /**
   * @private
   * @returns {module:baja/env/mux/BoxEnvelope}
   */
  BoxEnvelopeMux.prototype.$startFreshEnvelope = function () {
    var sessionId = baja.comm.getServerSessionId();
    var envelopeId = baja.$mux.nextEnvelopeId();
    var env = BoxEnvelope.makeOutgoing({
      sessionId: sessionId,
      envelopeId: envelopeId,
      maxMessageSize: this.$maxMessageSize,
      maxEnvelopeSize: this.$maxEnvelopeSize
    });
    env.append("[{\"p\":\"box\",\"n\":".concat(envelopeId, ",\"m\":["));
    this.$hasBufferedData = false;
    return env;
  };

  /**
   * @private
   * @param {Uint8Array|string} payload
   * @returns {boolean} if the payload can be buffered without overflowing the
   * buffer envelope
   */
  BoxEnvelopeMux.prototype.$canBuffer = function (payload) {
    // three bytes for closing ]}], and comma if there are existing messages in the array
    return this.$getBufferEnvelope().willFit(payload.length + (this.$bufferEmpty() ? 3 : 4));
  };

  /**
   * @private
   * @param {Uint8Array|string} data a string of JSON
   */
  BoxEnvelopeMux.prototype.$buffer = function (data) {
    var env = this.$getBufferEnvelope();
    if (!this.$bufferEmpty()) {
      env.append(',');
    }
    env.append(data);
    this.$hasBufferedData = true;
  };

  /**
   * @private
   * @returns {boolean}
   */
  BoxEnvelopeMux.prototype.$bufferEmpty = function () {
    return !this.$hasBufferedData;
  };

  /**
   * @private
   */
  BoxEnvelopeMux.prototype.$flush = function () {
    var env = this.$getBufferEnvelope();
    var sendToServer = this.$sendToServer;
    if (!this.$bufferEmpty()) {
      env.append(']}]');
      env.toBoxFragments().forEach(sendToServer);
      this.$bufferEnvelope = this.$startFreshEnvelope();
    }
  };
  return BoxEnvelopeMux;
});
