/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson
 */

/* eslint-env browser */
/*global WebKitMutationObserver */

/**
 * Hooks for running BajaScript inside of a browser.
 *
 * @module baja/env/browser
 * @private
 */
define(["bajaBrowserEnvUtil", "bajaScript/baja/comm/Callback", "bajaScript/alarm", "bajaScript/env/BrowserCommsManager", "bajaScript/boxcs", "bajaScript/bson", "bajaScript/coll", "bajaScript/comm", "bajaScript/comp", "bajaScript/ctypes", "bajaScript/file", "bajaScript/hist", "bajaScript/nav", "bajaScript/obj", "bajaScript/ord", "bajaScript/sys", "bajaScript/tag", "bajaScript/transfer", "bajaScript/virt"], function defineBrowser(envUtil, Callback, baja, BrowserCommsManager) {
  "use strict";

  if (!baja.comm) {
    throw new Error("comm.js required");
  }
  var skipNiagaraOriginUriSuffix = "view:?skipNiagaraOriginUri=true";

  ////////////////////////////////////////////////////////////////
  // Callback Promise Implementation
  //////////////////////////////////////////////////////////////// 

  envUtil.mixInPromises(Callback);

  /**
   * This namespace is for documentation purposes only and will not actually
   * be available to Bajascript apps. It details enhancements/decorations
   * applied to functions in <code>baja.comm</code> when Bajascript is deployed
   * to a web browser environment.
   * 
   * @namespace
   * @name baja.browser.comm
   */
  (function comm() {
    // Ensure the started callbacks only happen once the DOM is fully loaded.
    var doStart = baja.comm.start,
      doStop = baja.comm.stop,
      commsManager = new BrowserCommsManager(),
      connectionManager = commsManager.getConnectionManager();

    /**
     * Get the internal connection being used or null if one is being used at all.
     * In the case of a browser, this will most likely be a WebSocket.
     *
     * @name  baja.browser.comm.getConnection
     * @function
     * @private
     * 
     * @returns {module:baja/env/Connection~InternalConnection|null} The internal connection being used or null.
     */
    baja.comm.getConnection = function getConnection() {
      return connectionManager.getInternalConnection();
    };

    /**
     * @private
     * @returns {boolean} true if BajaScript is operating over a secure channel
     * @since Niagara 4.14
     */
    baja.comm.$isSecure = function () {
      return connectionManager.isSecure();
    };

    /**
     * Reuse the connection. This enables BajaScript to share a WebSocket 
     * connection with another instanceof BajaScript.
     *
     * @name  baja.browser.comm.reuseConnection
     * @function
     * @private
     * 
     * @param {module:baja/env/Connection~InternalConnection} connection The connection to reuse.
     * @return {Promise} promise to be resolved when the reused connection has
     * been installed and BajaScript is ready to be started.
     */
    baja.comm.reuseConnection = function reuseConnection(connection) {
      return connectionManager.start(connection);
    };

    /**
     * In a browser, <code>baja.comm.start</code> will wait for the DOM to be
     * ready (as well as for communications to the station be established)
     * before executing its callback.
     * 
     * @name baja.browser.comm.start
     * @function
     * @private
     * @inner
     */
    baja.comm.start = function start(obj) {
      var started = obj.started || baja.ok;
      obj.started = function () {
        envUtil.domReady(started);
      };
      connectionManager.start().then(function () {
        doStart(obj);
      })["catch"](obj.commFail || baja.comm.serverCommFail);
    };

    /**
     * In a browser, <code>baja.comm.stop</code> will shut down any open
     * web sockets.
     * 
     * @name baja.browser.comm.stop
     * @function
     * @private
     * @inner
     */
    baja.comm.stop = function stop(obj) {
      try {
        doStop(obj);
      } finally {
        // Shutdown any connections that might be open.
        connectionManager.close();
      }
    };

    /**
     * @private
     * @param {URL} url
     */
    baja.comm.$assignLocation = function (url) {
      baja.comm.$getLocation().assign(url);
    };

    /**
     * @private
     * @returns {Location}
     */
    baja.comm.$getLocation = function () {
      return location;
    };

    /**
     * In a browser, <code>baja.comm.reconnect</code> will simply reload the
     * browser page.
     *
     * @name baja.browser.comm.reconnect
     * @function
     * @private
     * @inner
     */
    baja.comm.reconnect = function reconnect() {
      // Delay the reconnection slightly - just incase there's a problem and this triggers another refresh!
      var location = baja.comm.$getLocation();
      var locationString = location.toString();
      if (!locationString.endsWith(skipNiagaraOriginUriSuffix)) {
        locationString += "|" + skipNiagaraOriginUriSuffix;
      }
      var url = new URL(locationString);
      baja.clock.schedule(function () {
        // If we can get to the login page then attempt a reconnection...
        // Reconnect by refreshing the page...
        if (location && location.assign) {
          baja.comm.$assignLocation(url);
        }
      }, 2500);
    };

    /**
     * In a browser, <code>baja.comm.BoxFrame#send</code> will delegate this method.
     *
     * @private
     * 
     * @param callback The asynchronous callback.
     */
    baja.comm.BoxFrame.prototype.send = function (callback) {
      var messages = this.$body.m;
      if (!(messages && messages.length)) {
        return callback.ok(this.$body);
      }
      return commsManager.sendBoxFrame(this, callback);
    };

    ////////////////////////////////////////////////////////////////
    // HTTP Comms
    //////////////////////////////////////////////////////////////// 

    /**
     * An Http Error.
     *
     * A HTTP Error happens as a result of problem with HTTP communication.
     * 
     * @class
     * @private
     * @param {XmlHttpRequest} x the object used for comms.
     */
    baja.comm.HttpError = function (x) {
      var t = x.responseText || "Session disconnected",
        status = x.status;
      this.name = "HttpError";
      this.message = t + " err: " + x.status;
      this.status = status;

      // Indicate to delay any reconnection if a 404 or no status is returned.
      this.delayReconnect = !status || status === 404;
    };
    baja.subclass(baja.comm.HttpError, baja.comm.ServerError);
  })();

  /**
   * This namespace is for documentation purposes only and will not actually
   * be available to Bajascript apps. It details enhancements/decorations
   * applied to functions in <code>baja</code> when Bajascript is deployed
   * to a web browser environment.
   * 
   * @namespace
   * @name baja.browser
   */
  (function sysUtil() {
    var _outln = baja.outln,
      _clearOut = baja.clearOut,
      _error = baja.error;

    /**
     * In a browser, <code>baja.outln</code> will (in addition to calling
     * <code>bajaJsPrint()</code>) look for a <code>&lt;pre&gt;</code> element 
     * with ID <code>bajaScriptOut</code> and append the message to that 
     * element as well.
     * 
     * @name baja.browser.outln
     * @function
     * @private
     * @inner
     */
    baja.outln = function (msg) {
      // If BajaScript has stopped then don't output anything else...
      if (baja.isStopping()) {
        return this;
      }
      var bajaScriptOut, escapedMsg;
      if (typeof document !== 'undefined' && document.getElementById) {
        bajaScriptOut = document.getElementById("bajaScriptOut");
        if (bajaScriptOut) {
          escapedMsg = String(msg).replace(/</g, "(").replace(/>/g, ")")
          // Should work around IE for ignoring newlines in 'pre' and 'textarea' elements
          .replace(/\n/g, "\r\n");
          bajaScriptOut.appendChild(document.createTextNode(escapedMsg));
        }
      }

      // If available, attempt to write out to Chrome's JavaScript window
      if (typeof console !== 'undefined' && console.log) {
        console.log(msg);
      }
      return _outln.call(this, msg);
    };

    /**
     * In a browser, <code>baja.clearOut</code> will look for a
     * <code>&lt;pre&gt;</code> element with ID <code>bajaScriptOut</code> and
     * wipe all text from it.
     * 
     * @name baja.browser.clearOut
     * @function
     * @private
     * @inner
     */
    baja.clearOut = function () {
      if (typeof document !== 'undefined' && document.getElementById) {
        var elem = document.getElementById("bajaScriptOut");
        if (elem) {
          elem.innerHTML = "";
        }
      }
      return _clearOut.apply(this, arguments);
    };

    /**
     * In a browser, <code>baja.error</code> will (in addition to calling
     * <code>baja.outln</code>) look for <code>window.console.error</code> and,
     * if it exists, pass the error message to it.
     * 
     * @name baja.browser.error
     * @function
     * @private
     * @inner
     */
    baja.error = function (msg) {
      var _arguments = arguments,
        _this = this;
      if (baja.isStopping()) {
        return this;
      }

      //if it is a LocalizableError and has the toStack function the stack will already be
      //correctly formatted
      if (msg instanceof Error && msg.name === 'LocalizableError' && typeof msg.toStack === 'function') {
        return msg.toStack().then(function (stack) {
          //To get the LocalizableError to show correctly in Firefox I had to create a new error
          //and send that error to the console.error as a fake LocalizableError
          var msg = stack.split('\n')[0];
          msg = msg.replace("LocalizableError: ", "");
          var newError = new Error(msg);
          newError.name = "LocalizableError";
          newError.stack = stack;
          console.error(newError);
          return _error.apply(_this, _arguments);
        });
      }

      // If available, attempt to write out to Chrome's JavaScript Window
      if (typeof console !== 'undefined' && console.error) {
        console.error(msg);
      }
      return _error.apply(this, arguments);
    };

    /**
     * In a browser, get the top window that doesn't have cross-origin frame problems.
     *
     * @name baja.browser.topWindow
     * @function
     * @private
     *
     * @param {object} [wnd=window] Html Window
     * @returns {object} the top available window
     */
    baja.topWindow = function (wnd) {
      if (!wnd) {
        wnd = window;
      }
      var bestWindow = wnd;
      try {
        while (wnd !== wnd.parent && (wnd = wnd.parent)) {
          if (wnd.location.href) {
            bestWindow = wnd;
          }
        }
      } catch (ignore) {
        //don't use the window when there are cross-origin frame problems
      }
      return bestWindow;
    };

    /**
     * Get the parent window that doesn't have cross-origin frame problems.
     *
     * @name baja.browser.parentWindow
     * @function
     * @private
     *
     * @param {object} [wnd=window] Html Window
     * @returns {*} the parent window or the window that was passed in
     */
    baja.parentWindow = function (wnd) {
      if (!wnd) {
        wnd = window;
      }
      var bestWindow = wnd;
      try {
        if (wnd !== wnd.parent && (wnd = wnd.parent)) {
          if (wnd.location.href) {
            bestWindow = wnd;
          }
        }
      } catch (ignore) {
        //don't use the window when there are cross-origin frame problems
      }
      return bestWindow;
    };

    /**
     * In a browser environment that supports it, swap out the default 
     * setTimeout(0) implementation of baja.runAsync with a 
     * much faster implementation.
     */
    (function setupAsync() {
      if (window.niagaraAsync) {
        baja.runAsync = window.niagaraAsync;
      } else {
        var Obs = typeof MutationObserver !== 'undefined' && MutationObserver || typeof WebKitMutationObserver !== 'undefined' && WebKitMutationObserver;
        if (Obs) {
          var runAsyncViaObserver = function runAsyncViaObserver(fn) {
            var node = document.createTextNode('');
            new Obs(fn).observe(node, {
              characterData: true
            });
            node.data = '1';
          };
          runAsyncViaObserver(function () {
            baja.runAsync = runAsyncViaObserver;
          });
        }
      }
    })();
  })();

  /**
   * This namespace is for documentation purposes only and will not actually
   * be available to Bajascript Apps. It details enhancements/decorations
   * applied to functions in <code>baja.storage</code> when Bajascript is 
   * deployed to a web browser environment.
   * 
   * @namespace
   * @name baja.browser.storage
   */
  (function sysRegistry() {
    var storage = baja.storage,
      store;
    try {
      store = window.localStorage;
    } catch (ignore) {}

    /**
     * In a browser, this overrides <code>baja.storage.removeItem</code>
     * and will clear the web storage content (if the browser supports web 
     * storage).
     * 
     * @name baja.browser.storage.removeItem
     * @function
     * @private
     * @throws {Error} if the item could not be deleted
     */
    storage.removeItem = function (key) {
      if (store) {
        return store.removeItem(key);
      }
    };

    /**
     * In a browser, this overrides <code>baja.storage.setItem</code>
     * and will write a key/value pair to web storage (if the browser supports 
     * web storage).
     * 
     * @name baja.browser.storage.setItem
     * @function
     * @private
     * 
     * @param {String} key
     * @param {String} data
     * @throws {Error} if the item could not be saved
     */
    storage.setItem = function (key, data) {
      // If available, save any cached registry information to web storage   
      if (store) {
        return store.setItem(key, data);
      }
    };

    /**
     * In a browser, this overrides <code>baja.storage.getItem</code> and will
     * retrieve a value from web storage (if the browser supports web storage).
     * 
     * @name baja.browser.storage.getItem
     * @function
     * @private
     * 
     * @param {String} key
     * @throws {Error} if the item could not be read
     */
    storage.getItem = function (key) {
      if (store) {
        return store.getItem(key);
      }
    };
  })();
  return baja;
});
