/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson
 */

/**
 * Defines {@link baja.StationScheme}.
 * @module baja/ord/StationScheme
 */
define(["bajaScript/sys", "bajaScript/baja/ord/OrdQuery", "bajaScript/baja/ord/OrdScheme", "bajaScript/baja/ord/OrdTarget"], function (baja, OrdQuery, OrdScheme, OrdTarget) {
  "use strict";

  var subclass = baja.subclass,
    callSuper = baja.callSuper;

  /**
   * Station ORD Scheme.
   * 
   * This scheme resolves to a Station.
   *
   * @class
   * @alias baja.StationScheme
   * @extends baja.OrdScheme
   * @private
   */
  var StationScheme = function StationScheme() {
    callSuper(StationScheme, this, arguments);
  };
  subclass(StationScheme, OrdScheme); // TODO: Need to extend from SpaceScheme eventually

  /**
   * Default Station ORD Scheme instance.
   * @private
   * @type {baja.StationScheme}
   */
  StationScheme.DEFAULT = new StationScheme();

  /**
   * Called when an ORD is resolved.
   *
   * @private
   *
   * @see baja.OrdScheme#resolve
   *
   * @param {module:baja/ord/OrdTarget} target  the current ORD Target.
   * @param {baja.OrdQuery} query  the ORD Query used in resolving the ORD.
   * @param {module:baja/ord/OrdQueryListCursor} cursor  the ORD Query List 
   * cursor used for helping to asynchronously resolve the ORD.
   * @param {Object} options  options used for resolving an ORD.
   */
  StationScheme.prototype.resolve = function (target, query, cursor, options) {
    var newTarget = new OrdTarget(target);
    newTarget.object = target.object;
    if (!newTarget.object.getType().is("baja:ComponentSpace")) {
      newTarget.object = baja.nav.localhost.station;
    }
    cursor.resolveNext(newTarget, options);
  };

  /**
   * Return an ORD Query for the scheme.
   *
   * @returns {baja.OrdQuery}
   */
  StationScheme.prototype.parse = function (schemeName, body) {
    return new OrdQuery({
      scheme: this,
      schemeName: schemeName,
      body: body,
      normalize: function normalize(list, index) {
        // Shift to session
        var i,
          q,
          modified = false;
        for (i = index - 1; i >= 0; --i) {
          q = list.get(i);
          if (!q.isHost() && !q.isSession()) {
            list.remove(i);
            modified = true;
          }
        }
        return modified;
      }
    });
  };
  return StationScheme;
});
