/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson
 */

/**
 * Defines {@link baja.ServiceScheme}.
 * @module baja/ord/ServiceScheme
 */
define(["bajaScript/comm", "bajaScript/baja/ord/OrdScheme", "bajaScript/baja/ord/OrdTarget", "bajaScript/baja/comm/Callback"], function (baja, OrdScheme, OrdTarget, Callback) {
  "use strict";

  var subclass = baja.subclass,
    callSuper = baja.callSuper;

  /**
   * Service ORD Scheme.
   * 
   * This scheme is used to resolve a Service running in a Station.
   * 
   * @class
   * @alias baja.ServiceScheme
   * @extends baja.OrdScheme
   * @private
   */
  var ServiceScheme = function ServiceScheme() {
    callSuper(ServiceScheme, this, arguments);
  };
  subclass(ServiceScheme, OrdScheme);

  /**
   * Default Service Scheme instance.
   * @private
   * @type {baja.ServiceScheme}
   */
  ServiceScheme.DEFAULT = new ServiceScheme();

  /**
   * Called when an ORD is resolved.
   *
   * @private
   *
   * @see baja.OrdScheme#resolve
   *
   * @param {module:baja/ord/OrdTarget} target  the current ORD Target.
   * @param {baja.OrdQuery} query  the ORD Query used in resolving the ORD.
   * @param {module:baja/ord/OrdQueryListCursor} cursor  the ORD Query List 
   * cursor used for helping to asynchronously resolve the ORD.
   * @param {Object} options  options used for resolving an ORD.
   */
  ServiceScheme.prototype.resolve = function (target, query, cursor, options) {
    var object = target.object,
      typeSpec = query.getBody(),
      space,
      ok,
      fail;
    if (!baja.hasType(object)) {
      throw new Error("Not based via ComponentSpace. Invalid Object");
    }
    if (object.getType().isComponent()) {
      space = object.getComponentSpace();
    } else if (object.getType().is("baja:ComponentSpace")) {
      space = object;
    } else if (object.getType().is("baja:ISession") && object.station) {
      space = object.station;
    }

    // Pick up whether the Space is null
    if (!baja.hasType(space)) {
      throw new Error("Not based via ComponentSpace");
    }
    if (!space.hasCallbacks()) {
      throw new Error("Unable to resolve Service: " + typeSpec);
    }
    ok = function ok(comp) {
      var newTarget = new OrdTarget(target);
      newTarget.object = comp;
      cursor.resolveNext(newTarget, options);
    };
    fail = function fail(err) {
      options.callback.fail(err);
    };
    space.getCallbacks().getService(typeSpec, new Callback(ok, fail));
  };
  return ServiceScheme;
});
